# -*- coding: utf-8 -*-
from copy import deepcopy
import os

import mpfs.engine.process
import mpfs.common.errors.share as errors

from mpfs.core.filesystem.photoslice_filter import is_photoslice_file
from mpfs.core.social.share import Group
from mpfs.core.social.share.notifier import IndexerNotifier
from mpfs.core.factory import get_resource_by_file_id
from mpfs.core.filesystem.indexer import DiskDataIndexer
from mpfs.metastorage.mongo.util import generate_version_number
from mpfs.core.services.index_service import SearchIndexer
from mpfs.core.services.search_service import SearchDB
from mpfs.core.services.smartcache_service import SmartcacheService
from mpfs.core.social.share import LinkToGroup
from mpfs.core.queue import mpfs_queue
from mpfs.config import settings
from mpfs.engine.queue2.celery import BaseTask, app


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

INDEXER_SEARCH_INDEX_BODY = settings.indexer['search_index_body']
INDEXER_PHOTOSLICE_NOTIFICATION_ON_INDEXER_SIDE = settings.indexer['photoslice_notification_on_indexer_side']
SMARTCACHE_WORKER_NOTIFICATION_ENABLED = settings.services['smartcache']['worker_notification_enabled']
MISSING_FILES_SEARCH_TIMEOUT = settings.indexer['missing_files_search_timeout']


@app.task(base=BaseTask)
def handle_reindex(uid, context=None, **kwargs):
    # DEPRECATED.
    # У :class:`DiskDataIndexer` нет атрибута `reindex`, значит не используется.
    DiskDataIndexer().reindex(uid)


@app.task(base=BaseTask)
def handle_reindex_search(uid, index_body, index_type, mediatype, force, context=None, **kwargs):
    DiskDataIndexer().search_reindex(uid, index_body, index_type, mediatype, force)


@app.task(base=BaseTask)
def handle_push_invite_activated(uid, gid, context=None, **kwargs):
    indexer = DiskDataIndexer()
    try:
        link = LinkToGroup.load(uid=uid, gid=gid)
    except errors.ShareNotFound:
        return

    root_folder = link.get_folder()
    full_index = root_folder.get_full_index(safe=False)

    indexer.push_tree(
        link.uid, full_index.values(),
        'modify', version=root_folder.version,
        group=link.group, uids=[link.uid],
        operation='invite_activated'
    )

    for gid, _indexer_data in indexer.group_data.iteritems():
        for _indexer, _data in _indexer_data.iteritems():
            if _data:
                data = {'data': _data, 'gid': gid}
                mpfs_queue.put(data, _indexer + '_group')


def _get_push_change_kwargs(is_photoslice):
    if not is_photoslice:
        return {
            'wait_search_response': False,
            'service': 'disk_queue',
            'append_smartcache_callback': False
        }
    elif INDEXER_PHOTOSLICE_NOTIFICATION_ON_INDEXER_SIDE:
        # фотосрез по новой схеме
        return {
            'wait_search_response': False,
            'service': 'photoslice',
            'append_smartcache_callback': True
        }
    else:
        # фотосрез по старой схеме
        return {
            'wait_search_response': True,
            'service': 'disk_queue',
            'append_smartcache_callback': False
        }


@app.task(base=BaseTask)
def handle_notify_search_indexer(data, is_photoslice=False, task_data=None, context=None, **kwargs):
    if task_data:  # В task_data данные передаются через монгу, если в data данных слишком много, а само оно сбрасывается в None
        data = task_data.data

    if not data:
        return

    _notify_search_indexer(data, is_photoslice)


@app.task(base=BaseTask)
def handle_notify_search_photoslice_indexer(data, task_data=None, context=None, **kwargs):
    if task_data:  # В task_data данные передаются через монгу, если в data данных слишком много, а само оно сбрасывается в None
        data = task_data.data

    if not data:
        return

    # Уведомляем SearchIndexer об изменениях в фотосрезе
    _notify_search_indexer(data, True)

    # Уведомляем SmartCache об изменениях в фотосрезе
    _notify_smartcache_worker({d['uid'] for d in data})


@app.task(base=BaseTask)
def handle_search_missing_photoslice(data, task_data=None, context=None, **kwargs):
    """
    Пытаемся починить индекс для фотосреза

    В data должен быть список [{'uid': UID, 'path': '/disk/path/to_file.jpg'}]
    path приходят из /info и /bulk_info, если не удалось их найти в базе
    для каждого path из data:
       находим в поиске записи с точным совпадением имени и с type=file
       отбираем из них фотосрезные
       для каждого file_id из найденных в поиске ищем в базе файл
       если находим, оповещаем поиск, что переименовали
       ели не нашли, то оповещаем поиск, что файл удалён

       Если в проверке окажется файл из публичной папки и он был переименован, то этот алгоритм не найдёт новый
       ресурс и в поиск мы отправим удаление. Это плохо, но не хуже, чем было, когда поиск выдавал несуществующий файл.
    """

    if task_data:  # В task_data данные передаются через монгу, если в data данных слишком много, а само оно сбрасывается в None
        data = task_data.data
    deleted = []
    renamed = []
    search = SearchDB()
    uid = data['uid']
    for path in data['paths']:
        search_results = search.find_indexed_files(uid, path,
                                                   fields=('file_id', 'key', 'etime', 'ctime', 'mimetype'),
                                                   # поиск работает медленно, если ходить в него со стандартным
                                                   # таймаутом 20 секунд, то у нас забиваются воркеры
                                                   alternate_timeout=MISSING_FILES_SEARCH_TIMEOUT)
        for search_result in search_results:
            # если на месте ненайденного файла был нефотосрезный файл, ничего не делаем
            if not is_photoslice_file(search_result['path'], 'file', search_result['mimetype'], None, search_result['etime']):
                continue
            try:
                resource = get_resource_by_file_id(uid, search_result['file_id'])
            except errors.ResourceNotFound:
                deleted.append({
                    # словарь, похожий на resource.dict(), который обычно передаётся в push_tree
                    # добавляем photoslice_time, чтобы push_tree подумал, что это фотосрезный файл и добавил колбек
                    # для обновления smartcache
                    'meta': {'file_id': search_result['file_id'], 'photoslice_time': 1},
                    'id': search_result['path'],
                    'uid': uid,
                    'version': generate_version_number()})
            else:
                if resource.path != path:
                    renamed.append(resource.dict())
    indexer = DiskDataIndexer()
    indexer.push_tree(uid, deleted, 'delete', operation='rm', metric='404_info_repair')
    indexer.push_tree(uid, renamed, 'modify', operation='move_resource', metric='404_info_repair')
    indexer.flush_index_data()


@app.task(base=BaseTask)
def handle_notify_smartcache_photoslice_changed(uid, context=None, **kwargs):
    SmartcacheService().notify_worker(uid)


@app.task(base=BaseTask)
def handle_notify_search_indexer_group(data, gid, is_photoslice=False, task_data=None, context=None, **kwargs):
    if task_data:  # В task_data данные передаются через монгу, если в data данных слишком много, а само оно сбрасывается в None
        data = task_data.data

    if not data:
        return

    _notify_search_indexer_group(data, gid, is_photoslice)


@app.task(base=BaseTask)
def handle_search_photoslice_indexer_group(data, gid, task_data=None, context=None, **kwargs):
    if task_data:  # В task_data данные передаются через монгу, если в data данных слишком много, а само оно сбрасывается в None
        data = task_data.data

    if not data:
        return

    _notify_search_indexer_group(data, gid, True)
    _notify_smartcache_worker({d['uid'] for d in data})


def _notify_search_indexer_group(data, gid, is_photoslice):
    push_kwargs = _get_push_change_kwargs(is_photoslice)
    indexer = SearchIndexer()

    try:
        group = Group.load(gid)
    except errors.GroupNotFound:
        return

    for link in group.iterlinks():
        indexer.push_change(_get_user_data(link, data), **push_kwargs)

    indexer.push_change(_get_owner_data(group.owner, data), **push_kwargs)


def _notify_search_indexer(data, is_photoslice):
    data = deepcopy(data)
    indexer = SearchIndexer()

    _update_file_bodies(indexer, data)
    push_kwargs = _get_push_change_kwargs(is_photoslice)
    indexer.push_change(data, **push_kwargs)


def _notify_smartcache_worker(uids):
    if INDEXER_PHOTOSLICE_NOTIFICATION_ON_INDEXER_SIDE or not SMARTCACHE_WORKER_NOTIFICATION_ENABLED:
        return

    for uid in uids:
        mpfs_queue.put(
            {'uid': uid},
            'notify_smart_cache_photo_slice_changed'
        )


def _update_file_bodies(indexer, data):
    if not INDEXER_SEARCH_INDEX_BODY:
        return

    items = (
        d for d in data if
        d['action'] == 'modify' and d['type'] == 'file' and 'body_text' not in d
    )
    for item in items:
        body = indexer.get_file_body(item)
        item.update(body)


def _get_owner_data(owner_uid, data):

    items = (d for d in deepcopy(data) if d.pop('actor', None) != owner_uid)

    data = []
    for item in items:
        uids = item.pop('uids', [])
        if uids and owner_uid not in uids:
            continue
        item['uid'] = int(owner_uid)
        data.append(item)

    return data


def _get_user_data(link, data):

    items = (d for d in deepcopy(data) if d.pop('actor', None) != link.uid)

    data = []
    for item in items:
        uids = item.pop('uids', [])
        if uids and link.uid not in uids:
            continue
        try:
            if 'id' in item:
                item['id'] = link.get_link_path(item['id'])
                indexer_base_version = link.get_search_indexer_base_version()
                if indexer_base_version and item['id'] != link.path and item['id'].startswith(link.path):
                    if 'real_resource_version' not in item:
                        # это условие нужно только на момент выкатки, если ты видишь этот код, смело удаляй условие
                        pass
                    else:
                        real_resource_version = item.pop('real_resource_version')
                        item['shared_folder_version'] = int(real_resource_version) + indexer_base_version
        finally:
            item['uid'] = int(link.uid)
            data.append(item)

    return data


@app.task(base=BaseTask)
def handle_group_user_kicked_notify_search_index(gid, path, file_id, uid, version, action, context=None, **kwargs):
    """Уведомить индексатор поиска о том, что пользователь был удалён из группы.

    К моменту когда будет вызвана это асинхронная задача уже может не быть линка на группу у пользователя,
    а соответственно и папки.
    Поэтому для того чтобы получить весь список ресурсов мы получаем его из папки владельца и подменяем
    название рутовой папки владельца на название рутовой папки у юзера, которое у него было, когда он
    состоял в группе.

    :param gid: Идентификатор группы (общей папки)
    :param path: Путь до содержимого папки *у приглашенного пользователя* на момент когда его удаляли.
    :param file_id: Файловый идентификатор папки *у приглашенного пользователя* на момент когда его удаляли.
    :param uid: UID *приглашенного пользователя*
    :param version: Версия папки на момент удаления пользователя
    :param action: Каким действием уведомить notifier
    """
    notifier_index = IndexerNotifier()

    group = Group.load(gid)
    owner_folder = group.get_folder()
    owner_folder_full_index = owner_folder.get_full_index()

    user_folder_full_index = {}

    for path_to_resource, data in owner_folder_full_index.iteritems():
        data['uid'] = uid
        if path_to_resource == owner_folder.path:
            # корневая папка
            # поденяем на данные и пути приглашенного пользователя
            user_folder_full_index[path] = data
            user_folder_full_index[path]['id'] = path
            user_folder_full_index[path]['meta']['file_id'] = file_id
            user_folder_full_index[path]['name'] = os.path.basename(path)
        else:
            # не корневой ресурс
            if data['id'].startswith(owner_folder.path + '/'):
                # подменяем пути на пользовательские
                data['id'] = data['id'].replace(owner_folder.path, path, 1)
            user_folder_full_index[data['id']] = data

    if user_folder_full_index:
        notifier_index.push_leave_folder(user_folder_full_index, action, uid, version=version)
