# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

import mpfs.engine.process
from mpfs.common import errors
from mpfs.common.static.tags import STATUS
from mpfs.config import settings
import mpfs.common.static.codes as codes
from mpfs.core.operations import manager as operation_manager
from mpfs.core.queue import mpfs_queue
from mpfs.engine.queue2.celery import BaseTask, app, SilentTaskRetryException


QUEUE2_EXECUTING_OPERATIONS_LIMITATION_ENABLED = settings.queue2['executing_operations_limitation']['enabled']
QUEUE2_EXECUTING_OPERATIONS_LIMITATION_DRY_RUN = settings.queue2['executing_operations_limitation']['dry_run']
QUEUE2_EXECUTING_OPERATIONS_LIMITATION_LIMITED_TYPES = settings.queue2['executing_operations_limitation']['limited_types']
QUEUE2_EXECUTING_OPERATIONS_LIMITATION_MAX_EXECUTING_OPERATIONS = settings.queue2['executing_operations_limitation']['max_executing_operations']
QUEUE2_EXECUTING_OPERATIONS_LIMITATION_RETRY_DELAY_SECS = settings.queue2['executing_operations_limitation']['retry_delay_secs']


default_log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


def check_executing_operations_limit(uid, oid):
    current_op = operation_manager.get_operation(uid, oid)
    if current_op is None or current_op.get_status()[STATUS] == codes.EXECUTING:
        return

    limited_operations_count = operation_manager.count_operations_by_status_and_type(
        uid,
        types=QUEUE2_EXECUTING_OPERATIONS_LIMITATION_LIMITED_TYPES,
        states=[codes.EXECUTING],
        limit=QUEUE2_EXECUTING_OPERATIONS_LIMITATION_MAX_EXECUTING_OPERATIONS + 1,
    )
    if limited_operations_count >= QUEUE2_EXECUTING_OPERATIONS_LIMITATION_MAX_EXECUTING_OPERATIONS:
        if QUEUE2_EXECUTING_OPERATIONS_LIMITATION_DRY_RUN:
            default_log.info('Executing operations limitation dry-run: uid %s should be limited with currently %s operations in EXECUTING state' %
                             (uid, limited_operations_count))
            return

        default_log.info('Executing operations count limit exceeded for uid `%s`, task suspended for `%d` secs' %
                         (uid, QUEUE2_EXECUTING_OPERATIONS_LIMITATION_RETRY_DELAY_SECS))
        if current_op and current_op.dtime < datetime.utcnow() - timedelta(minutes=10):
            current_op.update_dtime()
        raise SilentTaskRetryException(QUEUE2_EXECUTING_OPERATIONS_LIMITATION_RETRY_DELAY_SECS)


def _handle_operation_common(uid, oid, **kwargs):
    if len(kwargs):
        error_log.warning('unexpected arguments in operation %s (%s)' % ('handle_operation', str(kwargs)))

    if QUEUE2_EXECUTING_OPERATIONS_LIMITATION_ENABLED:
        check_executing_operations_limit(uid, oid)

    try:
        operation_manager.process_operation(uid, oid)
    except errors.OperationNotFound:
        error_log.error('operation %s:%s not found' % (uid, oid))
    except Exception as e:
        mpfs_queue.put(
            {
                'uid': uid,
                'oid': oid,
                'error': e
            },
            'operation_failing',
            delay=settings.operations['faildelay']
        )
        raise


@app.task(base=BaseTask)
def handle_operation(uid, oid, context=None, **kwargs):
    _handle_operation_common(uid, oid, **kwargs)


@app.task(base=BaseTask)
def handle_service_operation(uid, oid, context=None, **kwargs):
    _handle_operation_common(uid, oid, **kwargs)


@app.task(base=BaseTask)
def handle_support_operation(uid, oid, context=None, **kwargs):
    _handle_operation_common(uid, oid, **kwargs)


@app.task(base=BaseTask)
def handle_user_preliminary_delete_operation(uid, oid, context=None, **kwargs):
    # handled in djfs-worker
    raise NotImplementedError()


@app.task(base=BaseTask)
def handle_user_permanent_delete_operation(uid, oid, context=None, **kwargs):
    # handled in djfs-worker
    raise NotImplementedError()


@app.task(base=BaseTask)
def handle_filesystem_copy_operation(uid, oid, context=None, **kwargs):
    # handled in djfs-worker
    raise NotImplementedError()
