# -*- coding: utf-8 -*-
import time
import random

import mpfs.engine.process
import mpfs.core.support.controllers as support

from mpfs.common.util import mailer, ctimestamp, time_to_str
from mpfs.common.errors import ResourceNotFound
from mpfs.config import settings
from mpfs.core.address import ResourceId
from mpfs.core.billing import ProductCard
from mpfs.core.bus import Bus
from mpfs.core.email.logic import send_email_async_by_uid
from mpfs.core.filesystem.quota import Quota
from mpfs.core.metastorage.control import disk_info
from mpfs.core.office.logic.sharing_url import sync_public_fields_from_link_data
from mpfs.core.promo_codes.logic.discount_manager import DiscountManager
from mpfs.core.promo_codes.logic.errors import AttemptToActivateArchivedDiscount, AttemptToActivateDiscountRepeatedly
from mpfs.core.services.email_sender_service import email_sender
from mpfs.core.social.share import ShareProcessor
from mpfs.core.social.share import Group
from mpfs.common.errors.share import GroupNotFound
from mpfs.core.social.publicator import Publicator
from mpfs.core.queue import mpfs_queue
from mpfs.core.user.base import User
from mpfs.engine.http import client as http_client
from mpfs.engine.queue2.celery import BaseTask, app
from mpfs.core.services.passport_service import passport


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
FOLLOW_UP_EMAILS = 'follow_up_emails'


PROMO_FOLLOW_UP_EMAILS_DISCOUNT_TEMPLATE = settings.promo[FOLLOW_UP_EMAILS]['discount_template_id']
PROMO_FOLLOW_UP_EMAILS_EMAIL_CAMPAIGN_NAME = settings.promo[FOLLOW_UP_EMAILS]['email_campaign_name']
NO_PROMO_FOLLOW_UP_EMAILS_EMAIL_CAMPAIGN_NAME = 'no_discount_follow_up_email'


@app.task(base=BaseTask)
def handle_post_remove(uid, removed, extra=None, task_data=None, context=None, **kwargs):
    if task_data:  # В task_data данные передаются через монгу, если в removed и extra данных слишком много, а сами они сбрасываются в None
        removed = task_data.data['removed']
        extra = task_data.data['extra']

    if extra is None:
        extra = {}

    fs = Bus()
    fs.save_deleted_files(uid, removed)
    for type_, data in extra.iteritems():
        mpfs_queue.put(data, type_)


@app.task(base=BaseTask)
def handle_update_hardlink(data, context=None, **kwargs):
    Bus().update_hardlinks(data)


@app.task(base=BaseTask)
def handle_symlink_remove(uid, symlinks, task_data=None, context=None, **kwargs):
    if task_data:  # В task_data данные передаются через монгу, если в symlinks данных слишком много, а само оно сбрасывается в None
        symlinks = task_data.data
    Bus()._drop_symlinks_hierarchy(uid, symlinks)


@app.task(base=BaseTask)
def handle_symlink_update(uid, symlinks, task_data=None, context=None, **kwargs):
    if task_data:  # В task_data данные передаются через монгу, если в symlinks данных слишком много, а само оно сбрасывается в None
        symlinks = task_data.data
    Bus()._update_symlinks_hierarchy(uid, symlinks)


@app.task(base=BaseTask)
def handle_call_url(url, data, context=None, **kwargs):
    http_client.open_url(url, pure_data=data, rais=True, headers={'Content-Type': 'application/json'})


@app.task(base=BaseTask)
def handle_send_email(email_to, template_name, template_args, sender_email=None, sender_name=None,
                      context=None, headers=None, **kwargs):
    mailer.send(email_to, template_name, template_args, sender_email, sender_name, headers=headers)


@app.task(base=BaseTask)
def handle_send_email_space(uid, template, data, context=None, **kwargs):
    time.sleep(random.random())
    Quota().send_space_email(uid, template, data)


@app.task(base=BaseTask)
def handle_email_sender_send(email_address, campaign_id, template_args=None, async=True, use_mail_account=False, context=None, **kwargs):
    email_sender.send(email_address, campaign_id, template_args, async, use_mail_account=use_mail_account)


@app.task(base=BaseTask)
def handle_set_group_size(gid, context=None, **kwargs):
    try:
        ShareProcessor().set_group_size(gid)
    except GroupNotFound:
        log.info('Unable to set group size. Group %s not found.' % gid)


@app.task(base=BaseTask)
def handle_make_folder_content_private(uid, path, context=None, **kwargs):
    Bus().make_folder_content_private(uid, path)


@app.task(base=BaseTask)
def handle_make_shared_root_private(actor, gid, context=None, **kwargs):
    _switch_shared_root_publication(actor, gid, private=True)


@app.task(base=BaseTask)
def handle_make_shared_root_public(actor, gid, context=None, **kwargs):
    _switch_shared_root_publication(actor, gid, private=False)


@app.task(base=BaseTask)
def handle_sync_public_fields_from_link_data(uid, file_id, context=None, **kwargs):
    sync_public_fields_from_link_data(uid, file_id)


@app.task(base=BaseTask)
def handle_sync_office_fields_from_link_data(uid, file_id, context=None, **kwargs):
    from mpfs.core.filesystem.dao.file import FileDAO
    from mpfs.core.factory import get_not_removed_resource_by_file_id
    try:
        resource = get_not_removed_resource_by_file_id(uid, file_id)
    except ResourceNotFound:
        error_log.exception('Failed to find resource, while syncing office fields: %s:%s' % (uid, file_id))
        return
    if resource.is_fully_public():
        FileDAO().update_office_fields_from_link_data(uid, file_id)


@app.task(base=BaseTask)
def handle_trash_restore_all(uid, context=None, **kwargs):
    Bus().trash_restore_all(uid)


@app.task(base=BaseTask)
def handle_restore_deleted(uid, path, dest, force, context=None, **kwargs):
    Bus().restore_deleted(uid, path, dest, force)


@app.task(base=BaseTask)
def handle_block_uids_by_hid(hid, type, moderator, comment_text, additional_uids, context=None, **kwargs):
    support.block_uids_by_hid(hid, type, moderator, comment_text, additional_uids)


@app.task(base=BaseTask)
def handle_always_raise(context=None, **kwargs):
    raise NotImplementedError("This handler always raises exception.")


@app.task(base=BaseTask)
def handle_regenerate_preview_for_files(source_file_stid, source_file_name, source_file_size, source_file_mimetype,
                                        old_preview_stid, raw_resource_ids, context=None, **kwargs):
    resource_ids = [ResourceId.parse(raw_resource_id) for raw_resource_id in raw_resource_ids]
    Bus().regenerate_preview_for_files(source_file_stid, source_file_name, source_file_size, source_file_mimetype,
                                       old_preview_stid, resource_ids)


@app.task(base=BaseTask)
def handle_no_discount_follow_up_email(uid, context=None, **kwargs):
    user = User(uid)
    user_info = passport.userinfo(uid)
    language = user_info['language'].lower()
    is_pro = user.disk_pro_enabled
    key = 'no_discount_follow_up_email_last_send'

    no_discount_follow_up_email_last_send = disk_info.value(uid, key).value
    if no_discount_follow_up_email_last_send:
        no_discount_follow_up_email_last_send = no_discount_follow_up_email_last_send.data

    # Рассылка только на русском, как минимум пока
    # Не платный, не получал такую рассылку в последние сутки
    now = ctimestamp()
    if language != 'ru' or is_pro or \
            (no_discount_follow_up_email_last_send and now - no_discount_follow_up_email_last_send < 24 * 60 * 60):
        return

    log.info('Follow up email without discount for uid=%s at time %s(%s)' % (uid, now, time_to_str(now)))
    disk_info.put(uid, key, now)
    send_email_async_by_uid(uid, NO_PROMO_FOLLOW_UP_EMAILS_EMAIL_CAMPAIGN_NAME, locale=language)


@app.task(base=BaseTask)
def handle_with_discount_follow_up_email(uid, context=None, **kwargs):
    user = User(uid)
    user_info = passport.userinfo(uid)
    language = user_info['language'].lower()
    is_pro = user.disk_pro_enabled
    key = '/with_discount_follow_up_email_last_send'

    with_discount_follow_up_email_last_send = disk_info.value(uid, key).value
    if with_discount_follow_up_email_last_send:
        with_discount_follow_up_email_last_send = with_discount_follow_up_email_last_send.data

    # Рассылка только на русском, как минимум пока
    # Не платный, никогда не получал такую рассылку со скидкой
    if language != 'ru' or is_pro or with_discount_follow_up_email_last_send:
        return

    # Проверить что у пользователя нет более мощной скидки
    discount = DiscountManager.get_cheapest_available_discount(uid)
    if discount:
        percentage = ProductCard.LINE_TO_DISCOUNT_PERCENTAGE[discount.provided_line]
        if percentage >= 20:
            return
    try:
        DiscountManager.add_discount_to_user(uid, PROMO_FOLLOW_UP_EMAILS_DISCOUNT_TEMPLATE)
    except (AttemptToActivateArchivedDiscount, AttemptToActivateDiscountRepeatedly):
        return

    now = ctimestamp()
    log.info('Follow up emails discount provided for uid=%s at time %s(%s)' % (uid, now, time_to_str(now)))
    disk_info.put(uid, key, now)
    send_email_async_by_uid(uid, PROMO_FOLLOW_UP_EMAILS_EMAIL_CAMPAIGN_NAME, locale=language)


def _switch_shared_root_publication(uid, gid, private=False):
    group = Group.load(gid)
    group_root = group.get_folder()

    publicator = Publicator()
    switch_publication = publicator.make_one_element_public
    if private:
        switch_publication = publicator.make_one_element_private

    if group.owner != uid:
        switch_publication(resource=group_root)

    for link in group.iterlinks():
        link_root = link.get_folder()
        if link.uid != uid:
            switch_publication(resource=link_root)
