# -*- coding: utf-8 -*-
import datetime
import time
import mpfs.engine.process
from mpfs.config import settings
from mpfs.core.filesystem.dao.trash_cleaner_queue import TrashCleanerQueueDAO
from mpfs.engine.queue2.celery import BaseTask, app
from mpfs.common.util import iterdbuids
from mpfs.core.metastorage.control import support_prohibited_cleaning_users
from mpfs.core.filesystem.base import Filesystem
from mpfs.core.queue import mpfs_queue

SYSTEM_SYSTEM_TRASH_AUTOCLEAN_PERIOD = settings.system['system']['trash_autoclean_period']
SYSTEM_SYSTEM_TRASH_AUTOCLEAN_DELAY = settings.system['system']['trash_autoclean_delay']

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


@app.task(base=BaseTask)
def handle_trash_clean(uid=None, max_processes=None, context=None):
    prohibited_cleaning_users_uids = {
        o['uid'] for o in support_prohibited_cleaning_users.list_all() if o.get('uid')
    }
    fs = Filesystem(method="clean_trash")

    def clean_trash_with_params(uid):
        if uid in prohibited_cleaning_users_uids:
            # у пользователя отключена чистка
            return

        fs.clean_trash(uid, period=SYSTEM_SYSTEM_TRASH_AUTOCLEAN_PERIOD)

    if uid:
        clean_trash_with_params(uid)
    else:
        iterdbuids.run(
            clean_trash_with_params,
            None,
            '/tmp/trash_cleaner_uids_cache',
            max_processes=max_processes
        )


def process_queue(period=SYSTEM_SYSTEM_TRASH_AUTOCLEAN_PERIOD):
    batch_size = 5000
    bound_date = datetime.datetime.now() + datetime.timedelta(days=-period)
    bound_date = int(bound_date.strftime('%Y%m%d'))
    trash_cleaner_queue_dao = TrashCleanerQueueDAO()
    while True:
        tasks = trash_cleaner_queue_dao.find_expired(bound_date)

        for task_index, task in enumerate(tasks):
            if task_index >= batch_size:
                break
            task_id = mpfs_queue.put({'uid': task['uid']}, 'trash_clean')
            log.info("Created task %s for uid %s" % (task_id, task['uid']))
            time.sleep(SYSTEM_SYSTEM_TRASH_AUTOCLEAN_DELAY)
            trash_cleaner_queue_dao.remove(task)
        else:
            break
