# -*- coding: utf-8 -*-

import mpfs.engine.process
import mpfs.core.factory as factory

from mpfs.common import errors
from mpfs.common.errors import ResourceLocked
from mpfs.common.util.filesystem import remove_resource, copy_resource
from mpfs.config import settings
from mpfs.core.address import Address
from mpfs.core.bus import Bus
from mpfs.core.filesystem.helpers.lock import LockHelper, LockUpdateTimer
from mpfs.core.queue import mpfs_queue
from mpfs.core.services.rate_limiter_service import rate_limiter
from mpfs.core.services.staff_service import StaffService
from mpfs.core.user.base import User
from mpfs.core.user.constants import YATEAM_DIR_PATH
from mpfs.core.yateam.logic import resync_user, user_has_nda_rights
from mpfs.engine.queue2.celery import BaseTask, app, SilentTaskRetryException

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


YATEAM_REMOVE_FOLDER_COPY_TO_CHIEF = settings.yateam['remove_folder']['copy_to_chief']
YATEAM_REMOVE_FOLDER_ERROR_NOTIFICATION_ENABLED = settings.yateam['remove_folder']['error_notification_enabled']
YATEAM_REMOVE_FOLDER_ERROR_NOTIFICATION_EMAILS = settings.yateam['remove_folder']['error_notification_emails']


def _remove_resource_and_handle_exceptions(address):
    try:
        remove_resource(address)
    except ResourceLocked as e:
        log.info('ResourceLocked: %s, silent retry' % e.message)
        raise SilentTaskRetryException()
    except Exception as e:
        error_log.exception('NDA folder deletion failed for %s, sending email' % address.uid)
        if YATEAM_REMOVE_FOLDER_ERROR_NOTIFICATION_ENABLED:
            for email in YATEAM_REMOVE_FOLDER_ERROR_NOTIFICATION_EMAILS:
                data = {
                    'email_to': email,
                    'template_name': 'diskSupport',
                    'template_args': {
                        'title': "Failed to remove NDA folder for %s" % address.uid,
                        'body': str(e),
                    },
                }
                mpfs_queue.put(data, 'send_email')


def _copy_resource_and_handle_exceptions(source_address, target_address):
    try:
        copy_resource(source_address, target_address, autosuffix=True)
    except ResourceLocked as e:
        log.info('ResourceLocked: %s, silent retry' % e.message)
        raise SilentTaskRetryException()
    except Exception:
        error_log.exception('NDA folder copy failed from %s to %s' % (source_address.id, target_address.id))


def _make_yateam_folder(uid):
    address = Address.Make(uid, YATEAM_DIR_PATH)

    LockHelper().check(address.id)
    timer = LockUpdateTimer(addresses=[address])
    try:
        timer.start()
        Bus().mkdir(address.uid, address.id)
    finally:
        timer.stop()


@app.task(base=BaseTask)
def handle_move_yateam_folder_to_chief(uid, yateam_uid, context=None, **kwargs):
    user = User(uid)
    if user_has_nda_rights(user):
        log.info('%s is linked to %s, do nothing' % (uid, user.yateam_uid))
        return

    try:
        nda = factory.get_resource(uid, YATEAM_DIR_PATH)
    except errors.ResourceNotFound:
        log.info('%s has no NDA folder, do nothing' % uid)
        return

    try:
        staff_user = StaffService().get_user_info(yateam_uid)
    except errors.StaffUserNotFound:
        log.info('%s(%s) yateam user not in staff, remove NDA folder' % (uid, yateam_uid))
        _remove_resource_and_handle_exceptions(nda.address)
        return

    if not staff_user.is_dismissed and staff_user.uid == uid and user_has_nda_rights(user):
        log.info('%s is linked to %s in staff, do nothing' % (uid, yateam_uid))
        return

    if not YATEAM_REMOVE_FOLDER_COPY_TO_CHIEF:
        log.info('copy to chief is disabled, remove NDA folder')
        _remove_resource_and_handle_exceptions(nda.address)
        return

    if not staff_user.chief_yateam_uid:
        log.info('%s(%s) has no chief in staff, remove NDA folder' % (uid, yateam_uid))
        _remove_resource_and_handle_exceptions(nda.address)
        return

    try:
        staff_chief = StaffService().get_user_info(staff_user.chief_yateam_uid)
    except errors.StaffUserNotFound:
        log.info('%s(%s) has chief %s that is not on staff, remove NDA folder' % (uid, yateam_uid, staff_user.chief_yateam_uid))
        _remove_resource_and_handle_exceptions(nda.address)
        return

    if not staff_chief.uid:
        log.info('%s(%s) has chief %s without uid, remove NDA folder' % (uid, yateam_uid, staff_user.chief_yateam_uid))
        _remove_resource_and_handle_exceptions(nda.address)
        return

    if staff_chief.is_dismissed:
        log.info('%s(%s) has dismissed chief %s, remove NDA folder' % (uid, yateam_uid, staff_user.chief_yateam_uid))
        _remove_resource_and_handle_exceptions(nda.address)
        return

    try:
        chief = User(staff_chief.uid)
    except errors.StorageInitUser:
        log.info('%s(%s) has chief %s without Yandex.Disk, remove NDA folder' % (uid, yateam_uid, staff_user.chief_yateam_uid))
        _remove_resource_and_handle_exceptions(nda.address)
        return

    target_address = Address.Make(chief.uid, YATEAM_DIR_PATH + '/' + staff_user.yateam_login)
    _copy_resource_and_handle_exceptions(nda.address, target_address)
    _remove_resource_and_handle_exceptions(nda.address)


@app.task(base=BaseTask)
def handle_update_user_yateam_status(uid, context=None, **kwargs):
    if not rate_limiter.is_limit_exceeded('handle_update_user_yateam_status', 'default'):
        resync_user(uid)


@app.task(base=BaseTask)
def handle_restore_yateam_folder(uid, context=None, **kwargs):
    """Создаёт НДА папку яндексоидам после удаления\перемещения.

       ПО не может правильно засинкать создание папки после её удаления. Удаляем все дельты, чтобы зафорсить полное
       обновление в ПО.

       Если пользователь приглашён в общие папки, удаление дельт не поможет (возвращаются свои дельты и дельты всех
       владельцев общих папок). Для таких пользователей дельты не удаляем, ПО работать не будет.
    """
    user = User(uid)
    if not user.is_yateam():
        log.info('%s is not yateam, do nothing' % uid)
        return

    address = Address.Make(uid, YATEAM_DIR_PATH)
    try:
        yateam_folder = factory.get_resource_by_address(uid, address)
        log.info('%s already exists, do nothing' % yateam_folder.address.id)
    except errors.ResourceNotFound:
        pass

    try:
        LockHelper().check(address.id)
    except ResourceLocked as e:
        log.info('ResourceLocked: %s, silent retry' % e.message)
        raise SilentTaskRetryException()

    _make_yateam_folder(uid)
