# -*- coding: utf-8 -*-
import datetime
import itertools

from uuid import uuid4

from mpfs.config import settings
from mpfs.common.errors import BadRequestError
from mpfs.common.errors.share import GroupNotFound
from mpfs.engine.process import get_default_log, get_error_log
from mpfs.common.util import merge2, SuppressAndLogExceptions
from mpfs.common.util.slice_tools import offset_limit_to_slice
from mpfs.core.address import ResourceId
from mpfs.core import factory
from mpfs.core.filesystem.dao.resource import ResourceDAO
from mpfs.core.last_files.dao.cache import LastFilesCacheDAOItem, LastFilesCacheDAO
from mpfs.core.social.share.group import Group
from mpfs.core.queue import mpfs_queue
from mpfs.core.filesystem.resources.disk import append_meta_to_office_files


LAST_FILES_ITEMS_LIMIT = settings.last_files['items_limit']
LAST_FILES_CACHE_UPDATE_TASK_ENABLE = settings.last_files['cache_update_task']['enable']
LAST_FILES_CACHE_UPDATE_TASK_DELAY = settings.last_files['cache_update_task']['delay']


class SharedLastFilesProcessor(object):
    def __init__(self):
        self._cache_dao = LastFilesCacheDAO()
        self._resource_dao = ResourceDAO()

    def get_resources(self, uid, limit=10):
        """Получить ресурсы из кеша

        Отдает отсортированные по mtime ресурсы
        """
        cache_items = self._cache_dao.get(uid, limit=limit)
        if not cache_items:
            return []
        resources = factory.get_resources_by_resource_ids(
            uid,
            [ResourceId(i.owner_uid, i.file_id) for i in cache_items],
            enable_service_ids=['/disk'],
            enable_optimization=True,
            enable_multithreading=True
        )
        resources = filter(None, resources)
        if len(cache_items) != len(resources):
            get_default_log().info('Last files cache mismatch. Uid: %s', uid)
            try:
                # собираем группы, от которых мы не получили ресурсы
                file_id_cache_map = {c.file_id: c for c in cache_items}
                resources_file_ids = {r.meta['file_id'] for r in resources}
                not_found_file_ids = file_id_cache_map.viewkeys() ^ resources_file_ids
                need_to_update_gids = {file_id_cache_map[i].gid for i in not_found_file_ids}
                for gid in need_to_update_gids:
                    self.update_for_group_async(gid)
            except Exception:
                get_error_log().exception("Can't put update cache jobs.")
        return resources

    def update_for_group(self, gid):
        """Обновляет кеши для группы"""
        self._update_for_group(gid, self._update_cache_selectively)

    def force_update_for_group(self, gid):
        """Полностью дропает кэш и записывает заново для обновления группы"""
        self._update_for_group(gid, self._update_cache_whole)

    def _update_cache_selectively(self, gid, uid, cache_items):
        """Выборочное удаление и вставка записей из кэша"""
        actual_cache_items = self._cache_dao.get_by_gid_uid(gid, uid)
        cache_items_to_insert, cache_items_to_drop = self.diff_caches_for_gid_uid(cache_items, actual_cache_items)
        if cache_items_to_drop:
            ids_to_drop = [c.id for c in cache_items_to_drop]
            self._cache_dao.drop_by_uid_ids(uid, ids=ids_to_drop)
        self._cache_dao.set(uid, cache_items_to_insert)

    def _update_cache_whole(self, gid, uid, cache_items):
        """Полное удаление кэша пользователя с записью новых данных"""
        self._cache_dao.drop_by_gid_uid(gid, uid)
        self._cache_dao.set(uid, cache_items)

    def _update_for_group(self, gid, update_function):
        """Обновляет кэш для группы"""
        try:
            group = Group.load(gid=gid)
        except GroupNotFound:
            self._cache_dao.drop(gid)
            return
        group_links = list(group.iterlinks())
        if not group_links:
            self._cache_dao.drop(group.gid)
            return
        file_dao_items = list(self._resource_dao.get_last_files_for_subtree(
            group.owner,
            group.path,
            limit=LAST_FILES_ITEMS_LIMIT
        ))
        if not file_dao_items:
            self._cache_dao.drop(group.gid)
            return
        now_dt = datetime.datetime.now()
        updated_uids = []
        for group_link in group_links:
            cache_items = []
            for file_dao_item in file_dao_items:
                cache_item = LastFilesCacheDAOItem()
                cache_item.id = uuid4().hex
                cache_item.uid = group_link.uid
                cache_item.owner_uid = group.owner
                cache_item.gid = group.gid
                cache_item.file_id = file_dao_item.file_id
                cache_item.creation_time = now_dt
                cache_item.file_date_modified = file_dao_item.modification_time
                cache_items.append(cache_item)
            update_function(group.gid, group_link.uid, cache_items)
            updated_uids.append(group_link.uid)
        self._cache_dao.drop(group.gid, ignore_uids=updated_uids)

    @staticmethod
    def update_for_group_async(gid):
        """Асинхронно обновляет кеши для группы"""
        if not LAST_FILES_CACHE_UPDATE_TASK_ENABLE:
            return
        with SuppressAndLogExceptions(get_default_log(), Exception):
            mpfs_queue.put({'gid': gid}, 'update_last_files_cache',
                           deduplication_id='update_last_files_cache__%s' % gid,
                           delay=LAST_FILES_CACHE_UPDATE_TASK_DELAY)

    @staticmethod
    def diff_caches_for_gid_uid(new_cache_items, old_cache_items):
        # откидываем tz, т.к. в PG дата хранится с tz, а в монге в виде ts и получаемый из него dt не содержит таймзоны
        # так же считаем, что у всех наших серверов tz +3
        new_cache_map = {(i.file_id, i.file_date_modified.replace(tzinfo=None)): i for i in new_cache_items}
        old_cache_map = {(i.file_id, i.file_date_modified.replace(tzinfo=None)): i for i in old_cache_items}
        new_cache_map_viewkeys = new_cache_map.viewkeys()
        old_cache_map_viewkeys = old_cache_map.viewkeys()
        keys_to_insert = new_cache_map_viewkeys - old_cache_map_viewkeys
        keys_to_drop = old_cache_map_viewkeys - new_cache_map_viewkeys
        return [new_cache_map[i] for i in keys_to_insert], [old_cache_map[i] for i in keys_to_drop]


def get_last_files(uid, offset=0, amount=10, req=None):
    slice_obj = offset_limit_to_slice(offset, amount)
    if slice_obj.stop > LAST_FILES_ITEMS_LIMIT:
        raise BadRequestError(extra_msg="Can't get over %i item" % LAST_FILES_ITEMS_LIMIT)

    resource_dao_items_gen = ResourceDAO().get_last_files(uid, limit=slice_obj.stop)
    self_resources = (factory.get_resource_from_doc(uid, i.get_mongo_representation()) for i in resource_dao_items_gen)
    shared_resources = SharedLastFilesProcessor().get_resources(uid, limit=slice_obj.stop)

    result = list(itertools.islice(
        merge2(
            (self_resources, shared_resources),
            key=lambda r: r.mtime,
            reverse=True
        ),
        slice_obj.start,
        slice_obj.stop
    ))
    if req:
        # добавляем офисные поля. В идеале отказаться от реквеста
        for resource in result:
            append_meta_to_office_files(resource, req)
            if req.meta is not None:
                if 'views_counter' in req.meta or not req.meta:
                    # not req.meta = передать всю мету
                    resource.load_views_counter()
    return result
