# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import itertools

import mpfs.engine.process

from mpfs.config import settings
from mpfs.common.util.filetypes import builtin_extensions
from mpfs.common.errors import FolderNotFound, LentaBadArgumentsUnhandledError, ResourceNotFound
from mpfs.core import factory
from mpfs.core.address import Address, LentaResourceId, ResourceId
from mpfs.core.bus import Bus
from mpfs.core.global_gallery.logic.controller import GlobalGalleryController
from mpfs.core.lenta.utils import is_resource_visible_for_user
from mpfs.core.queue import mpfs_queue
from mpfs.core.services.passport_service import Passport
from mpfs.core.metastorage.decorators import enable_secondary_preferred_if, user_exists_and_not_blocked
from mpfs.core.user.base import User
from mpfs.core.user.constants import PHOTOUNLIM_AREA, PHOTOUNLIM_AREA_PATH

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

FEATURE_TOGGLES_SECONDARY_PREFERRED_ENABLED = settings.feature_toggles['secondary_preferred_enabled']
LENTA_RESOURCE_OPTIMIZATION_ENABLED = settings.lenta['resource_optimization_enabled']
LENTA_MAX_RESULT_COUNT = settings.lenta['max_results_count']


def _construct_iteration_key(resources):
    """
    Создать iteration_key для получения следующей порции ресурсов.

    Считает с конца количество ресурсов с одинаковым mtime и на основе этого
    строит iteration_key, зная наименьший mtime и количество (offset).
    """

    if not resources:
        return
    previous = None
    offset = 0
    for idx in xrange(len(resources) - 1, -1, -1):
        current = resources[idx]
        if previous and current.mtime != previous.mtime:
            break
        else:
            offset += 1
        previous = current
    mtime_lte = previous.mtime
    iteration_key = '%s/%s' % (mtime_lte, offset)
    return iteration_key


def _do_actions_for_new_empty_block(path, qs, iteration_key):
    # записать в лог сообщение, которое подхватывает бэкенд ленты и удаляет у себя блок
    log.info('lenta_block_list empty response uri=%s' % path)
    if not iteration_key:
        mpfs_queue.put({'qs': qs}, 'lenta_delete_empty_block')


@user_exists_and_not_blocked
@enable_secondary_preferred_if(FEATURE_TOGGLES_SECONDARY_PREFERRED_ENABLED)
def lenta_block_list(req):
    """
    Отдаёт содержимое блока ленты.

    Для /photounlim всегда достаем 1000 записей, сортируем их по etime, а затем возвращаем в соответствии с offset/limit

    Query string аргументы:
      * uid [обязательный] -- uid пользователя просматривающего ленту.
      * mtime_gte [обязательный] -- минимальне значение mtime файлы с которым будут попадать в листинг.
      * path -- путь к папке по которой требуется получить листинг.
      * resource_id -- resource_id папки по которой требуется получить листинг,
                       для корневой папок может быть <uid>:/disk т.к. у них нет file_id.
      * media_type -- фильтр по медиа типу файлов, поддерживает список через запятую.
      * mtime_lte -- максимальное значение mtime файлы с которым будут попадать в листинг.
      * filter_photounlim -- если 1, то на ресурсы из /photounlim должны отдавать 404.
      * amount [по умолчанию 100] -- максимальное количество записей на странице.
      * offset -- отступ от начала выборки.
    """
    request_uri = req.http_req.environ.get('REQUEST_URI')
    request_query_string = req.http_req.environ.get('QUERY_STRING')

    # запоминаем эти значения здесь, поскольку в загрузке ресурсов они могут поменяться.
    amount = req.args['bounds']['amount']
    offset = req.args['bounds']['offset']

    # Если передали /photostream, то ищем в фотостримных папках и возвращаем первый непустой результат
    if req.path == '/photostream':
        camera_path = Bus(request=req).get_photostream_address(req.uid).path
        if User(req.uid).is_unlimited_autouploading_enabled():
            paths = [PHOTOUNLIM_AREA_PATH, camera_path]
        else:
            paths = [camera_path, PHOTOUNLIM_AREA_PATH]
        resource = _try_to_load_from_paths(req, paths, request_uri, request_query_string)
    else:
        resource = _try_to_load_from_path(req, request_uri, request_query_string)

    if resource.path == PHOTOUNLIM_AREA_PATH:
        resource.children_items['files'] = sorted(
            resource.children_items['files'], key=lambda x: int(x.meta.get('etime', 0)), reverse=True
        )[offset:offset + amount]
    resources = list(itertools.chain([resource], resource.children_items['files']))

    if resource.meta['total_results_count'] == 0:
        _do_actions_for_new_empty_block(request_uri, request_query_string, req.iteration_key)

    if offset + len(resource.children_items['files']) < resource.meta['total_results_count']:
        # есть еще ресурсы для будущей догрузки
        # resource.meta['total_results_count'] - количество ресурсов, удовлетворяющих запросу без учета offset
        iteration_key = _construct_iteration_key(resource.children_items['files'])
    else:
        # последняя партия (количество ресурсов в ней не более запрошенного), всё ок
        iteration_key = None

    resource.meta['iteration_key'] = iteration_key

    if (req.meta and 'users' in req.meta) or not req.meta:
        modifiers = {r.meta['modify_uid'] for r in resources}
        users = _load_users(modifiers)
        resource.meta['users'] = users

    load_source_ids = req.meta and 'source_ids' in req.meta
    if load_source_ids:
        GlobalGalleryController.load_and_set_source_ids_for_resources(resources)

    return resources


def _load_resource(req, path_to_use=None, force_set_request=False):
    """Загрузить папку и вложенные файлы (блок ленты), соответствующие условиям запроса."""
    uid = req.uid
    path = path_to_use or req.path
    modify_uid = req.modify_uid
    raw_lenta_resource_id = req.resource_id
    filter_photounlim = req.filter_photounlim

    if req.media_type:
        media_types = req.media_type.split(',')
        for media_type in media_types:
            if media_type not in builtin_extensions:
                raise LentaBadArgumentsUnhandledError('Got unsupported media_type %s.' % media_type)

    if path:
        if filter_photounlim:
            address = Address.Make(uid, path)
            if address.storage_name == PHOTOUNLIM_AREA:
                raise ResourceNotFound('Filtering resources from /photounlim')

        resource = factory.get_resource(uid, path)
    elif raw_lenta_resource_id:
        enabled_service_ids = ['/disk']
        if not filter_photounlim:
            enabled_service_ids.append(PHOTOUNLIM_AREA_PATH)
        fs = Bus(request=req)
        common_id = LentaResourceId.parse(raw_lenta_resource_id).make_common_id()
        if isinstance(common_id, ResourceId):
            resources = fs.resources_by_resource_ids(
                uid, [common_id], enable_service_ids=tuple(enabled_service_ids),
                enable_optimization=LENTA_RESOURCE_OPTIMIZATION_ENABLED)
            if not resources:
                raise ResourceNotFound(str(common_id))
            else:
                resource = resources[0]
        elif isinstance(common_id, Address):
            # ... и таки да -- это костыль.
            if filter_photounlim and common_id.storage_name == PHOTOUNLIM_AREA:
                raise ResourceNotFound(str(common_id))

            resource = factory.get_resource(uid, common_id)
            if not is_resource_visible_for_user(resource, uid):
                raise ResourceNotFound('Not available for user')
        else:
            raise NotImplementedError()
    else:
        raise ValueError('path or resource_id parameters must be defined')

    if resource.is_file:
        raise FolderNotFound()

    # Адский хак. Не повторять! Опасно для здоровья!
    # Мы не можем правильно сформировать поле с фильтром в frontend, т.к. там мы ничего не знаем о ресурсе.
    if modify_uid:
        is_shared = resource.is_group or resource.is_shared
        if not is_shared:
            # нет фильтра  для обычных файлов
            modify_uid_filter = None
        elif modify_uid == resource.group.owner:
            # фильтр для владельцa группы
            modify_uid_filter = {
                '$or': [
                    {'data.modify_uid': {'$exists': False}},
                    {'data.modify_uid': modify_uid}
                ]
            }
        else:
            # фильтр для для приглашенных в группу пользователей
            modify_uid_filter = {
                'data.modify_uid': modify_uid
            }

        if modify_uid_filter:
            req.args_data.data['filter']['modify_uid'] = modify_uid_filter
    if resource.path == PHOTOUNLIM_AREA_PATH:
        req.args_data.data['bounds']['amount'] = LENTA_MAX_RESULT_COUNT
        req.args_data.data['bounds']['offset'] = 0
    resource.set_request(req, force_set_request)

    # вопреки названию, этот метод загружает из базы содержимое папки
    # и добавляет некоторые дополнительные атрибуты в дочерние файлы
    resource.list()

    return resource


def _try_to_load_from_paths(req, paths, request_uri, request_qs):
    # Если нашли ресурс с детьми - возвращаем
    # Если нашли хотя бы один без детей, а остальных не существует - возвращаем того, что без детей
    # Если ни одного ресурса не существует - бросаем исключение от послледней итерации
    resource = None
    for e, path in enumerate(paths):
        is_last = len(paths) - 1 == e
        try:
            resource = _load_resource(req, path_to_use=path, force_set_request=True)
        except FolderNotFound:
            if is_last and resource is None:
                raise
            continue
        except ResourceNotFound:
            _do_actions_for_new_empty_block(request_uri, request_qs, req.iteration_key)
            if is_last and resource is None:
                raise
            continue
        if resource.meta['total_results_count'] > 0:
            return resource
    return resource


def _try_to_load_from_path(req, request_uri, request_qs):
    try:
        resource = _load_resource(req)
    except FolderNotFound:
        raise
    except ResourceNotFound:
        _do_actions_for_new_empty_block(request_uri, request_qs, req.iteration_key)
        raise
    return resource


def _load_users(uids):
    """Загрузить информацию по пользователям из паспорта."""
    uids = set(uids)
    users_data = Passport().bulk_userinfo(uids=uids, load_all_emails=False, add_private_data=True)
    users = []
    for user_data in users_data:
        if user_data['uid'] is None:
            continue
        users.append({
            'uid': user_data['uid'],
            'login': user_data['login'],
            'firstname': user_data['firstname'],
            'lastname': user_data['lastname'],
            'sex': user_data['sex']
        })
    return users
