# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from mpfs.common.util import from_json
from mpfs.common.errors.lenta import LentaEmptyBlockError, LentaNoRequestBodyError, LentaNotSupportedBlockTypeError
from mpfs.config import settings
from mpfs.core.address import ResourceId
from mpfs.core.factory import get_resources_by_resource_ids
from mpfs.core.lenta.logic.lenta_block_list import _load_resource
from mpfs.core.albums.interface import _albums_create
from mpfs.core.user.base import User
from mpfs.core.user.constants import NY_2018_ALBUM_NAMES
from mpfs.metastorage.postgres.schema import AlbumType

MAX_BLOCK_SIZE_TO_CREATE_ALBUM_FOR = settings.lenta['max_block_size_to_create_album_for']


def lenta_block_public_link(req):
    """
    Позволяет поделиться блоком ленты.

    Если в блоке один элемент, возвращается публичная ссылка на этот элемент.
    Если в блоке несколько элементов, создаётся альбом и возвращается публичная ссылка на него.
    Если в блоке больше MAX_BLOCK_SIZE_TO_CREATE_ALBUM_FOR элементов, альбом создаётся только для первых MAX_BLOCK_SIZE_TO_CREATE_ALBUM_FOR элементов.

    Принимает в теле блоки ленты разных типов.

    Примеры:

    {
      "id" : "000000152404848920204510000001523351943024",
      "files_count" : 1000,
      "media_type" : "image",
      "folder_id" : "26345337:4800599aca389116b7347159b0440ac661509b446a056a9c10307d73fd1af198",
      "revision" : 7197,
      "area" : "photounlim",
      "type" : "content_block",
      "collection_id" : "index",
      "action" : "photounlim",
      "order" : 7197,
      "modifier_uid" : "26345337",
      "mtime" : 1524130902859,
      "mfrom" : 1524048187000,
      "group_key" : "26345337:4800599aca389116b7347159b0440ac661509b446a056a9c10307d73fd1af198:image:26345337",
      "mtill" : 1524134587000
    }

    {
      "id" : "000000152828390609302670000001527852949720",
      "interval_start" : 1464607102173,
      "resource_ids" : [
        "26345337:924fdc5f70abe1c817ae49ce23c6d3576cc7529ca1427ab3d434606732e2e4c8",
        "26345337:99bfe5e01049488e9da3ff61fe9a6f6708ceb1949e7a4057114a380725f9c822",
        "26345337:b9e6cfd808e571f575a1052ddce3cf276a3db8a6c1ea0eb457860f7c4e7b3b02",
        "26345337:c5dbc53a3181d41b9339cc5e405d8d2ca2ed44aac88257b73a1d708728e5d91e",
        "26345337:d4fb1014f7ddc46fcb7ec793a80d68000fd21e013ad7ab19ba1f54e147a08b18",
        "26345337:eb0ecac559b2c84c5f400c2b30fd52165e31f6e5bd8e5fb6a4362a0f72dd07ec"
      ],
      "revision" : 7826,
      "interval_end" : 1465211902173,
      "type" : "photo_remind_block",
      "collection_id" : "index",
      "years_ago" : 2,
      "order" : 7826,
      "mtime" : 1528283906093,
      "mfrom" : 1528283606093,
      "group_key" : "1465211902173-2",
      "mtill" : 1528370006093
    }

    https://st.yandex-team.ru/CHEMODAN-44440

    Перед тем как завязываться на новое поле, нужно проверить, что оно есть в body_serializer_cls хэндлера
    LentaBlockPublicLinkHandler, иначе оно не будет прокидываться из REST
    """
    if not req.http_req.data:
        raise LentaNoRequestBodyError()

    data = from_json(req.http_req.data)
    if not data:
        raise LentaNoRequestBodyError()

    if data['type'] == 'content_block':
        resources = _get_resources_from_content_block(req, data)
    elif data['type'] in {'photo_remind_block', 'photo_selection_block'}:
        resources = get_resources_by_resource_ids(req.uid, [ResourceId.parse(x) for x in data['resource_ids'][:MAX_BLOCK_SIZE_TO_CREATE_ALBUM_FOR]])
    else:
        raise LentaNotSupportedBlockTypeError()

    if len(resources) == 0:
        raise LentaEmptyBlockError()

    user = User(req.uid)
    icon_type = data.get('icon_type')
    album_title = None
    if icon_type == 'new_year_2018_magic_wand':
        locale = user.get_supported_locale()
        album_title = NY_2018_ALBUM_NAMES[locale]

    raw_album = {
        'cover': 0,
        'items': [{'type': 'resource', 'path': x.id} for x in resources if x is not None],
        'title': album_title,
        'album_type': AlbumType.PERSONAL.value,
    }
    return _albums_create(req, raw_album)


def _get_resources_from_content_block(req, data):
    req.path = None
    req.modify_uid = data['modifier_uid']
    req.resource_id = data['folder_id']
    req.filter_photounlim = False
    req.media_type = data.get('media_type', None)

    req.args_data.data['bounds'] = {}
    req.args_data.data['bounds']['sort'] = 'mtime'
    req.args_data.data['bounds']['order'] = 0
    req.args_data.data['bounds']['offset'] = 0
    req.args_data.data['bounds']['amount'] = min(data['files_count'], MAX_BLOCK_SIZE_TO_CREATE_ALBUM_FOR)

    req.args_data.data['filter'] = {}
    req.args_data.data['filter']['resource_type'] = 'file'
    if 'media_type' in data:
        req.args_data.data['filter']['media_type'] = data['media_type']

    mtime_gte = data.get('mfrom', None)
    mtime_lte = data.get('mtill', None)

    if mtime_gte and mtime_lte:
        req.args_data.data['filter']['mtime_interval'] = '%s,%s' % (int(mtime_gte) // 1000, int(mtime_lte) // 1000)
    elif mtime_gte:
        req.args_data.data['filter']['mtime_gte'] = int(mtime_gte) // 1000
    elif mtime_lte:
        req.args_data.data['filter']['mtime_lte'] = int(mtime_lte) // 1000

    folder_resource = _load_resource(req)
    return folder_resource.children_items['files']
