# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import itertools
import operator

from mpfs.common.util import merge2
from mpfs.common import errors
from mpfs.core.services.passport_service import Passport
from mpfs.core.social.share import Group
from mpfs.core.social.share.invite import GroupInvites
from mpfs.core.social.share.processor import ShareProcessor


passport_service = Passport()


class LentaListGroupInvitesIterationKey(object):
    """
    Виды ключа итерации:
        * '' - взять верхушку решившихся и добрать по надобности из нерешившихся
        * ctime_accepted/offset_accepted/ctime_rejected/offset_rejected/ctime_new/offset_new
    """

    SEP = '/'
    NONE_PLACEHOLDER = 'None'

    def __init__(self, ctime_accepted=0, offset_accepted=0, ctime_rejected=0, offset_rejected=0,
                 ctime_new=0, offset_new=0):
        self.ctime_accepted = ctime_accepted
        self.offset_accepted = offset_accepted
        self.ctime_rejected = ctime_rejected
        self.offset_rejected = offset_rejected
        self.ctime_new = ctime_new
        self.offset_new = offset_new

    @classmethod
    def _split_into_parts(cls, key):
        if not key:
            key = ''
        return key.split(cls.SEP)

    @classmethod
    def _parse_value(cls, value):
        assert isinstance(value, basestring)
        if value.isdigit():
            return int(value)
        elif value == cls.NONE_PLACEHOLDER:
            return None
        else:
            return ValueError('Expected integer or None')

    @classmethod
    def parse(cls, key):
        assert isinstance(key, basestring)

        if not key:
            return cls(None, 0, None, 0, None, 0)

        parts = cls._split_into_parts(key)
        if len(parts) != 6:
            raise ValueError('Wrong `key` format.')

        (
            ctime_accepted, offset_accepted,
            ctime_rejected, offset_rejected,
            ctime_new, offset_new
        ) = map(cls._parse_value, parts)

        return cls(
            ctime_accepted=ctime_accepted, offset_accepted=offset_accepted,
            ctime_rejected=ctime_rejected, offset_rejected=offset_rejected,
            ctime_new=ctime_new, offset_new=offset_new
        )

    def get_ctime_accepted(self):
        return self.ctime_accepted

    def get_offset_accepted(self):
        return self.offset_accepted

    def get_ctime_rejected(self):
        return self.ctime_rejected

    def get_offset_rejected(self):
        return self.offset_rejected

    def get_ctime_new(self):
        return self.ctime_new

    def get_offset_new(self):
        return self.offset_new

    @classmethod
    def _serialize_value(cls, value):
        if value is None:
            return cls.NONE_PLACEHOLDER
        else:
            return value

    def serialize(self):
        return '%s/%s/%s/%s/%s/%s' % tuple(map(self._serialize_value, (
            self.ctime_accepted, self.offset_accepted,
            self.ctime_rejected, self.offset_rejected,
            self.ctime_new, self.offset_new
        )))


def _get_iteration_key_from_raw_result(raw_result, limit):
    """
    :param raw_result: Список объектов, из которых необходимо выбрать максимум `limit` штук и отправить в ответе.
    :param limit: Максимальное количество объектов, которые будут отправлены в результате.
    :return: Ключ итерации для следующего запроса, None в случае если данных для следующего запроса нет.
    :rtype: str | None
    """

    def _is_accepted_invite_data(data):
        """Проверить является ли объект подтвержденным «приглашением»."""
        return not hasattr(data, 'status')

    def _is_rejected_invite_data(data):
        """Проверить является ли объект отклонённым приглашением."""
        return hasattr(data, 'status') and data.status == GroupInvites.Status.REJECTED

    def _is_new_invite_data(data):
        """Проверить является ли объект новым приглашением."""
        return hasattr(data, 'status') and data.status == GroupInvites.Status.NEW

    if not raw_result:
        return None

    if len(raw_result) < limit + 1:
        # даже из нескольких мест не смогли набрать нужного количества плюс одного элемента
        # а это значит больше в базе ничего нет для следующего запроса
        return None

    result = raw_result[:limit]
    # первый лишний элемент в результатах,
    # начиная с которого идут уже лишние данные
    last_result_object = result[limit-1]
    last_result_object_ctime = last_result_object.ctime
    first_extra_object = raw_result[limit]
    first_extra_object_ctime = first_extra_object.ctime

    if _is_new_invite_data(first_extra_object):
        # если лишний элемент - новый, значит далее идут уже только новые, тк новые в конце
        offset_new = len([x for x in result if x.ctime == first_extra_object_ctime and _is_new_invite_data(x)])
        return LentaListGroupInvitesIterationKey(
            ctime_accepted=0,
            offset_accepted=0,
            ctime_rejected=0,
            offset_rejected=0,
            ctime_new=first_extra_object_ctime,
            offset_new=offset_new
        ).serialize()
    else:
        # если лишний элемент - принявший или подтвердивший, то как минимум в следующем запросе его можно вернуть
        offset_accepted = len(
            [x for x in result if x.ctime == last_result_object_ctime and _is_accepted_invite_data(x)]
        )
        offset_rejected = len(
            [x for x in result if x.ctime == last_result_object_ctime and _is_rejected_invite_data(x)]
        )
        return LentaListGroupInvitesIterationKey(
            ctime_accepted=last_result_object_ctime,
            offset_accepted=offset_accepted,
            ctime_rejected=last_result_object_ctime,
            offset_rejected=offset_rejected,
            ctime_new=0,
            offset_new=0
        ).serialize()


def _parse_and_get_data(iteration_key, gid, limit):
    """Распарсить ключ и получить из базы данные, необходимые для возвращения результата."""

    accepted = ShareProcessor.get_invite_like_objects(
        status=GroupInvites.Status.ACTIVATED,
        gid=gid,
        limit=limit,
        ctime_lte=iteration_key.get_ctime_accepted(),
        offset=iteration_key.get_offset_accepted()
    )

    rejected = ShareProcessor.get_invite_like_objects(
        status=GroupInvites.Status.REJECTED,
        gid=gid,
        limit=limit,
        ctime_lte=iteration_key.get_ctime_rejected(),
        offset=iteration_key.get_offset_rejected()
    )

    new = ShareProcessor.get_invite_like_objects(
        status=GroupInvites.Status.NEW,
        gid=gid,
        limit=limit,
        ctime_lte=iteration_key.get_ctime_new(),
        offset=iteration_key.get_offset_new()
    )

    return list(
        itertools.islice(
            merge2(
                (accepted, rejected),
                key=operator.attrgetter('ctime'),
                reverse=True
            ),
            limit
        )
    ) + new


def _lenta_list_group_invites(group, limit, iteration_key=None):
    assert limit > 0

    if not isinstance(group, Group):
        group = Group.load(group)

    gid = group.gid

    # для того чтобы точно знать есть ли еще последующие результаты мы будем брать всегда на 1
    # элемент больше из коллекции, и если его не сможем получить, значит эта порция последняя.
    # при возвращении результата мы будем его убирать из ответа.
    magic_limit = limit + 1

    data_list = _parse_and_get_data(
        iteration_key=iteration_key,
        gid=gid,
        limit=magic_limit
    )

    next_iteration_key = _get_iteration_key_from_raw_result(data_list, limit)
    return data_list[:limit], next_iteration_key


def lenta_list_group_invites(req):
    """Возвратить список инвайтов в общую папку.

    Результат можно условно разделить на 2 части (идущие последовательно).
    В первой части идут вперемешку инвайты, подтвержденные или отклонённые, отсортированные
    по убыванию времени этого события.
    Во второй части идут новые (не отклонённые и
    не подтверждённые), сортированные по времени убывания создания приглашения.

    Принимаемые GET-параметры:
        :uid: Идентификатор пользователя, делающего запрос.
        :gid: Идентификатор группы.
        :limit: Максимальное количество возвращаемых результатов.
        :iteration_key: Ключ итерации. Передавая его из возвращаемого результата в следующий вызов функции
        можно получить следующую порцию результатов. Первый вызов обычно делается с опущенным ключом.
    """
    gid = req.gid
    limit = req.limit
    result = {
        'data': [],
        'iteration_key': None
    }
    if not limit:
        return result
    raw_iteration_key = req.iteration_key
    if not raw_iteration_key:
        raw_iteration_key = ''
    iteration_key = LentaListGroupInvitesIterationKey.parse(raw_iteration_key)

    group = Group.load(gid)
    if req.uid != group.owner:
        raise errors.Forbidden()

    r, iteration_key = _lenta_list_group_invites(group, limit, iteration_key)
    data = []
    passport_user_info_fields = ('display_name', 'decoded_email', 'avatar')
    for invite in r:
        if not hasattr(invite, 'status'):
            user_info = passport_service.userinfo_summary(invite.uid, fields=passport_user_info_fields)
            data.append({
                'status': GroupInvites.Status.ACTIVATED,
                'name': user_info['display_name'],
                'uid': user_info['uid'],
                'userid': user_info['decoded_email'],
                'avatar': user_info['avatar'],
                'ctime': invite.ctime
            })
        elif invite.status in (GroupInvites.Status.REJECTED, GroupInvites.Status.NEW):
            data.append({
                'status': invite.status,
                'name': invite.name,
                'uid': invite.uid,
                'userid': invite.universe_login,
                'avatar': invite.avatar,
                'service': invite.universe_service,
                'ctime': invite.ctime
            })
        else:
            raise ValueError('Unexpected `status`. Got %s.' % invite.status)

    result['data'] = data
    result['iteration_key'] = iteration_key
    return result
