# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from mpfs.common import errors
from mpfs.core.factory import get_resource
from mpfs.core.services.clck_service import clck
from mpfs.core.event_history import logger
from mpfs.core.social.publicator import Publicator
from mpfs.core.address import Address
from mpfs.core.albums.logic.common import get_public_album
from mpfs.core.lenta.utils import LentaMediaType


def lenta_public_log_visit(req):
    """
    Залогировать посещение пользователем публичной ссылки.

    Принимаемые GET-параметры:
        * uid - идентификатор пользователя, посетившего публичную ссылку.
        * short_url - короткая ссылка, которую посетил пользователь (пример: https://yadi.sk/d/Fui9LPbuPpva).
    """
    if not req.uid:
        raise errors.BadArguments('Required non empty value for `uid`.')
    if not req.short_url:
        raise errors.BadArguments('Required non empty value for `short_url`.')

    uid, short_url = req.uid, req.short_url

    log_data = {
        'uid': uid,
        'short_url': short_url
    }

    public_hash = clck.short_url_to_public_hash(short_url)

    if not public_hash:
        # не смогли разрезолвить урл в public_hash, пишем, что есть
        logger.log_raw_event(event_type='public-visit', data=log_data)
        return

    log_data['public_hash'] = public_hash

    if clck.check_short_url_is_album_url(short_url):
        album = get_public_album(public_hash, uid)
        uid_is_owner = album.uid == uid
        log_data.update({
            'uid_is_owner': uid_is_owner,
            'resource_type': 'album'
        })
    else:
        raw_address = Publicator().get_address(public_hash)['rawaddress']
        address = Address(raw_address)
        resource = get_resource(address.uid, address)

        is_shared = resource.is_group or resource.is_shared
        uid_is_owner = resource.uid == uid
        uid_is_invited = False
        if is_shared and not uid_is_owner:
            try:
                next((u for u in resource.group.iteruids() if u == uid))
            except StopIteration:
                uid_is_invited = False
            else:
                uid_is_invited = True

        media_type = getattr(resource, 'media_type', None)  # может ли вообще быть такое?
        log_data.update({
            'resource_type': resource.type,
            'resource_is_shared': is_shared,
            'resource_media_type': media_type,
            'uid_is_owner': uid_is_owner,
            'uid_is_invited': uid_is_invited,
            'lenta_media_type': LentaMediaType.convert_to_lenta_media_type(
                media_type
            ) if media_type else None,
        })
    logger.log_raw_event(event_type='public-visit', data=log_data)
