# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import traceback

from mpfs.common.static import tags
from mpfs.common.util.filetypes import (
    builtin_extensions, MediaType
)
from mpfs.core.filesystem.resources.group import GroupResource
from mpfs.core.filesystem.resources.share import SharedResource
from mpfs.core.services.lenta_loader_service import lenta_loader
import mpfs.engine.process


error_log = mpfs.engine.process.get_error_log()


class LentaMediaType(object):
    VIDEO = MediaType.VIDEO
    IMAGE = MediaType.IMAGE
    AUDIO = MediaType.AUDIO
    DOCUMENT = MediaType.DOCUMENT
    OTHER = b'other'

    # типы, которые мапятся 1 в 1
    CORRESPONDING = (
        VIDEO, IMAGE,
        AUDIO, DOCUMENT
    )
    OTHER_MEDIA_TYPES = builtin_extensions.viewkeys() - set(CORRESPONDING)

    @classmethod
    def convert_to_lenta_media_type(cls, media_type):
        """Получить Ленточный медиа тип соответствующий переданному медиа типу."""
        if media_type not in builtin_extensions:
            raise ValueError()

        if media_type in cls.CORRESPONDING:
            return media_type

        return cls.OTHER

    @classmethod
    def convert_from_lenta_media_type(cls, lenta_media_type):
        """Получить медиа тип(ы) из Ленточного медиа типа."""
        if lenta_media_type in LentaMediaType.CORRESPONDING:
            return [lenta_media_type]
        elif lenta_media_type == LentaMediaType.OTHER:
            return sorted(cls.OTHER_MEDIA_TYPES)
        else:
            raise ValueError()

    @classmethod
    def clean_up_the_mess(cls, media_types):
        """Разгрести кашу из ленточных медиа типов и обычных медиа типов и вернуть только обычные.

        Несуществующий медиа тип будет добавлен в результат, а ленточный преобразован в соответствующий/ие медиа типы.
        Убирает дубликаты из возвращаемого результата.
        """
        # по сути все что тут делается сейчас, это "other" (если он есть)
        # разворачивается в набор медиа типов и удаляются дубликаты

        mt_lst = []
        for mt in media_types:
            # сначала проверяем что это обычный медиа тип
            if mt in builtin_extensions:
                # да, обычный - добавляем его
                mt_lst.append(mt)
                continue

            # иначе предполагаем что это ленточный медиа тип и пытаемся развернуть его
            # в набор обычных медиа типов
            try:
                mt = LentaMediaType.convert_from_lenta_media_type(mt)
            except ValueError:
                # если передан несуществующий медиа тип - добавляем его
                mt_lst.append(mt)
            else:
                mt_lst.extend(mt)

        return list(set(mt_lst))


class EventLogMessagesTransformer(object):
    """Класс для преобразования сообщений из эвент лога во что-то полезное."""

    @staticmethod
    def to_lenta_block_id(messages):
        lenta_block_id = None
        try:
            if messages:
                for msg in messages:
                    lenta_block_id = lenta_loader.process_log_line_and_return_created_block_id(msg)
                    break
        except Exception:
            error_log.info(traceback.format_exc())

        return lenta_block_id

    @classmethod
    def to_lenta_block_id_for_op(cls, messages, operation, save=False):
        """Выбрать подходящее по списку паттернов сообщение (из эвент лога) и
        обменять его на идентификатор блока в Ленте"""

        lenta_block_id = cls.to_lenta_block_id(messages=messages)
        if lenta_block_id:
            try:
                operation.data[tags.LENTA_BLOCK_ID] = lenta_block_id
                if save:
                    operation.save()
            except Exception:
                error_log.info(traceback.format_exc())

    @staticmethod
    def to_lenta_block_id_for_public_op(messages, operation, save=False):
        try:
            lenta_block_id = None
            if messages:
                for msg in messages:
                    try:
                        lenta_block_id = lenta_loader.save_file_from_public(msg)
                    except Exception:
                        error_log.info(traceback.format_exc())
                    break

            if lenta_block_id:
                operation.data[tags.LENTA_BLOCK_ID] = lenta_block_id
                if save:
                    operation.save()
        except Exception:
            error_log.info(traceback.format_exc())


def is_resource_visible_for_user(resource, uid):
    """
    Проверка видимости ресурса для юзера

    Предназначен для починки бага с чтением чужого блока ленты https://st.yandex-team.ru/CHEMODAN-38996. По-хорошему
    вся эта логика должна содержаться в fs.resources_by_resource_ids (как это уже сделано) и factory.get_resource.
    """
    if isinstance(resource, GroupResource) or isinstance(resource, SharedResource):
        return uid in resource.group.all_uids()
    return resource.owner_uid == uid
