# -*- coding: utf-8 -*-
"""

MPFS
CORE

Декораторы по хранилищу

"""
from functools import wraps
from contextlib import contextmanager

from pymongo.read_preferences import ReadPreference

import mpfs.engine.process

from mpfs.config import settings
from mpfs.common import errors
from mpfs.core.user.base import User
from mpfs.core.user.attach import AttachUser
from mpfs.core.address import Address
from mpfs.core.services.passport_service import Passport

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
passport = Passport()

FEATURE_TOGGLES_ALLOW_ONLY_MAIL_FOR_ATTACH_USER = settings.feature_toggles['allow_only_mail_for_attach_user']


def _check_user_exists_and_not_blocked(req,
                                       work_in_location=None,
                                       blocked_user_fail_strategy=None,
                                       attachuser_fail_strategy=None,
                                       allow_uninitialized_user=False):
    if work_in_location is None:
        work_in_location = UserExistsAndNotBlockedParams.WorkInLocation.Default

    if attachuser_fail_strategy is None:
        attachuser_fail_strategy = UserExistsAndNotBlockedParams.AttachuserFailStrategy.Default

    if blocked_user_fail_strategy is None:
        blocked_user_fail_strategy = UserExistsAndNotBlockedParams.BlockedUserFailStrategy.Default

    try:
        user = req.user
        if user is None:
            raise errors.StorageInitUser()
        user.check_blocked()
        # Пытаемся райзить правильную ошибку, когда не заведенный в диске
        # пользователь, но имеющий почтовые аттачи, пытается загружать
        # данные в диск.
        if (FEATURE_TOGGLES_ALLOW_ONLY_MAIL_FOR_ATTACH_USER and
                isinstance(user, AttachUser) and
                req.get_type() != 'mail' and
                attachuser_fail_strategy is UserExistsAndNotBlockedParams.AttachuserFailStrategy.Raise):
            raise errors.StorageInitUser()
    except errors.UserBlocked:
        if ((work_in_location is UserExistsAndNotBlockedParams.WorkInLocation.All
                 or req.get_type() in UserExistsAndNotBlockedParams.WorkInLocation.Default)
                and blocked_user_fail_strategy is UserExistsAndNotBlockedParams.BlockedUserFailStrategy.Raise):
            raise
        if blocked_user_fail_strategy is UserExistsAndNotBlockedParams.BlockedUserFailStrategy.DoNotCallMethod:
            return False
    except (errors.StorageInitUser, errors.StorageEmptyDiskInfo) as e:
        if not allow_uninitialized_user and req.uid not in ('0', mpfs.engine.process.share_user()):
            raise e

    return True


class UserExistsAndNotBlockedParams:
    class WorkInLocation:
        All = 'all'
        Default = ['mail', 'desktop', 'json']

    class BlockedUserFailStrategy:
        Raise = object()
        DoNotCallMethod = object()
        Default = Raise

    class AttachuserFailStrategy:
        Raise = object()
        Skip = object()
        Default = Raise


def user_exists_and_not_blocked(work_in_location=None, attachuser_fail_strategy=None, blocked_user_fail_strategy=None):
    """
    Декоратор, проверяющий два момента:
    - пользователь существует
    - пользователь не заблокирован (для локаций /mail, /json и /desktop)
    - параметры: UserExistsAndNotBlockedParams
    """

    if callable(work_in_location):
        return user_exists_and_not_blocked()(work_in_location)

    def wrapper(fn):
        @wraps(fn)
        def wrapped(*args, **kwargs):
            req = args[0]
            user_is_ok = _check_user_exists_and_not_blocked(req,
                                                            work_in_location=work_in_location,
                                                            blocked_user_fail_strategy=blocked_user_fail_strategy,
                                                            attachuser_fail_strategy=attachuser_fail_strategy)
            if user_is_ok:
                return fn(*args, **kwargs)

        return wrapped

    return wrapper


def replace_thrown_exception(exception, new_exception_instance):
    """Заменяет брошенное исключение другим исключением.

    Можно использовать для изменения ошибки для MPFS-ручек: нужно навесить декоратор на нучку из `mpfs.core.base`.
    """

    def decorator(fn):
        @wraps(fn)
        def change_status_code(*args, **kwargs):
            try:
                return fn(*args, **kwargs)
            except exception:
                raise new_exception_instance

        return change_status_code

    return decorator


def disallow_pdd(fn):
    """
    Временный декоратор, запрещающий действия ПДД-шникам

    https://st.yandex-team.ru/CHEMODAN-19729
    """

    @wraps(fn)
    def check_pdd(*args, **kwargs):
        req = args[0]
        if not settings.feature_toggles['allow_pdd']:
            if req.uid and req.uid.isdigit():
                userinfo = passport.userinfo(req.uid)
                if userinfo.get('pdd'):
                    raise errors.PermissionDenied(req.uid)
        return fn(*args, **kwargs)

    return check_pdd


def user_is_writeable(fn):
    """
    Декоратор, проверяющий, что юзер доступен на запись для всех стораджей, кроме указанного

    :param name: Имя стораджа для которого не нужно выполнять проверку юзера.
    :param param_name: Имя параметра запроса содержащего path.
    """

    @wraps(fn)
    def decorator(*args, **kwargs):
        req = args[0]

        if hasattr(req, 'uid'):
            try:
                mpfs.engine.process.usrctl().assert_writable(req.uid)
            except errors.StorageInitUser:
                # Если юзер не инициализрован, то он writable.
                pass
        return fn(*args, **kwargs)

    return decorator


def slave_read_ok(fn):
    """
    Декоратор, разрешающий при выполнении читать данные со слейва

    DEPRECATED: Сейчас ничего не делает, остаётся как информация о том,
    какие ручки мы считали возможным запускать с чтением со слейва
    """
    # заглушка
    return fn


def enable_secondary_preferred_if(condition):
    def decorator(fn):
        if not condition:
            return fn

        @wraps(fn)
        def wrapper(*args, **kwargs):
            try:
                mpfs.engine.process.set_read_preference(
                    ReadPreference.SECONDARY_PREFERRED)
                return fn(*args, **kwargs)
            finally:
                # if `None` then user defined is used
                mpfs.engine.process.set_read_preference(None)

        return wrapper

    return decorator


@contextmanager
def secondary_preferred_mode():
    before = mpfs.engine.process.get_read_preference()
    try:
        mpfs.engine.process.set_read_preference(ReadPreference.SECONDARY_PREFERRED)
        yield
    finally:
        mpfs.engine.process.set_read_preference(before)


def allow_empty_disk_info(fn):
    """
    Декоратор, разрещающий пустые выборки из disk_info
    """

    @wraps(fn)
    def process_with_empty_disk_info(self, *args, **kwargs):
        try:
            mpfs.engine.process.dbctl().allow_empty_disk_info(True)
            result = fn(self, *args, **kwargs)
        except Exception:
            mpfs.engine.process.dbctl().allow_empty_disk_info(False)
            raise
        else:
            mpfs.engine.process.dbctl().allow_empty_disk_info(False)
            return result

    return process_with_empty_disk_info


def without_fsync_safe_w(fn):
    """
    Декоратор, отключающий fsync и safe в запросах к базе
    """

    @wraps(fn)
    def process_without_fsync_safe_w(*args, **kwargs):
        try:
            mpfs.engine.process.dbctl().allow_fsync_safe_w(False)
            result = fn(*args, **kwargs)
        except:
            mpfs.engine.process.dbctl().allow_fsync_safe_w(True)
            raise
        else:
            mpfs.engine.process.dbctl().allow_fsync_safe_w(True)
            return result

    return process_without_fsync_safe_w


def allow_user_init_in_progress(fn):
    """
    Декоратор, проставляющий функции атрибут, по которому фронтенд понимает, что
    вызов функции разрешен даже если пользователь находится в процессе инициализации
    """
    fn._allow_user_init_in_progress = True
    return fn
