# -*- coding: utf-8 -*-
from bson import ObjectId
from bson.errors import InvalidId


class ModelField(object):
    """Класс для отличения обычных аттрибутов моделей от аттрибутов которые мапаются на поля БД."""
    def __init__(self, required=False, source=None, *args, **kwargs):
        super(ModelField, self).__init__()
        self.required = required
        self._source = source
        self.name = None

    @property
    def source(self):
        return self._source or self.name

    def to_native(self, value):
        return value

    def from_native(self, value):
        return value


class EmbeddedModelField(ModelField):
    def __init__(self, model, *args, **kwargs):
        super(EmbeddedModelField, self).__init__(*args, **kwargs)
        self.model = model

    _model = None

    def _get_model(self):
        if hasattr(self._model, '__call__'):
            return self._model()
        return self._model

    def _set_model(self, value):
        assert hasattr(value, '__call__') or issubclass(value, Model)
        self._model = value

    model = property(_get_model, _set_model)

    def to_native(self, value):
        if value:
            return self.model.from_db(value)
        return None

    def from_native(self, value):
        if value:
            return value.as_db_object()
        return None


class EnumModelField(ModelField):
    def __init__(self, enum_cls, *args, **kwargs):
        super(EnumModelField, self).__init__(*args, **kwargs)
        self.enum_cls = enum_cls

    def to_native(self, value):
        return self.enum_cls(value)

    def from_native(self, value):
        return value.value


class ObjectIdField(ModelField):
    def to_native(self, value):
        if value is None:
            return value
        return str(value)

    def from_native(self, value):
        if isinstance(value, ObjectId) or value is None:
            return value
        try:
            return ObjectId(value)
        except InvalidId:
            return None


class ModelMetaClass(type):
    """
    Упаковывает атрибуты класса содержащие экземпляры ModelField в проперти
    и прячет экземпляры ModelField в атрибут модели fields.
    """
    def __new__(mcls, name, parents, attrs):
        # собираем все филды из родительских классов
        fields = {}
        # Закидываем родительские филды, если таковые имеются, чтоб поддержать наследование филдов в филдсетах.
        for parent in parents:
            if hasattr(parent, 'fields'):
                fields.update(parent.fields)
        # Обновляем собранные из родителей филды собственными филдами создаваемого класса.
        fields.update([(k, v) for k, v in attrs.iteritems() if isinstance(v, ModelField)])
        attrs['fields'] = fields
        # заменяем филды на проперти
        for field_name, field in fields.iteritems():
            field.name = field_name
            attrs[field_name] = mcls.wrap_field(field_name)

        # создаём маппинг для поиска филдов по source, чтоб быстрее искать
        attrs['_source_field_map'] = dict([(f.source, f) for f in fields.values()])

        # создаём класс
        return super(ModelMetaClass, mcls).__new__(mcls, name, parents, attrs)

    @staticmethod
    def wrap_field(field_name):
        def get_field(self):
            return self._instance_data.get(field_name, None)

        def set_field(self, value):
            self._instance_data[field_name] = value
            self.changed_fields.add(field_name)

        def del_field(self):
            del self._instance_data[field_name]
        return property(get_field, set_field, del_field)

    @property
    def controller(cls):
        """Контроллер модели."""
        if not hasattr(cls, '_controller'):
            controllers_module = '%s.controllers' % cls.__module__.rsplit('.', 1)[0]
            try:
                # при импорте модуля с контроллерами, контроллеры автоматически пропишутся в модели
                __import__(controllers_module)
            except ImportError:
                # Если не удалось импортнуть модуль контроллеров, то устанавливаем атрибут _controller,
                # чтобы не пытаться импортировать его каждый раз при обращении к проперте.
                setattr(cls, '_controller', None)
        return getattr(cls, '_controller', None)

    @controller.setter
    def controller(cls, value):
        setattr(cls, '_controller', value)

    @property
    def is_sharded(cls):
        if cls.controller is None:
            raise AttributeError('Model without controller doesn\'t know sharded it or not.')
        return cls.controller.collection.is_sharded

    _source_field_map = None
    """Маппинг имён полей в БД на филды модели."""

    @property
    def uid_field(cls):
        if cls.controller is None:
            raise AttributeError('Model without controller doesn\'t know which field is used as "uid".')
        if cls.is_sharded:
            field = cls._source_field_map.get(cls.controller.collection.uid_field, None)
            if field:
                return field.name
            else:
                raise AttributeError('Sharded model doesn\'t have field used to shard its collection.')
        else:
            return None


class Model(object):
    __metaclass__ = ModelMetaClass

    controller = None
    """Контроллер модели."""

    primary_key_field = '_id'
    """Поле используемое моделью в качестве первичного ключа."""

    uid_field = None
    """
    Поле используемое в качестве uid для запросов по шардированным коллекциям.
    Если коллекция модели не шардирована, то None.
    Устанавливается автоматически в соответствии с атрибутом коллекции в которой хранится модель.
    """

    fields = {}

    def __init__(self, **kwargs):
        """
        Инициализирует новый объект.

        :param kwargs: Значения атрибутов объекта.
        """
        self.changed_fields = set()
        self._instance_data = {}

        self.controller = type(self).controller
        self.uid_field = type(self).uid_field
        self.is_sharded = type(self).is_sharded

        for attr, value in kwargs.iteritems():
            if attr in self.fields:
                setattr(self, attr, value)

    @classmethod
    def from_db(cls, db_object):
        """
        Создаёт новый объект из объекта базы данных (пока у нас монга, таким объектом всегда является dict).
        """
        data = cls.map_to_obj_fields(db_object)
        ret = cls(**data)
        ret.changed_fields = set()
        return ret

    def _get_pk(self):
        return self._instance_data.get(self.primary_key_field, None)

    def _set_pk(self, value):
        self._instance_data[self.primary_key_field] = value

    pk = property(_get_pk, _set_pk)
    """Значение первичного ключа (Primary Key) записи в БД не зависимо от имени этого поля в БД и в модели."""

    def is_field_set(self, name):
        """Возвращает задано ли значение для поля объекта."""
        return name in self._instance_data

    @classmethod
    def map_to_db_fields(cls, data):
        """
        Преобразует dict содержащий значения атрибутов объекта в dict содержащий значения полей в БД.
        """
        ret = {}
        for name, value in data.iteritems():
            if name not in cls.fields:
                raise AttributeError('Model "%s" doesn\'t have field "%s".' % (cls.__name__, name))
            field = cls.fields[name]
            ret[field.source] = field.from_native(value)
        return ret

    @classmethod
    def map_to_obj_fields(cls, db_data):
        """
        Преобразует dict содержащий значения полей в БД в dict содержащий значения атрибутов объекта.
        """
        ret = {}
        for name, field in cls.fields.iteritems():
            if field.source in db_data:
                ret[name] = field.to_native(db_data.get(field.source))
        return ret

    def as_db_object(self):
        """
        Возвращает представление объекта в виде пригодном для сохранения в БД.
        """
        db_obj = self.map_to_db_fields(self._instance_data)
        if not self.pk:
            db_obj.pop(self.fields[self.primary_key_field].source, None)
        return db_obj

    def as_dict(self):
        """
        Возвращает представление объекта в виде дикта.
        """
        return self._instance_data

    def as_json_dict(self):
        """
        Возвращает представление объекта пригодное для отдачи наружу.
        """
        obj = self.as_dict()
        ret = {}
        for k, v in obj.iteritems():
            if isinstance(v, Model):
                v = v.as_json_dict()
            ret[k] = v
        return ret

    def as_tskv(self):
        """
        Возвращает представление объекта пригодное для записи в tskv лог.
        """
        ret = []
        for name, field in self.fields.iteritems():
            field_value = field.to_native(self._instance_data.get(name, ''))
            if field_value is None:
                field_value = ''
            ret.append('%s=%s' % (name, field_value))
        return '\t'.join(sorted(ret))

    def validate(self):
        """
        Проверяет корректность заполенния полей модели. В частности, чтобы все обязательные поля были заданы.

        :raises ValueError: Если не обязательное поле не задано.
        """
        for name, field in self.fields.iteritems():
            if field.required and not self.is_field_set(name):
                raise ValueError('Field "%s" is required in %s objects.' % (name, type(self).__name__))

    def _get_object_query(self):
        """
        Возвращает запрос позволяющий получить данный объект.

        Понимает шардирована или нет коллекция содержащая объект и, в зависимости от этого, умеет добавлять в запрос uid.
        """
        result = {self.primary_key_field: self.pk}
        if self.is_sharded:
            result[self.uid_field] = getattr(self, self.uid_field, None)
        return result

    def delete(self):
        """Удаялет объект из базы данных."""
        self.controller.filter(**self._get_object_query()).delete()

    def save(self, force_insert=False, update_fields=None, save_changed=True, upsert=False):
        """
        Сохраняет объект в БД.

        :param force_insert: Создать в БД новую запись, даже если у объекта установлен `pk`.
        :param update_fields: Список имён атрибутов которые следует сохранить.
        :param save_changed: Сохранить только изменённые атрибуты.
        :param upsert: Принудительно сохраняет документ update-ом с upsert-ом.

        :raise ZeroUpdateControllerMPFSError: при попытке изменить(update) запись ни один документ не обновлен.
        """
        self.validate()
        update_fields = self.get_update_fields(update_fields=update_fields, save_changed=save_changed)
        self.changed_fields = set()
        self._save(force_insert=force_insert, update_fields=update_fields, upsert=upsert)

    def is_new(self, force_insert=False, *args, **kwargs):
        """
        Вернёт True, если вызов метода `save` создаст новый объект в БД,
        или False, если обновит существующий.

        Принимает все те же аргументы, что и метод `save`.
        """
        return force_insert or not self.pk

    def get_update_fields(self, update_fields=None, save_changed=True, *args, **kwargs):
        """
        Возвращает список атрибутов, которые будут обновлены у объекта при сохранени в БД.

        Принимает все те же аргументы, что и метод `save`.
        """
        result = None
        if save_changed:
            result = []
            if update_fields:
                result += update_fields
            result += self.changed_fields
            result = list(set(result))  # фильтруем дубли
        return result

    def _save(self, force_insert=False, update_fields=None, upsert=False):
        """
        Внутренний метод сохранения. Выполняет самые общие действия. Поэтому переопределять его не стоит.
        """
        if self.controller is None:
            raise AttributeError('Unable to save model without controller.')
        # если у объекта нет первичного ключа или явно указано, что нужно добавить новый объект,
        # то добавляем объект в БД
        if self.is_new(force_insert=force_insert) and not upsert:
            self.pk = self.controller._insert(self.as_db_object())
        else:
            data = self.as_dict()
            if update_fields:
                data = dict([(k, data[k]) for k in update_fields if k in data])
            self.controller.filter(**self._get_object_query()).update(upsert=upsert, **data)
