# -*- coding: utf-8 -*-
import datetime

from mpfs.engine.process import setup_admin_script, dbctl
setup_admin_script()
from mpfs.core.billing.product.catalog import ALL_PRODUCTS
from mpfs.core.mrstat.stat_utils import StatPublisher, quit_if_mrstat_disabled


REPORT_CONFIG_YAML = """
---
dimensions:
- fielddate: date
- pid: string
measures:
- num: number
- total_size: number
- is_singleton: string
- size: number
titles:
  num: Кол-во услуг
  size: Размер одной услуги (для статич. услуг)
  total_size: Суммарное занимаемое место (для статич. услуг)
  is_singleton: Услуга может быть выдана пользователю один раз
view_types:
  size:
    type: BytesSize
  total_size:
    type: BytesSize
sort:
  fields:
    - num
  reverse: 1
"""


def get_billing_services_stat():
    today_date = str(datetime.date.today())
    common_conn = dbctl().connection('common')
    bs_coll = common_conn.billing_services.billing_services
    pipeline = [
        {"$group": {"_id": "$pid", "count": {"$sum": 1}}},
    ]
    cursor = bs_coll.aggregate(pipeline)

    result = []
    total_count = 0
    total_size = 0
    for item in cursor['result']:
        pid, count = item['_id'], item['count']
        try:
            size = ALL_PRODUCTS[pid]['attributes']['amount']
        except (AttributeError, KeyError):
            size = 0
        try:
            is_singleton = ALL_PRODUCTS[pid]['singleton']
        except (AttributeError, KeyError):
            is_singleton = None
        try:
            ru_name = ALL_PRODUCTS[pid]['name']['ru']
        except (AttributeError, KeyError):
            ru_name = '-'

        service_name = '%s (%s)' % (pid, ru_name)
        result.append({
            'fielddate': today_date,
            'pid': service_name,
            'num': count,
            'size': size,
            'total_size': size * count,
            'is_singleton': str(is_singleton),
        })
        total_size += size * count
        total_count += count

    result.append({
        'fielddate': today_date,
        'pid': '_total_',
        'num': total_count,
        'size': 0,
        'total_size': total_size,
        'is_singleton': str(is_singleton),
    })
    return result


if __name__ == '__main__':
    quit_if_mrstat_disabled()
    report_data = get_billing_services_stat()
    StatPublisher.create_and_upload(
        'Disk/DiskInternal/DiskServices',
        'Услуги диска',
        REPORT_CONFIG_YAML,
        report_data
    )
