#!/usr/bin/env python
# -*- coding: utf-8 -*-

import yt.wrapper as yt
from nile.api.v1 import clusters, aggregators
from qb2.api.v1 import extractors as se
from qb2.api.v1 import filters as sf

from mpfs.config import settings
from mpfs.core.mrstat.stat_utils import StatPublisher, set_yt_proxy, quit_if_mrstat_disabled
from mpfs.core.mrstat.report import parse_args


REPORT_CONFIG_YAML = """
---
dimensions:
- fielddate: date
measures:
- uniq_md5_num: number
- file_uploads: number
titles:
  uniq_md5_num: Уникальных восстановленных файлов
  file_uploads: Загрузок
"""


def calc_recovery_stat(dates, result_path):
    cluster = clusters.Hahn(settings.mrstat['yt_token'])
    job = cluster.job()

    input_tables = [job.table('//statbox/ydisk-mpfs-access-log/%s' % d) for d in dates]
    job.concat(*input_tables) \
        .qb2(
            log='ydisk-mpfs-access-log',
            fields=[
                'date',
                se.parameter('md5'),
            ],
            filters=[
                sf.equals('status', '200'),
                sf.equals('appname', 'platform'),
                sf.startswith('page', '/v1/disk/restore/upload'),
            ]
        ) \
        .groupby('date', 'md5') \
        .aggregate(hits=aggregators.count()) \
        .groupby('date') \
        .aggregate(uniq_md5_num=aggregators.count(), file_uploads=aggregators.sum('hits')) \
        .project(se.all(exclude=('date',)), fielddate='date') \
        .put(result_path)
    job.run()


if __name__ == '__main__':
    quit_if_mrstat_disabled()
    dates, _, _ = parse_args()

    result_path = '//home/mpfs-stat/results/mpfs_file_recovery/%s' % dates[0]
    calc_recovery_stat(dates, result_path)
    set_yt_proxy()
    if yt.exists(result_path):
        report_data = list(yt.read_table(result_path, format="json", raw=False))
        StatPublisher.create_and_upload(
            'Disk/DiskInternal/FileRecoveryFileUploading',
            'Восстановление. Загрука файлов от ПО',
            REPORT_CONFIG_YAML,
            report_data
        )
