# -*- coding: utf-8 -*-
# https://wiki.postgresql.org/wiki/Disk_Usage

import datetime
from collections import defaultdict
from mpfs.engine.process import setup_admin_script
setup_admin_script()

from mpfs.metastorage.postgres.query_executer import PGQueryExecuter
from mpfs.dao.base import Session
from mpfs.metastorage.postgres.services import Sharpei
from mpfs.core.mrstat.stat_utils import StatPublisher, quit_if_mrstat_disabled


sharpei = Sharpei()
SHARPEI_ID_NAME_MAP = {k: v['name'] for k, v in sharpei._get_stat_data().iteritems()}

SQL_RELATION_SIZES = """
SELECT
    nspname || '.' || relname AS relation,
    pg_relation_size(C.oid) AS size
FROM pg_class C
LEFT JOIN pg_namespace N ON (N.oid = C.relnamespace)
WHERE nspname = 'disk'
ORDER BY pg_relation_size(C.oid) DESC;
"""
SQL_TABLE_STAT = """
SELECT
    reltuples AS row_estimate,
    pg_total_relation_size(c.oid) AS total_bytes,
    pg_indexes_size(c.oid) AS index_bytes,
    pg_total_relation_size(reltoastrelid) AS toast_bytes
FROM pg_class as c
WHERE oid = 'disk.%s'::regclass;
"""
SQL_GET_ALL_TABLES = "SELECT table_name FROM information_schema.tables WHERE table_schema = 'disk';"

RELATION_SIZES_REPORT_CONFIG_YAML = """
---
dimensions:
- fielddate: date
- shard_name: string
- relation_name: string
measures:
- size: number
view_types:
  size: BytesSize
  relation_name:
    type: Selector
    default: _in_table_
  shard_name:
    type: Selector
    default: _total_
sort:
  fields:
    - size
  reverse: 1
"""


def relation_sizes():
    query_executer = PGQueryExecuter()
    shard_ids = query_executer.get_all_shard_ids()

    stat_data = defaultdict(lambda: defaultdict(lambda: defaultdict(int)))
    for shard_id in shard_ids:
        shard_name = SHARPEI_ID_NAME_MAP[shard_id]
        session = Session.create_from_shard_id(shard_id)
        total = 0
        for result_proxy in session.execute(SQL_RELATION_SIZES):
            relation_name, size = result_proxy
            for sn in (shard_name, '_total_'):
                for rn in (relation_name, '_total_'):
                    stat_data[sn][rn]['size'] += size

    today = str(datetime.date.today())
    rows = []
    for shard_name, data in stat_data.iteritems():
        for relation_name, values in data.iteritems():
            row = {
                'fielddate': today,
                'shard_name': shard_name,
                'relation_name': relation_name,
                'size': values['size'],
            }
            rows.append(row)
            print row

    StatPublisher.create_and_upload(
        'Disk/DiskInternal/PostgresRelationSizes',
        'Postgres. Relation sizes',
        RELATION_SIZES_REPORT_CONFIG_YAML,
        rows,
        scale='d'
    )
        

SHARDS_AND_TABLES_STAT_REPORT_CONFIG_YAML = """
---
dimensions:
- fielddate: date
- shard_name: string
- table_name: string
measures:
- row_estimate: number
- table_bytes: number
- index_bytes: number
- toast_bytes: number
- total_bytes: number
view_types:
  table_bytes: BytesSize
  index_bytes: BytesSize
  toast_bytes: BytesSize
  total_bytes: BytesSize
  table_name:
    type: Selector
    default: _total_
  shard_name:
    type: Selector
    default: _in_table_
sort:
  fields:
    - shard_name
hidden:
  table_bytes: 1
  index_bytes: 1
  toast_bytes: 1
"""


def shards_and_tables_stat():
    query_executer = PGQueryExecuter()
    shard_ids = query_executer.get_all_shard_ids()
    stat_data = defaultdict(lambda: defaultdict(lambda: defaultdict(int)))
    for shard_id in shard_ids:
        shard_name = SHARPEI_ID_NAME_MAP[shard_id]
        session = Session.create_from_shard_id(shard_id)
        table_names = session.execute(SQL_GET_ALL_TABLES)
        for table_name in table_names:
            table_name = table_name[0]

            result_proxy = session.execute(SQL_TABLE_STAT % table_name)
            row_estimate, total_bytes, index_bytes, toast_bytes = result_proxy.fetchone()
            toast_bytes = toast_bytes or 0
            table_bytes = total_bytes - index_bytes - toast_bytes

            for sn in (shard_name, '_total_'):
                for tn in (table_name, '_total_'):
                    stat_data[sn][tn]['table_bytes'] += table_bytes
                    stat_data[sn][tn]['row_estimate'] += row_estimate
                    stat_data[sn][tn]['index_bytes'] += index_bytes
                    stat_data[sn][tn]['total_bytes'] += total_bytes
                    stat_data[sn][tn]['toast_bytes'] += toast_bytes

    today = str(datetime.date.today())
    rows = []
    for shard_name, data in stat_data.iteritems():
        for table_name, values in data.iteritems():
            row = {
                'fielddate': today,
                'shard_name': shard_name,
                'table_name': table_name,
            }
            row.update(values)
            rows.append(row)
            print row

    StatPublisher.create_and_upload(
        'Disk/DiskInternal/PostgresShardsAndTablesStat',
        'Postgres. Shards and tables stat',
        SHARDS_AND_TABLES_STAT_REPORT_CONFIG_YAML,
        rows,
        scale='d'
    )


if __name__ == '__main__':
    quit_if_mrstat_disabled()
    relation_sizes()
    shards_and_tables_stat()
