# -*- coding: utf-8 -*-
import datetime
import re
from argparse import ArgumentParser

import yt.wrapper as yt
from nile.api.v1 import aggregators

from mpfs.common.static.tags import ID
from mpfs.common.util import from_json
from mpfs.config import settings
from nile.api.v1 import clusters, Record

from mpfs.core.mrstat.stat_utils import set_yt_proxy, quit_if_mrstat_disabled, StatPublisher
from mpfs.core.rostelecom_unlim.catalog import rostelecom_unlim_products, rostelecom_unlim_test_products

set_yt_proxy()

SERIVCE_DELETED_REGEX = re.compile(r'^deleted service ([^ ]*) (\d*):(\w*).*')
SERVICE_CREATED_REGEX = re.compile(r'^created service ([^ ]*) (\d*):(\w*).*')
SERVICE_UPGRADED_REGEX = re.compile(
    r'^Rostelecom service upgraded old_sid=(\w*) old_pid=([^ ]*) new_sid=(\w*) new_pid=([^ ]*) uid=(\d*).*')
SERVICE_ACTION_REGEX = re.compile(r'^rostelecom_unlim: status_before=([^ ]*) status_after=([^ ]*) action=([^ ]*)'
                                  r' uid=([^ ]*) rostelecom_uid=([^ ]*) pid=([^ $\n]*).*')
ROSTELECOM_PIDS = [p[ID] for p in rostelecom_unlim_products + rostelecom_unlim_test_products]

ROSTELECOM_UNLIM_PRODUCTS_IDS = {p[ID] for p in rostelecom_unlim_products + rostelecom_unlim_test_products}

REPORT_CONFIG_YAML = """
---
dimensions:
- fielddate: date
- pid: string
measures:
- num: number
titles:
  num: Кол-во активных услуг
"""


class RostelecomUnlimStatsDailyReportBuilder(object):
    def __init__(self, date_str=None):
        if date_str is None:
            self.today = datetime.date.today()
        else:
            self.today = datetime.datetime.strptime(date_str, '%Y-%m-%d').date()
        self.yesterday = (self.today - datetime.timedelta(days=1))
        self.yesterday_stats_table = '//home/mpfs-stat/storage/rostelecom_active_services/%s' % self.yesterday
        self.today_stats_table = '//home/mpfs-stat/storage/rostelecom_active_services/%s' % self.today
        self.today_stats_table_aggregated = '//home/mpfs-stat/storage/rostelecom_active_services_aggregated/%s' % self.today
        self.default_table_path = '//statbox/ydisk-mpfs-default-log/%s' % self.yesterday

    @staticmethod
    def filter_default_log_mapper(records):
        for record in records:
            if not record.get('message'):
                continue
            if 'rostelecom_unlim:' in record['message']:
                match = SERVICE_ACTION_REGEX.match(record['message'])
                if match is None:
                    continue
                _, status_after, action, uid, rostelecom_uid, pid = match.groups()
                data = {
                    'pid': pid,
                    'uid': uid,
                    'rostelecom_uid': rostelecom_uid,
                    'action': action,
                    'status_after': status_after,
                    'timestamp': record.get('iso_eventtime')
                }
                yield Record(**data)
            elif 'Rostelecom service upgraded' in record['message']:
                match = SERVICE_UPGRADED_REGEX.match(record['message'])
                if match is None:
                    continue
                _, old_pid, _, new_pid, uid = match.groups()
                data = {
                    'pid': old_pid,
                    'uid': uid,
                    'action': 'deleted_by_upgrade',
                    'status_after': 'inactive',
                    'timestamp': record.get('iso_eventtime')
                }
                yield Record(**data)
                data = {
                    'pid': new_pid,
                    'uid': uid,
                    'action': 'created_by_upgrade',
                    'status_after': 'active',
                    'timestamp': record.get('iso_eventtime')
                }
                yield Record(**data)

    @staticmethod
    def stat_delta_reducer(groups):
        for key, group in groups:
            stat_data_delta = {
                'pid': key.get('pid'),
                'uid': key.get('uid'),
                'rostelecom_uid': None,
                'active': None,
                'activate_deactivate': False,
                'timestamp': None,
                'is_delta': True,
                'action': None,
                'action_timestamp': None,
                'last_activation_timestamp': None,
                'last_deactivation_timestamp': None,
                'last_status': None
            }
            for record in group:
                # случаи created_by_upgrade и deleted_by_upgrade не считаются действиями юзера, поэтому для них не
                # запоминаем времена активаций, деактиваций и действие
                stat_data_delta['timestamp'] = record.get('timestamp')
                stat_data_delta['last_status'] = record.get('status_after')
                stat_data_delta['rostelecom_uid'] = record.get('rostelecom_uid') or stat_data_delta.get('rostelecom_uid')
                action = record.get('action')
                if action in {'activate', 'unfreeze'}:
                    stat_data_delta['active'] = True
                    stat_data_delta['activate_deactivate'] = True
                    stat_data_delta['action'] = action
                    stat_data_delta['last_activation_timestamp'] = record.get('timestamp')
                    stat_data_delta['action_timestamp'] = record.get('timestamp')
                elif action == 'created_by_upgrade':
                    stat_data_delta['active'] = True
                    stat_data_delta['activate_deactivate'] = True
                elif action in {'deactivate', 'freeze'}:
                    stat_data_delta['active'] = False
                    stat_data_delta['action'] = action
                    stat_data_delta['last_deactivation_timestamp'] = record.get('timestamp')
                    stat_data_delta['action_timestamp'] = record.get('timestamp')
                elif action == 'deleted_by_upgrade':
                    stat_data_delta['active'] = False
                    stat_data_delta['activate_deactivate'] = False
                else:
                    raise NotImplementedError('Unexpected action')

            if stat_data_delta['active']:
                stat_data_delta['activate_deactivate'] = False
            yield Record(**stat_data_delta)

    @staticmethod
    def stat_merge_reducer(groups):
        for key, groups in groups:
            pid = key.get('pid')
            uid = key.get('uid')
            if pid is None or uid is None:
                continue
            stat_data = {
                'pid': pid,
                'uid': uid,
                'activate_deactivate': False,
                'is_delta': False,
            }
            for record in groups:
                stat_data['active'] = record.get('active')
                stat_data['timestamp'] = record.get('timestamp')
                stat_data['last_status'] = record.get('last_status')
                stat_data['action'] = record.get('action') or stat_data.get('action')
                stat_data['last_activation_timestamp'] = record.get('last_activation_timestamp') or stat_data.get('last_activation_timestamp')
                stat_data['last_deactivation_timestamp'] = record.get('last_deactivation_timestamp') or stat_data.get('last_deactivation_timestamp')
                stat_data['action_timestamp'] = record.get('action_timestamp') or stat_data.get('action_timestamp')
                stat_data['rostelecom_uid'] = record.get('rostelecom_uid') or stat_data.get('rostelecom_uid')
                if record.get('is_delta'):
                    stat_data['activate_deactivate'] = record.get('activate_deactivate')
            yield Record(**stat_data)

    @staticmethod
    def filter_active_services_mapper(records):
        for record in records:
            if not record.get('active') and not record.get('activate_deactivate'):
                continue
            yield Record(**{'pid': record.get('pid')})

    def build_stats_for_previous_day(self):
        cluster = clusters.Hahn(settings.mrstat['yt_token'])
        job = cluster.job()
        job.table(self.default_table_path) \
            .map(self.filter_default_log_mapper) \
            .groupby('pid', 'uid') \
            .sort('timestamp') \
            .reduce(self.stat_delta_reducer) \
            .concat(job.table(self.yesterday_stats_table)) \
            .groupby('pid', 'uid') \
            .sort('timestamp') \
            .reduce(self.stat_merge_reducer) \
            .put(self.today_stats_table)
        job.run()

        job = cluster.job()
        job.table(self.today_stats_table) \
            .map(self.filter_active_services_mapper) \
            .groupby('pid') \
            .aggregate(active_services=aggregators.count()) \
            .put(self.today_stats_table_aggregated)
        job.run()

    def get_today_staticstics(self):
        if not yt.exists(self.today_stats_table_aggregated):
            print "Table %s not exists" % self.today_stats_table_aggregated
            return
        results = []
        for row in yt.read_table(self.today_stats_table_aggregated, format="json", raw=True):
            data = from_json(row)
            results.append({
                'pid': data.get('pid'),
                'num': data.get('active_services'),
                'fielddate': str(self.today)
            })
        return results


if __name__ == '__main__':
    parser = ArgumentParser(description=__doc__)
    parser.add_argument('-d', dest='date', default=None, help='Date for which compute statistics. yyyy-mm-dd')
    args = parser.parse_args()

    quit_if_mrstat_disabled()
    rostelecom_unlim_stats_builder = RostelecomUnlimStatsDailyReportBuilder(args.date)
    rostelecom_unlim_stats_builder.build_stats_for_previous_day()
    report_data = rostelecom_unlim_stats_builder.get_today_staticstics()
    if report_data is None:
        exit(1)

    if report_data:
        StatPublisher.create_and_upload(
            'Disk/DiskInternal/RostelecomUnlimServices',
            'Услуги безлимитного Ростелекома в Диске',
            REPORT_CONFIG_YAML,
            report_data
        )
