# -*- coding: utf-8 -*-
import datetime
from argparse import ArgumentParser

import yt.wrapper as yt
from dateutil.relativedelta import relativedelta
from nile.api.v1 import aggregators

from mpfs.common.util import from_json
from mpfs.config import settings
from nile.api.v1 import clusters, Record

from mpfs.core.mrstat.stat_utils import set_yt_proxy, quit_if_mrstat_disabled, StatPublisher

set_yt_proxy()

REPORT_CONFIG_YAML = """
---
dimensions:
- fielddate: date
- pid: string
measures:
- num: number
titles:
  num: Кол-во активных услуг
"""


class RostelecomUnlimStatsMonthlyVASReportBuilder(object):
    """
    Составить месячный отчет для VAS услуг

    Если хотим посчитать за месяц, то first_date = YYYY.MM.02, а last_day = YYYY.MM+1.01, потому что в
    rostelecom_active_services в таблице лежит результат за логи из предыдущего дня
    """

    def __init__(self, first_date_str=None, last_date_str=None):
        if last_date_str is None:
            self.last_day = datetime.date.today()
        else:
            self.last_day = datetime.datetime.strptime(last_date_str, '%Y-%m-%d').date() + relativedelta(days=1)

        if first_date_str is None:
            self.first_day = self.last_day - relativedelta(months=1) + relativedelta(days=1)
        else:
            self.first_day = datetime.datetime.strptime(first_date_str, '%Y-%m-%d').date() + relativedelta(days=1)

        if self.first_day >= self.last_day:
            raise ValueError('First day of the period is greater than the last')

        self.table_range_paths = '//home/mpfs-stat/storage/rostelecom_active_services/{%s..%s}' % (self.first_day, self.last_day)
        self.tmp_table_path = '//home/mpfs-stat/tmp/rostelecom_monthly_vas_report/%s' % self.last_day

    @staticmethod
    def get_active_vas_services_mapper(records):
        for record in records:
            if '_vas_' not in record.get('pid'):
                continue
            if not (record.get('active') or record.get('activate_deactivate')):
                continue
            yield Record(**{'pid': record.get('pid'), 'uid': record.get('uid')})

    @staticmethod
    def unique_pid_uid_records_reducer(groups):
        for key, group in groups:
            yield Record(**{'pid': key.get('pid'), 'uid': key.get('uid'), })

    def build_stats_for_period(self):
        cluster = clusters.Hahn(settings.mrstat['yt_token'])
        job = cluster.job()

        job.table(self.table_range_paths) \
            .map(self.get_active_vas_services_mapper) \
            .groupby('pid', 'uid') \
            .reduce(self.unique_pid_uid_records_reducer) \
            .groupby('pid') \
            .aggregate(active_services=aggregators.count()) \
            .put(self.tmp_table_path)
        job.run()

    def get_period_statistiscs(self):
        if not yt.exists(self.tmp_table_path):
            print("Table %s not exists" % self.tmp_table_path)
            return
        results = []
        for row in yt.read_table(self.tmp_table_path, format="json", raw=True):
            data = from_json(row)
            results.append({
                'pid': data.get('pid'),
                'num': data.get('active_services'),
                'fielddate': str(self.last_day)
            })
        return results


if __name__ == '__main__':
    parser = ArgumentParser(description=__doc__)
    parser.add_argument('-fd', dest='first_date', default=None, help='First day of the period. yyyy-mm-dd')
    parser.add_argument('-ld', dest='last_date', default=None, help='Last day of the period. yyyy-mm-dd')
    args = parser.parse_args()
    quit_if_mrstat_disabled()
    rostelecom_unlim_stats_builder = RostelecomUnlimStatsMonthlyVASReportBuilder(args.first_date, args.last_date)
    rostelecom_unlim_stats_builder.build_stats_for_period()
    report_data = rostelecom_unlim_stats_builder.get_period_statistiscs()
    if report_data is None:
        exit(1)

    if report_data:
        StatPublisher.create_and_upload(
            'Disk/DiskInternal/RostelecomUnlimMonthlyVASServices',
            'VAS услуги безлимитного Ростелекома в Диске по месяцам',
            REPORT_CONFIG_YAML,
            report_data
        )
