# encoding: utf-8

import re
import json
import requests

import yt.wrapper as yt
from mpfs.config import settings


def quit_if_mrstat_disabled():
    if not settings.mrstat['enabled']:
        print 'MRstat calculation is disabled. Check "settings->mrstat->enabled"'
        quit()


def set_yt_proxy():
    yt.config['proxy']['url'] = settings.mrstat['yt_proxy']
    yt.config['token'] = settings.mrstat['yt_token']


class StatPublisher(object):
    host = settings.mrstat['publish_host']
    auth_headers = settings.mrstat['publish_user'].copy()

    @classmethod
    def create_and_upload(cls, report_path, report_title, yaml_config, report_data, scale='d'):
        """Опубликовать отчет и данные"""
        cls.create_report(report_path, report_title, yaml_config)
        cls.upload_report_data(report_path, report_data, scale=scale)

    @classmethod
    def create_report(cls, report_path, report_title, yaml_config):
        """Создание/изменение отчета

        https://wiki.yandex-team.ru/statbox/statface/externalreports/#sozdanie/izmenenieotcheta
        """
        resp = requests.post(
            '%s/_api/report/config' % cls.host,
            headers=cls.auth_headers,
            data={
                'cube_config': yaml_config,
                'title': report_title,
                'name': report_path,
            },
        )
        resp.raise_for_status()

    @classmethod
    def upload_report_data(cls, report_path, report_data, scale='d'):
        """Загрузка данных

        https://wiki.yandex-team.ru/statbox/statface/externalreports/#zagruzkadannyx
        """
        resp = requests.post(
            '%s/_api/report/data' % cls.host,
            headers=cls.auth_headers,
            data={
                'name': report_path,
                'scale': scale,
                'data': json.dumps({'values': report_data}),
                'replace_mask': 'fielddate',
            },
        )
        resp.raise_for_status()

    @classmethod
    def delete_report(cls, report_path):
        """Удаление отчета

        https://wiki.yandex-team.ru/statbox/statface/externalreports/#udalenieotchetapolnostjuvseskejjlyikonfig
        """
        resp = requests.post(
            '%s/_api/report/delete_report' % cls.host,
            headers=cls.auth_headers,
            data={
                'name': report_path
            },
        )
        resp.raise_for_status()


def is_uid(chunk):
    """
    Определям uid или нет для "REST API"™
    """
    if chunk.isdigit():
        return True
    for prefix in ("uid-", "yaid-", "device-"):
        if chunk.startswith(prefix):
            return True
    return False


def get_api_type(path):
    if not path or not path.startswith('/v'):
        return None

    res = re.search(r'^/(v\d+)/(?:[^/]+/)?(disk|personality|schema|data|case|batch)', path, flags=re.I)
    if res:
        return ("/%s/%s" % res.groups()).lower()
    else:
        return None


def status_expander(status):
    return ['_total_', "%sxx" % (status[0],), status]


if __name__ == '__main__':
    assert get_api_type('/v1/1417423320/peRsonality') == '/v1/personality'

    assert status_expander('404') == ['_total_', '4xx', '404']
