# -*- coding: utf-8 -*- 
import time
import datetime
import uuid
import json
import yt.wrapper as yt

from mpfs.config import settings
from mpfs.engine.process import setup_admin_script, get_default_log
setup_admin_script()
from mpfs.core.services.juggler import JEvent, JStatus
from mpfs.core.mrstat.stat_utils import StatPublisher, set_yt_proxy, quit_if_mrstat_disabled
from mpfs.core.operations.dao.operation import OperationDAO

default_log = get_default_log()

YT_PATH = '//home/mpfs-stat/tmp/waiting_operations'
MONITORING_DELTA = datetime.timedelta(hours=1, minutes=15)
WAITING_OPERATIONS_CRIT_LIMIT = 20
# Размер окна, с которым ходим в базу
FROM_DELTA = datetime.timedelta(minutes=20)
TO_DELTA = datetime.timedelta(minutes=15)

PENDED_TYPE_SUBTYPE = {k: set(settings.operations['types'][k]['pended'])
                       for k, v in settings.operations['types'].iteritems()
                       if v['pended']}


REPORT_CONFIG_YAML = """
---
dimensions:
- fielddate: date
measures:
- waiting_operations: number
"""


class LostOperationsJEvent(JEvent):
    host = 'disk_mworker_devops'
    service = 'not_started_operations_host'


def send_to_juggler(total_waiting_op):
    if total_waiting_op > WAITING_OPERATIONS_CRIT_LIMIT:
        status = JStatus.CRIT
    else:
        status = JStatus.OK
    event = LostOperationsJEvent(status, description="Find '%s' waiting operations" % total_waiting_op)
    default_log.info('SEND EVENT: %r', event)
    event.send()


def get_waiting_pended_operations(from_dt, to_dt):
    dao_items_gen = OperationDAO().fetch_by_date_period_and_states(from_dt, to_dt, [0])
    for dao_item in dao_items_gen:
        if (dao_item.type not in PENDED_TYPE_SUBTYPE or
                dao_item.subtype not in PENDED_TYPE_SUBTYPE[dao_item.type]):
            continue

        yield {
            'uid': dao_item.uid,
            'oid': dao_item.id,
            'type': dao_item.type,
            'subtype': dao_item.subtype,
            'dtime': str(dao_item.dtime),
        }


def main():
    now_dt = datetime.datetime.utcnow()
    from_dt = now_dt - FROM_DELTA
    to_dt = now_dt - TO_DELTA
    results = list(get_waiting_pended_operations(from_dt, to_dt))

    # сохраняем результаты мониторинга на YT
    yt_write_data = {
        'check_id': uuid.uuid4().hex,
        'check_dt': str(now_dt),
        'monitoring_dt_window': [str(from_dt), str(to_dt)],
        'waiting_oids': results,
    }
    set_yt_proxy()
    yt.write_table("<append=true>%s/%s" % (YT_PATH, now_dt.date()), json.dumps(yt_write_data), raw=True, format="json")

    # получаем список зависших операций за 1 час с YT и шлём событие в juggler
    table_data = yt.read_table("%s/%s" % (YT_PATH, now_dt.date()), format='json')
    last_hour_oids = set()
    for data in table_data:
        if (not data['waiting_oids'] or data['check_dt'] < "%s" % (now_dt - MONITORING_DELTA)):
            continue
        for waiting_oid in data['waiting_oids']:
            last_hour_oids.add((waiting_oid['uid'], waiting_oid['oid']))
    send_to_juggler(len(last_hour_oids))

    # публикуем результат на stat
    StatPublisher.create_and_upload(
        'Disk/DiskInternal/WaitingOperations',
        'Зависшие операции',
        REPORT_CONFIG_YAML,
        [{'fielddate': from_dt.strftime('%Y-%m-%d %H:%M:00'), 'waiting_operations': len(results)}],
        scale='i'
    )


if __name__ == '__main__':
    quit_if_mrstat_disabled()
    script_stop_dt = datetime.datetime.now() + datetime.timedelta(hours=1)
    while True:
        default_log.info("start.")
        try:
            main()
        except Exception as e:
            default_log.exception("%r", e)

        if datetime.datetime.now() > script_stop_dt:
            break
        else:
            default_log.info("sleep...")
            time.sleep(60 * 3)
