# -*- coding: utf-8 -*-
import os
import urllib

import mpfs.engine.process
import mpfs.core.base

from mpfs.common.errors import ResourceNotFound, FolderNotFound, NotFolder, HardLinkNotFound, InvalidResourcePathError
from mpfs.common.util import AutoSuffixator, UnicodeBase64
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.config import settings
from mpfs.core import factory
from mpfs.core.address import Address
from mpfs.core.bus import Bus
from mpfs.core.filesystem import hardlinks
from mpfs.core.filesystem.resources.attach import AttachFile
from mpfs.core.filesystem.resources.disk import DiskFolder, DiskFile
from mpfs.core.filesystem.resources.share import SharedFile, SharedResource
from mpfs.core.office.logic.microsoft import MicrosoftEditor
from mpfs.core.office.logic.only_office import OnlyOfficeEditor
from mpfs.core.office.logic.only_office_utils import check_and_change_editor_to_only_office, \
    get_public_info_for_sharing_url, check_country_for_only_office
from mpfs.core.office.logic.sharing_url import generate_and_set_office_doc_short_id
from mpfs.core.office.static import OfficeServiceIDConst, OfficeClientIDConst, OfficeSourceConst, OfficeAccessStateConst
from mpfs.core.operations import manager
from mpfs.core.services.discovery_service import DiscoveryService
from mpfs.core.services.mail_service import Mail, MailStidService
from mpfs.core.social.publicator import Publicator
from mpfs.core.social.share.constants import SharedFolderRights
from mpfs.core.user.anonymous import AnonymousUID
from mpfs.core.user.base import User
from mpfs.core.user.constants import (DEFAULT_OFFICE_FILE_NAMES, DEFAULT_LOCALE, EMPTY_FILE_HASH_MD5,
                                      EMPTY_FILE_HASH_SHA256)
from mpfs.core.office.errors import (OfficeNoHandlerFoundError, OfficeError,
                                     OfficeIsNotAllowedError, OfficePathNotSpecified, OfficeServiceIdNotSpecified,
                                     OfficeStorageNotSupported, OfficeInvalidParameters, OfficeMailAttachmentNotFound,
                                     OfficeServiceFileIdNotSpecified, OfficeFileIsReadOnlyError)
from mpfs.core.office.util import (make_resource_id, make_session_context, check_office_online_size_limits,
                                   SharingURLAddress, SharingURLAddressHelper, build_office_online_url,
                                   generate_office_doc_short_id, get_editor)


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

OFFICE_TEMPLATES = settings.office['templates']
OFFICE_ALTERED_TEMPLATES = settings.office['altered_templates']
SYSTEM_SYSTEM_LIMITS_MAX_NAME_LENGTH = settings.system['system']['limits']['max_name_length']
SYSTEM_SYSTEM_LIMITS_MAX_PATH_LENGTH = settings.system['system']['limits']['max_path_length']
FEATURE_TOGGLES_EDITOR_BY_OWNER_ENABLED = settings.feature_toggles['editor_by_owner_enabled']


_action_handlers_map = {}
mail_service = Mail()
mail_stid_service = MailStidService()


def dispatcher(editor, request, action, ext, locale):
    handler = _action_handlers_map.get(action)
    if handler:
        return handler(editor, request, action, ext, locale)
    raise OfficeNoHandlerFoundError()


def add_to_dispatcher(actions):
    def func_wrapper(handler):
        global _action_handlers_map
        for action in actions:
            _action_handlers_map[action] = handler
        return handler
    return func_wrapper


@add_to_dispatcher(['editnew'])
def editnew_handler(editor, request, action, ext, locale):
    service_id, service_file_id = _get_request_source(request)
    if service_id != 'disk':
        raise OfficeStorageNotSupported()

    ext = ext.lower()
    if ext not in OFFICE_TEMPLATES:
        raise OfficeIsNotAllowedError()

    folder_path = service_file_id  # folder for create new file
    try:
        folder_resource = factory.get_resource(request.uid, folder_path)
    except ResourceNotFound:
        raise FolderNotFound(folder_path)

    if not isinstance(folder_resource, DiskFolder):
        raise NotFolder()

    file_path = _make_file_path(ext, folder_path, request.uid, locale, filename=request.filename)
    log.info('editnew_handler uid: %s adress %s' % (request.uid, file_path))
    address = Address.Make(request.uid, file_path)

    try:
        templates = OFFICE_TEMPLATES
        if experiment_manager.is_feature_active('altered_templates'):
            templates = OFFICE_ALTERED_TEMPLATES
        size = templates[ext]['size']
        md5 = templates[ext]['md5']
        sha256 = templates[ext]['sha256']
        hardlinks.HardLink(md5, size, sha256)
        action = 'edit'
    except HardLinkNotFound:
        log.info('Fallback to zero file')
        size = 0
        md5 = EMPTY_FILE_HASH_MD5
        sha256 = EMPTY_FILE_HASH_SHA256

    fs = Bus(request=request)
    result = fs.store(request.uid, address.id, size=size, md5=md5, sha256=sha256)
    if not (isinstance(result, dict) and result.get('status') == 'hardlinked'):
        error_log.error(
            'Unable to hardlink (uid: %s, path: %s, ext: %s)' % (request.uid, service_file_id, ext)
        )
        raise OfficeError()

    resource = factory.get_resource(request.uid, address)

    if (FEATURE_TOGGLES_EDITOR_BY_OWNER_ENABLED and
            isinstance(folder_resource, SharedResource)):
        editor = get_editor(folder_resource.owner_uid, request)
        if not editor:
            raise OfficeIsNotAllowedError
        if not SharedFolderRights.is_rw(folder_resource.meta['group']['rights']):
            raise OfficeFileIsReadOnlyError
    elif (request.source == OfficeSourceConst.DOCS or
              request.source == OfficeSourceConst.MOBILE):
        editor = check_and_change_editor_to_only_office(
            user=User(request.uid), editor=editor, resource=resource,
            office_selection_strategy=request.set_office_selection_strategy,
            source=request.source, region=request.region
        )

    result = editor.get_edit_data(request.uid, resource,
                                  post_message_origin=request.post_message_origin,
                                  locale=locale,
                                  action='editnew', request=request)
    result.update(get_public_info_for_sharing_url(resource, request_uid=request.uid))
    return result


def _get_request_source(request):
    if request.service_id is not None:
        if request.service_file_id is None:
            raise OfficeServiceFileIdNotSpecified()

        service_id = request.service_id
        service_file_id = request.service_file_id
    elif request.path is not None:
        service_id = 'disk'
        service_file_id = request.path
    else:
        raise OfficeServiceIdNotSpecified()

    return service_id, service_file_id


class EditAction(object):
    action = 'edit'

    @staticmethod
    @add_to_dispatcher(['edit'])
    def strategy(editor, request, action, ext, locale):
        service_id, _ = _get_request_source(request)
        if service_id == 'disk':
            return EditAction.disk(editor, request, locale)
        elif service_id == 'mail':
            return EditAction.mail(editor, request, locale)
        elif service_id == 'web':
            return EditAction.web(editor, request, locale)
        elif service_id == 'public':
            return EditAction.public(editor, request, locale)
        elif service_id == 'sharing_url':
            return EditAction.sharing_url(editor, request, locale)
        elif service_id == 'browser':
            return EditAction.browser(editor, request, locale)
        else:
            raise OfficeStorageNotSupported()

    @classmethod
    def disk(cls, editor, request, locale):
        uid = request.uid
        service_id, service_file_id = _get_request_source(request)
        fs = Bus(request=request)

        ext = os.path.splitext(service_file_id)[1][1:].lower()
        is_convert_required, new_ext, app = editor.get_is_convert_required_new_ext_app_name(ext)
        sharing_url_addr = None
        if not is_convert_required:
            address = Address.Make(uid, service_file_id)
            resource = fs.get_resource(uid, address, unzip_file_id=True)

            if (FEATURE_TOGGLES_EDITOR_BY_OWNER_ENABLED and
                    isinstance(resource, SharedResource)):
                editor = get_editor(resource.owner_uid, request)
                if not editor:
                    raise OfficeIsNotAllowedError
                if not SharedFolderRights.is_rw(resource.meta['group']['rights']):
                    raise OfficeFileIsReadOnlyError
            elif (request.source == OfficeSourceConst.DOCS or
                      request.source == OfficeSourceConst.MOBILE):
                editor = check_and_change_editor_to_only_office(
                    user=User(uid), editor=editor, resource=resource,
                    office_selection_strategy=request.set_office_selection_strategy,
                    source=request.source, region=request.region
                )

            if editor.type_label == OnlyOfficeEditor.type_label:
                # после перехода на ОО могут остаться открытие файлы в MSO
                # или остаться лок (хотя редактирования нет)
                # открываем их в MSO попрежнему
                lock = Bus().get_lock(resource)
                if (lock and
                        lock.get('data', {}).get('office_online_editor_type') == MicrosoftEditor.type_label):
                    editor = MicrosoftEditor

            result = editor.get_edit_data(uid, resource,
                                          post_message_origin=request.post_message_origin,
                                          locale=locale,
                                          action=cls.action, request=request)
            if (resource.is_fully_public() and
                    editor.type_label == OnlyOfficeEditor.type_label and
                    (uid == resource.owner_uid or resource.office_access_state == OfficeAccessStateConst.ALL)):
                office_doc_short_id = None
                if resource.office_doc_short_id is None:
                    try:
                        office_doc_short_id = generate_and_set_office_doc_short_id(resource)
                    except Exception:
                        error_log.exception('Failed to generate office_doc_short_id')
                else:
                    office_doc_short_id = resource.office_doc_short_id

                if office_doc_short_id is not None:
                    sharing_url_addr = SharingURLAddressHelper.build_sharing_url_addr(
                        uid=resource.owner_uid,
                        office_doc_short_id=office_doc_short_id
                    )
            result.update(cls._get_public_info(resource, uid))
        else:
            src_rawaddr = Address.Make(request.uid, service_file_id).id
            fs.check_source(src_rawaddr)

            src_resource = factory.get_resource(uid, src_rawaddr)
            if src_resource.address.storage_name not in ('disk',):
                raise OfficeStorageNotSupported()

            file_ext = os.path.splitext(src_resource.path)[1][1:].lower()
            file_size = src_resource.get_size()

            dst_path = '%s.%s' % (os.path.splitext(service_file_id)[0], new_ext)
            dst_address = fs.autosuffix_address(Address.Make(uid, dst_path))

            fs.check_address(dst_address.id, strict=True)
            fs.check_rights(uid, rawaddress=dst_address.id)

            src_resource = factory.get_resource(uid, service_file_id)

            service_file_id = cls._get_mulca_service_file_id(src_resource)
            result = cls._create_convert_operation(uid, app, service_id, service_file_id,
                                                   file_size, file_ext, dst_address, request.connection_id)

        if sharing_url_addr is not None:
            result['office_online_sharing_url'] = build_office_online_url(OfficeClientIDConst.SHARING_URL,
                                                                          sharing_url_addr.serialize(),
                                                                          request.tld)

        return result

    @classmethod
    def mail(cls, editor, request, locale):
        uid = request.uid
        connection_id = request.connection_id
        ext, size, filename = request.ext, request.size, request.filename
        if ext is None or size is None or filename is None:
            raise OfficeInvalidParameters()

        service_id, service_file_id = _get_request_source(request)
        mail_mid, mail_hid = service_file_id.split('/')
        stid = mail_stid_service.get_mail_stid(uid, mail_mid)

        if stid is None:
            raise OfficeMailAttachmentNotFound()

        service_file_id = mail_stid_service.get_service_file_id(uid, mail_mid, mail_hid)

        is_convert_required, new_ext, app = editor.get_is_convert_required_new_ext_app_name(ext)

        fs = Bus(request=request)
        dst_address = cls._create_destination_address(fs, uid, filename, new_ext)

        if is_convert_required:
            return cls._create_convert_operation(uid, app, service_id, service_file_id,
                                                 size, ext, dst_address, connection_id)
        free_space = fs.check_available_space(uid, dst_address.id)
        return cls._create_mail_to_disk_operation(uid, service_file_id, dst_address, free_space, connection_id)

    @classmethod
    def web(cls, editor, request, locale):
        uid = request.uid
        connection_id = request.connection_id
        ext, size, filename = request.ext, request.size, request.filename
        if ext is None or size is None or filename is None:
            raise OfficeInvalidParameters()

        service_id, service_file_id = _get_request_source(request)

        is_convert_required, new_ext, app = editor.get_is_convert_required_new_ext_app_name(ext)

        fs = Bus(request=request)
        dst_address = cls._create_destination_address(fs, uid, filename, new_ext)

        try:
            service_file_id = UnicodeBase64.urlsafe_b64decode(service_file_id)
        except TypeError:
            # подстраховываемся, если верстка не сделает url decode и повторно
            # закодирует символы.
            service_file_id = urllib.unquote(service_file_id)
            service_file_id = UnicodeBase64.urlsafe_b64decode(service_file_id)

        if is_convert_required:
            return cls._create_convert_operation(uid, app, service_id, service_file_id,
                                                 size, ext, dst_address, connection_id)

        return cls._create_external_copy_operation(uid, service_file_id, dst_address.id, connection_id)

    @classmethod
    def public(cls, editor, request, locale):
        uid = request.uid
        connection_id = request.connection_id

        # service_file_id == public_hash
        service_id, service_file_id = _get_request_source(request)

        address = Publicator().get_address(service_file_id)
        # owner_uid - кто опубликовал
        owner_uid, raw_address_id = address['uid'], address['rawaddress']
        fs = Bus(request=request)
        resource = fs.resource(owner_uid, raw_address_id)
        if not isinstance(resource, (AttachFile, DiskFile, SharedFile)):
            raise OfficeError('Only files are allowed to edit')

        size = resource.size
        filename, ext = os.path.splitext(resource.name)
        ext = ext[1:].lower()
        is_convert_required, new_ext, app = editor.get_is_convert_required_new_ext_app_name(ext)

        dst_address = cls._create_destination_address(fs, uid, filename, new_ext)

        if is_convert_required:
            service_file_id = cls._get_mulca_service_file_id(resource)
            return cls._create_convert_operation(uid, app, service_id, service_file_id,
                                                 size, ext, dst_address, connection_id)

        setattr(request, 'private_hash', service_file_id)
        setattr(request, 'name', '')
        setattr(request, 'save_path', None)
        return mpfs.core.base.async_public_copy(request)

    @classmethod
    def sharing_url(cls, editor, request, locale):
        uid = request.uid
        service_id, service_file_id = _get_request_source(request)
        fs = Bus(request=request)
        sharing_url_addr = SharingURLAddress.parse(service_file_id)
        owner_uid = SharingURLAddressHelper.decrypt_uid(sharing_url_addr.encrypted_uid)

        resource = factory.get_resource_by_uid_and_office_doc_short_id(
            uid=uid,
            owner_uid=owner_uid,
            office_doc_short_id=sharing_url_addr.office_doc_short_id
        )

        editor = get_editor(owner_uid, request)
        if not editor:
            raise OfficeIsNotAllowedError()

        # sharing_url only for Only Office
        if (uid != owner_uid and
                editor.type_label != OnlyOfficeEditor.type_label):
            raise OfficeIsNotAllowedError(data={'short_url': resource.get_short_url()})

        actor_editor = None
        if (not AnonymousUID.is_anonymous_uid(uid)):
            actor_editor = get_editor(uid, request)
        check_and_change_editor_to_only_office(user=User(uid), editor=actor_editor, resource=resource,
                                               office_selection_strategy=request.set_office_selection_strategy,
                                               source=request.source, region=request.region)

        ext = os.path.splitext(resource.name)[1][1:].lower()
        is_convert_required, new_ext, app = editor.get_is_convert_required_new_ext_app_name(ext)

        if not is_convert_required:
            if uid == owner_uid:
                # после перехода на ОО могут остаться открытие файлы в MSO
                # или остаться лок (хотя редактирования нет)
                # открываем их в MSO попрежнему
                lock = Bus().get_lock(resource)
                if (lock and
                        lock.get('data', {}).get('office_online_editor_type') == MicrosoftEditor.type_label):
                    editor = MicrosoftEditor

            result = editor.get_edit_data(uid, resource,
                                          post_message_origin=request.post_message_origin,
                                          locale=locale,
                                          action=cls.action, request=request,
                                          sharing_url_address_schema=True)
            result.update(cls._get_public_info(resource, uid))
        else:
            fs.check_source(resource.address.id)

            if resource.address.storage_name not in ('disk',):
                raise OfficeStorageNotSupported()

            file_ext = os.path.splitext(resource.path)[1][1:].lower()
            file_size = resource.get_size()

            dst_path = '%s.%s' % (os.path.splitext(resource.path)[0], new_ext)
            dst_address = fs.autosuffix_address(Address.Make(uid, dst_path))

            fs.check_address(dst_address.id, strict=True)
            fs.check_rights(uid, rawaddress=dst_address.id)

            storage_file_id = cls._get_mulca_service_file_id(resource)
            result = cls._create_convert_operation(uid, app, service_id, storage_file_id,
                                                   file_size, file_ext, dst_address, request.connection_id)

        return result

    @classmethod
    def browser(cls, editor, request, locale):
        uid = request.uid
        connection_id = request.connection_id
        ext, size, filename = request.ext, request.size, request.filename
        if ext is None or size is None or filename is None:
            raise OfficeInvalidParameters()

        service_id, service_file_id = _get_request_source(request)

        is_convert_required, new_ext, app = editor.get_is_convert_required_new_ext_app_name(ext)

        fs = Bus(request=request)
        dst_address = cls._create_destination_address(fs, uid, filename, new_ext)

        if is_convert_required:
            return cls._create_convert_operation(uid, app, service_id, service_file_id,
                                                 size, ext, dst_address, connection_id)
        free_space = fs.check_available_space(uid, dst_address.id)
        return cls._create_browser_to_disk_operation(uid, service_file_id, dst_address, free_space, connection_id)

    @staticmethod
    def _get_mulca_service_file_id(resource):
        """Сформировать идентификатор файла в мульке `owner_uid:stid`

        ..notes::
            @metal: для service=mulca первая часть не используется и owner_uid нужен только
                для концептуальности и правильных логов (ну и для того, чтобы парсер service-file-id не падал)

        :type resource: :class:`~Resource`
        :rtype: str
        """
        owner_uid = resource.uid
        if isinstance(resource, SharedFile):
            owner_uid = resource.meta['group']['owner']['uid']

        stid = resource.meta['file_mid']
        return '%s:%s' % (owner_uid, stid)

    @staticmethod
    def _create_destination_address(fs, uid, filename, new_ext):
        """Вернуть свободный путь файла в папке `Загрузки`. При необходимости создать папку.

        :rtype: :class:`~Address`
        """
        downloads_folder_address = fs.get_downloads_address(uid)
        fs.mksysdir(uid, type='downloads')
        dst_path = '%s.%s' % (downloads_folder_address.get_child_file(filename).path, new_ext)
        return fs.autosuffix_address(Address.Make(uid, dst_path))

    @classmethod
    def _create_convert_operation(cls, uid, app, service_id, service_file_id, file_size, file_ext,
                                  dst_address, connection_id):
        data = {
            'app': app,
            'action': cls.action,
            'service_id': service_id,
            'service_file_id': service_file_id,
            'service_file_size': file_size,
            'service_file_ext': file_ext,
            'dst_address': dst_address,
            'connection_id': connection_id,
        }
        operation = manager.create_operation(uid, 'office', 'convert', odata=data)
        return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}

    @staticmethod
    def _create_mail_to_disk_operation(uid, service_file_id, dst_address, free_space, connection_id):
        data = {
            'target': dst_address.id,
            'service_file_id': service_file_id,
            'free_space': free_space,
            'connection_id': connection_id,
        }
        operation = manager.create_operation(uid, 'office', 'move_attach', odata=data)
        return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}

    @staticmethod
    def _create_browser_to_disk_operation(uid, service_file_id, dst_address, free_space, connection_id):
        data = {
            'target': dst_address.id,
            'service_file_id': service_file_id,
            'free_space': free_space,
            'connection_id': connection_id,
        }
        operation = manager.create_operation(uid, 'office', 'move_browser', odata=data)
        return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}

    @staticmethod
    def _create_external_copy_operation(uid, file_url, dst_raw_address, connection_id=None):
        data = {
            'connection_id': connection_id,
            'target': dst_raw_address,
            'service_file_url': file_url,
            'provider': 'web',
            'disable_redirects': False,
            'disable_retries': False,
        }
        operation = manager.create_operation(uid, 'external_copy', 'web_disk', odata=data)
        return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}

    @staticmethod
    def _get_public_info(resource, request_uid):
        short_url = resource.get_short_url()
        if short_url:
            # если файл со ссылкой - проверить не нужно ли сделать ее nda-ссылкой
            short_url = Publicator.get_short_url(resource)
        return {'short_url': short_url,
                'is_owner': resource.owner_uid == request_uid,
                'office_access_state': resource.office_access_state,
                'disk_resource_id': resource.resource_id.serialize()}


@add_to_dispatcher(['view', 'getinfo'])
def _handle_wopitest(editor, request, action, _, locale):
    if request.path is None:
        raise OfficePathNotSpecified()
    return editor.handle_wopitest(request.uid, request.path, action, locale)


def _make_file_path(ext, folder_path, uid, locale, filename=None):
    if filename:
        shortname = filename
    else:
        format_names = DEFAULT_OFFICE_FILE_NAMES.get(ext, DEFAULT_OFFICE_FILE_NAMES['docx'])
        shortname = format_names.get(locale, format_names[DEFAULT_LOCALE])

    fullname = '.'.join([shortname, ext])
    file_path = os.path.join(folder_path, fullname)

    folder = factory.get_resource(uid, Address.Make(uid, folder_path))
    folder.load_files()
    folder.load_folders()
    reserved_paths = folder.child_files.viewkeys() | folder.child_folders.viewkeys()
    reserved_names = {Address.Make(uid, p).name for p in reserved_paths}

    auto_suffixator = AutoSuffixator()
    tmp = Address.Make(uid, auto_suffixator(file_path))
    if (len(tmp.path) > SYSTEM_SYSTEM_LIMITS_MAX_PATH_LENGTH or
            len(tmp.name) > SYSTEM_SYSTEM_LIMITS_MAX_NAME_LENGTH):
        raise InvalidResourcePathError
    while tmp.name in reserved_names:
        tmp = Address.Make(uid, auto_suffixator(file_path))
    file_path = tmp.path

    return file_path
