# -*- coding: utf-8 -*-
import time

from mpfs.config import settings
from mpfs.common.errors import DecryptionError
from mpfs.common.util.crypt import CryptAgent
from mpfs.core.office import errors
from mpfs.core.office.util import parse_resource_id


OFFICE_DEFAULT_ACCESS_TOKEN_TTL = settings.office['default_access_token_ttl']
OFFICE_ACCESS_TOKEN_SECRET_KEY = settings.office['access_token_secret_key']
OFFICE_ACCESS_TOKEN_SECRET_KEY_OLD = settings.office['access_token_secret_key_old']
ACCESS_TOKEN_SRC_SEPARATOR = ':'


def _timestamp_millis():
    return int(time.time() * 1000)


def generate_access_token(uid, resource_id, ttl=None, secret_key=None):
    """
    Генерирует access_token для office-online

    :param str uid: UID пользователя который получил токен. Не обязательно совпадает с UID-владельца ресурса.
    :param str resource_id: Идентификатор ресурса сгенерированный методом `mpfs.core.office.util.make_resource_id
    :param int ttl: Время жизни токена.
    :param str secret_key: Ключ используемый для шифрования токена.

    :rtype: tuple
    :return: (<access_token>, <срок годности токена в миллисекундах>)
    """
    ttl = ttl or OFFICE_DEFAULT_ACCESS_TOKEN_TTL
    secret_key = secret_key or OFFICE_ACCESS_TOKEN_SECRET_KEY

    crypt_agent = CryptAgent(key=secret_key)
    expires = int(_timestamp_millis() + ttl)
    src = ACCESS_TOKEN_SRC_SEPARATOR.join([uid, resource_id, str(expires)])

    if isinstance(src, unicode):
        src = src.encode('utf-8')

    return crypt_agent.encrypt(src, urlsafe=True), expires


def parse_access_token(access_token, secret_key=None, secret_key_old=None):
    """
    Извлекает из access_token'а uid, resource_id и срок годности токена в миллисекундах.
    Если не удалось расшифровать, то вернёт None.

    :param access_token: Токен сгенерированный функцией `generate_access_token`
    :param secret_key: Ключ используемый для расшифровки токена.
    :param secret_key_old: Ключ используемый для расшифровки токена если `secret_key` не подошел.
                           Используется при смене токенов, когда нужно поддержать одновременно 2 токена.

    :rtype: tuple | None
    :return: (<uid>, <resource_id>, <expires>) или None
    """
    if isinstance(access_token, unicode):
        access_token = access_token.encode('utf-8')

    secret_key = secret_key or OFFICE_ACCESS_TOKEN_SECRET_KEY
    secret_key_old = secret_key_old or OFFICE_ACCESS_TOKEN_SECRET_KEY_OLD
    result = None
    # Пытаемся сначала расшифровать актуальным ключём, если не подошел и задан старый, то пытаемся расшифровать старым.
    for key in filter(None, [secret_key, secret_key_old]):
        try:
            crypt_agent = CryptAgent(key=key)
            src = crypt_agent.decrypt(access_token, urlsafe=True)
            uid, resource_id, expires = src.split(ACCESS_TOKEN_SRC_SEPARATOR)
            result = (uid, resource_id, int(expires))
        except (DecryptionError, ValueError):
            pass
    return result


def check_access_token(access_token, secret_key=None, secret_key_old=None):
    """
    Проверяет корректность access_token'а и возвращает uid, resource_id и срок годности токена в миллисекундах.
    Если токен не удалось разобрать или он просрочен, вернёт None.

    :param access_token: Токен сгенерированный функцией `generate_access_token`
    :param secret_key: Ключ используемый для расшифровки токена.
    :param secret_key_old: Ключ используемый для расшифровки токена если `secret_key` не подошел.
                           Используется при смене токенов, когда нужно поддержать одновременно 2 токена.

    :rtype: tuple | None
    :return: (<uid>, <resource_id>, <expires>) или None
    """
    token_data = parse_access_token(access_token, secret_key=secret_key, secret_key_old=secret_key_old)
    if token_data:
        _, _, expires = token_data
        if expires > _timestamp_millis():
            return token_data
    # по умолчанию токен не валиден
    return None


def authenticate_user(access_token, resource_id):
    token_data = check_access_token(access_token)
    if not token_data:
        raise errors.OfficeUnauthorizedError()
    uid, parsed_resource_id, _ = token_data

    if parsed_resource_id != resource_id:
        raise errors.OfficeUnauthorizedError()

    try:
        owner_uid, file_id = parse_resource_id(resource_id)
    except ValueError:
        raise errors.OfficeBadResourceIdError()

    return uid, owner_uid, file_id
