# -*- coding: utf-8 -*-
import os
import traceback
from datetime import datetime, timedelta
from urlparse import urlparse

import jwt

import mpfs.engine.process
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.core.address import ResourceId, SharingURLAddress
from mpfs.config import settings
from mpfs.common.errors import MPFSError, ResourceLocked, ResourceLockFailed, ResourceNotFound, SymlinkNotFound, BadArguments
from mpfs.common.util import from_json, unquote_qs_param, UnicodeBase64, to_json
from mpfs.common.util.urls import update_qs_params, urlencode
from mpfs.core import factory
from mpfs.core.address import Address
from mpfs.core.bus import Bus
from mpfs.core.filesystem.helpers.lock import LockHelper
from mpfs.core.filesystem.resources.disk import DiskFile, append_meta_to_office_files, MPFSFile
from mpfs.core.filesystem.resources.group import GroupFile
from mpfs.core.filesystem.resources.share import SharedResource, SharedFile
from mpfs.core.filesystem.symlinks import Symlink
from mpfs.core.metastorage.decorators import user_exists_and_not_blocked
from mpfs.core.office.logic.base_editor import Editor
from mpfs.core.office.logic.microsoft import MicrosoftEditor
from mpfs.core.office.logic.only_office import OnlyOfficeEditor
from mpfs.core.office.logic.only_office_utils import (
    SEP,
    check_and_change_editor_to_only_office,
    is_old_mso_user_by_last_edit_date,
    check_and_change_editor_to_only_office_for_docs,
    check_country_for_only_office,
)
from mpfs.core.office.logic.sharing_url import sync_office_fields_from_link_data, get_resource_by_office_doc_short_id
from mpfs.core.office.static import OfficeAccessStateConst, OfficeClientIDConst, OfficeServiceIDConst, OfficeSourceConst, \
    SettingsVerstkaEditorConst
from mpfs.core.operations.manager import create_operation, get_operation
from mpfs.core.services import mulca_service
from mpfs.core.services.only_office_balancer_service import only_office_balancer_service
from mpfs.core.services.orchestrator_service import orchestrator_service
from mpfs.core.social.publicator import Publicator
from mpfs.core.social.share.constants import SharedFolderRights
from mpfs.core.user import constants
from mpfs.core.user.anonymous import AnonymousUID
from mpfs.core.user.base import User
from mpfs.core.user.constants import PUBLIC_UID
from mpfs.core.office import auth, actions
from mpfs.core.office.errors import (
    OfficeError,
    OfficeServiceNotSupportedError,
    OfficeIsNotAllowedError,
    OfficeFileIsReadOnlyError,
    OfficeInvalidParameters,
    OfficeStorageNotSupported,
    OfficeUnsupportedExtensionError,
    OfficeOnlyOfficeKeyMismatch,
    OfficeInvalidAccessStateParameter,
    OfficeOnlyOfficeOutdatedData,
    OfficeInvalidSelectionStrategyParameter,
    OfficeEditorNotSupportedForSharedEditError,
    OfficeRegionNotSupportedForSharedEditError,
)
from mpfs.core.office.util import (
    make_new_path,
    build_office_online_url,
    check_user_country_match,
    get_editor,
    generate_office_doc_short_id,
    SharingURLAddressHelper,
)
from mpfs.core.user.constants import DISK_AREA
from mpfs.engine.process import get_global_real_tld
from mpfs.platform.utils import parse_cookie
from mpfs.platform.v1.case.handlers import DocviewerYabrowserUrlService
from mpfs.core.office.logic.hancom import AutosaveManager, HancomEditor

default_log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

OFFICE_OPEN_DOCUMENT_BUTTON_URL = settings.office['open_document_button']['url']
OFFICE_OPEN_DOCUMENT_BUTTON_ALLOWED_TLDS_FOR_STANDALONE_APP = settings.office['open_document_button']['allowed_tlds_for_standalone_app']
OFFICE_ONLY_OFFICE_OUTBOX_SECRET = settings.office['only_office']['outbox_secret']
OFFICE_SHARING_URL_SKIP_ACCESS_CHECK = settings.office['sharing_url']['skip_access_check']
DOCVIEWER_URL = settings.system['docviewer']['url']
USER_DEFAULT_TLD = settings.user['default_tld']
OFFICE_LOCK_TTL = 1800  # s
mulca = mulca_service.Mulca()
OFFICE_OP_TYPE = 'office'
FEATURE_TOGGLES_EDITOR_BY_OWNER_ENABLED = settings.feature_toggles['editor_by_owner_enabled']


@user_exists_and_not_blocked
def office_show_change_editor_buttons(request):
    user = User(request.uid)
    is_force_oo = user.get_office_selection_strategy() == OnlyOfficeEditor.STRATEGY_FORCE_OO
    is_old = is_old_mso_user_by_last_edit_date(user.get_last_mso_usage())
    is_mso = get_editor(user, skip_force_oo=True) is MicrosoftEditor
    is_not_oo = not get_editor(user) is OnlyOfficeEditor

    return {'return_old_editor': is_force_oo and is_old and is_mso, 'use_new_editor': is_not_oo}


def office_action_data(request):
    """Вернуть URL для действия с файлом.

    Query string аргументы:
      * uid
      * action - Действие, для которого требуется получить URL. Одно из `ACTIONS`.
      * service_id - идентификатор сервиса в котором находится файл. Пока это 3 сервиса: disk, mail, browser.
        Используется для сценария action=edit.
      * service_file_id - идентификатор файла в этом сервисе. Для disk <path>, для mail это <mail_id>/<part_id>,
        для browser - это <mds_key>.
        Используется для сценария action=edit.
      * path
       Для сценария editnew: путь к файлу или папке (в зависимости от сценария).
       Для сценария edit: поддерживаем для обратной совместимости с версткой - если передали этот параметр, а не
         service_id и service_file_id, то service_id=disk, а service_file_id=path.
      * ext - Расширение файла (для сценария editnew).
      * filename - Имя для нового файла в случае, если service_id != disk и != public
      * locale - Локаль интерфейса пользователя.
      * post_message_origin - значение хоста, с которым будет происходить обмен PostMessages

    :type request: :class:`~mpfs.frontend.request.JSONRequest`
    :rtype: dict
    """
    from mpfs.frontend.api.auth import TVM2
    # Используем TVM авторизацию и юзер-тикет помогает понять какой пользователь инициализировал запрос.
    # Текущая ручка иногда ходит в Почту, в которую нам нужно передать юзер-тикет, чтобы там проверили
    # что запрос пришел на редактирование своего файла в Почте
    TVM2.set_tvm_2_0_user_ticket_from_request(request)
    user = User(request.uid)
    # Костыль на время эксперимента с Ханкомом. В будущем выпилить.
    if request.service_id == 'browser' or HancomEditor.is_tld_forbidden(get_global_real_tld()):
        user.disable_hancom()
    editor = get_editor(request.user or request.uid, request)
    if not editor and request.service_id != OfficeServiceIDConst.SHARING_URL:
        raise OfficeIsNotAllowedError()

    locale = request.locale
    if locale not in constants.SUPPORTED_LOCALES:
        locale = constants.DEFAULT_LOCALE

    action = request.action
    ext = request.ext
    if ext is not None:
        ext = request.ext.lower()
    result = actions.dispatcher(editor, request, action, ext, locale)
    # в actions.dispatcher эдитор мог поменяться, потому проверяем ещё раз
    if ('office_online_editor_type' not in result or
        result['office_online_editor_type'] != OnlyOfficeEditor.type_label):
        possibly_new_editor = get_editor(request.uid, request)
        try:
            if (not AnonymousUID.is_anonymous_uid(request.uid) and
                    possibly_new_editor == MicrosoftEditor):
                user.set_last_mso_usage(datetime.now().strftime('%Y-%m-%d'))
        except Exception as err:
            error_log.error(traceback.format_exc())
    if 'office_online_editor_type' not in result and editor:
        result['office_online_editor_type'] = editor.type_label
    return result


def office_action_check(request):
    """Проверяет, доступно ли указанное действие. Учитывает возможность конвертации и проверяет размер.

    Query string аргументы:
      * uid
      * action - Действие, для которого требуется получить URL. Одно из `ACTIONS`.
      * service_id - идентификатор сервиса в котором находится файл. Пока это 3 сервиса: disk, mail, browser.
      * service_file_id - идентификатор файла в этом сервисе. Для disk <path>, для mail это <mail_id>/<part_id>
      * file_name - Расширение файла.
      * size - Размер файла.

    Cookie параметры:
      * yandexuid - используется для шифрования service_file_id в ссылке
        Только при service_id=browser

    :type request: :class:`~mpfs.frontend.request.JSONRequest`
    :rtype: dict
    """

    if request.service_id == 'browser':
        cookies = parse_cookie(request.request_headers.get('cookie', ''))
        if 'yandexuid' not in cookies:
            raise OfficeInvalidParameters()
        yandexuid = cookies['yandexuid']

    editor = get_editor(request.user or request.uid, request)
    if (not editor and
            (request.service_id and request.service_id != OfficeServiceIDConst.SHARING_URL or
             not request.service_id)):
        raise OfficeIsNotAllowedError()

    if request.source == OfficeSourceConst.DOCS:
        editor = check_and_change_editor_to_only_office_for_docs(request.user, editor)

    action = request.action

    query_args = ('action', 'service_id', 'service_file_id', 'file_name', 'size')
    if not any(getattr(request, a, None) for a in query_args):
        # https://st.yandex-team.ru/CHEMODAN-26116
        return {
            'editnew': True,
            'office_online_editor_type': editor.type_label,
        }

    filename = None
    client_id = request.service_id
    sharing_url_addr = None
    resource = None
    if request.service_id == 'disk':
        # if the file is from disk - get resource extension and get size from meta
        address = Address.Make(request.uid, request.service_file_id)
        resource = factory.get_resource(request.uid, address)

        if action == 'edit' and isinstance(resource, SharedResource) and resource.meta['group']['rights'] != 660:
            raise OfficeFileIsReadOnlyError()

        size = resource.size
        ext = address.ext.lower()
        document_id = address.path[1:] if address.path.startswith('/') else address.path  # split leading / for url
        if resource.office_doc_short_id:
            sharing_url_addr = SharingURLAddressHelper.build_sharing_url_addr(resource.owner_uid,
                                                                              resource.office_doc_short_id)
    elif request.service_id in ('mail', 'web', 'public', 'browser'):
        # otherwise get size from request
        size = request.size
        filename, ext = os.path.splitext(request.file_name)
        if ext:
            ext = ext[1:].lower()
        document_id = request.service_file_id  # <mail_id>/<part_id> or url or <mds_key>
        if request.service_id == 'web':
            document_id = UnicodeBase64.urlsafe_b64encode(document_id)
        if request.service_id == 'browser':
            document_id = DocviewerYabrowserUrlService().create_yabrowser_url(document_id, yandexuid)
    elif request.service_id == 'sharing_url':
        sharing_url_addr = SharingURLAddress.parse(request.service_file_id)
        owner_uid = SharingURLAddressHelper.decrypt_uid(sharing_url_addr.encrypted_uid)

        editor = get_editor(owner_uid, request)
        if not editor:
            raise OfficeIsNotAllowedError()

        resource = factory.get_resource_by_uid_and_office_doc_short_id(
            uid=request.uid,
            owner_uid=owner_uid,
            office_doc_short_id=sharing_url_addr.office_doc_short_id
        )

        if (request.uid != resource.owner_uid and
                resource.office_access_state != OfficeAccessStateConst.ALL):
            raise ResourceNotFound()

        if (action == 'edit' and
                isinstance(resource, SharedResource) and
                not SharedFolderRights.is_rw(resource.meta['group']['rights'])):
            raise OfficeFileIsReadOnlyError()

        size = resource.size
        ext = os.path.splitext(resource.name)[1][1:].lower()
        client_id = OfficeClientIDConst.DISK
        document_id = resource.address.path
        if document_id.startswith('/'):
            document_id = document_id[1:]
    else:
        raise OfficeServiceNotSupportedError()

    editor.check_action_possible(ext, size, action=action)

    url = build_office_online_url(client_id, document_id, request.tld)

    if request.source:
        url = update_qs_params(url, {'source': request.source})

    if request.service_id in ('mail', 'web', 'public', 'browser'):
        filename = unquote_qs_param(filename)
        additional_params = {
            'filename': filename,
            'ext': ext,
            'size': size,
        }
        url = update_qs_params(url, additional_params)

    result = {
        'office_online_url': url,
        'office_online_editor_type': editor.type_label,
    }
    if resource:
        filename, ext = os.path.splitext(resource.name)
        ext = ext[1:].lower()
        is_convert_required, _, _ = editor.get_is_convert_required_new_ext_app_name(ext)
        if not is_convert_required:
            result['office_access_state'] = resource.office_access_state

    if (request.service_id in (OfficeServiceIDConst.DISK, OfficeServiceIDConst.SHARING_URL) and
            sharing_url_addr is not None):
        result['office_online_sharing_url'] = build_office_online_url(OfficeClientIDConst.SHARING_URL,
                                                                      sharing_url_addr.serialize(),
                                                                      request.tld)
    return result


def office_store(request):
    code, headers, body = 200, {}, {}
    try:
        code, headers, body = _office_store(request)
    except (OfficeError, ResourceNotFound) as e:
        code = e.response
    except Exception:
        code = 500

    response = request.http_resp
    response.status = code
    body.update({
        'response_code': code,
        'response_headers': headers,
        'response_body': ''
    })
    return body


def _office_store(request):
    """Создать операцию по перезаписи файла по его resource_id.

    Ручка, имея uid, owner_uid и file_id, получает путь, где файл находится в настоящий момент
    у пользователя uid. Создает операцию на перезапись (force = 1) существующего файла для uid:path.

    Query string аргументы:
      * resource_id
      * access_token

    Body аргументы:
      * headers

    :type request: :class:`~mpfs.frontend.request.JSONRequest`
    :rtype: tuple[int, dict, dict]
    """
    data = from_json(request.http_req.data)
    if not data:
        raise MPFSError('Expected body')

    uid, owner_uid, file_id = auth.authenticate_user(request.access_token, request.resource_id)

    fs = Bus(request=request)
    resource = get_resource_by_office_doc_short_id(uid=uid, owner_uid=owner_uid, office_doc_short_id=file_id)
    if resource:
        # дальше создадим операцию от лица Владельца
        uid = owner_uid
    if not resource:
        resource = fs.resource_by_file_id(uid, file_id, owner_uid)

    headers = {k.upper(): v for k, v in data['headers'].iteritems()}
    office_lock_id = headers.get('X-WOPI-LOCK')

    lock, existing_office_lock_id = _get_office_lock(fs, resource)
    if existing_office_lock_id and office_lock_id != existing_office_lock_id:
        default_log.info('Already locked with another lock. lock: %s, existing_lock: %s' % (office_lock_id, existing_office_lock_id))
        return 409, {'X-WOPI-Lock': existing_office_lock_id}, {}

    if not existing_office_lock_id and resource.size != 0:
        default_log.info('Resource have no lock but have size')
        return 409, {'X-WOPI-Lock': ''}, {}

    raw_address = resource.address.id
    if isinstance(resource, SharedFile):
        raw_address = resource.address.visible_id

    changes = None
    if resource.meta.get('public', False):
        changes = {'public': True}
    try:
        # Хеши мы не передаем, с нулевым файлом в эту ручку Офис никогда не приходит,
        # следовательно всегда будет возвращаться операция.
        operation = fs.store(
            uid, raw_address, force=True, client_type=request.client_type,
            skip_self_lock=True, skip_check_space=True,
            oper_type='office', oper_subtype='overwrite', changes=changes
        )
    except (ResourceLocked, ResourceLockFailed):
        error_log.exception('Lock error')
        return 409, {'X-WOPI-Lock': ''}, {}

    store_info = operation.kladun._preprocess_body_arguments(operation.kladun_data())
    return 200, {}, {'store_info': store_info}


def office_download_redirect(request):
    """Вернуть ответ с редиректом на скачивание файла с заберуна.

    Ручка, получив resource_id и access_token, проверяет аутентификацию и возвращает ответ для редиректа на заберун
    с помощью nginx'а

    Query string аргументы:
      * resource_id
      * access_token

    :type request: :class:`~mpfs.frontend.request.JSONRequest`
    :rtype: dict
    """

    uid, owner_uid, file_id = auth.authenticate_user(request.access_token, request.resource_id)

    fs = Bus(request=request)
    resource = get_resource_by_office_doc_short_id(uid=uid, owner_uid=owner_uid, office_doc_short_id=file_id)

    if not resource:
        resource = fs.resource_by_file_id(uid, file_id, owner_uid=owner_uid)

    mid = resource.meta['file_mid']
    file_direct_download_url = mulca.get_local_mulcagate_url(mid)

    headers = {
        'Content-Type': resource.mimetype,
        'Content-Disposition': 'attachment; filename="%s"' % resource.name.encode('utf-8'),
        'Location': file_direct_download_url
    }
    response = request.http_resp
    response.headers.update(headers)
    response.status = 302

    return {}


def office_lock(request):
    """Залочить, перелочить ресурс или продлить TTL лока.

    Query string аргументы:
      * uid
      * owner_uid
      * file_id
      * office_lock_id
      * old_office_lock_id

    :type request: :class:`~mpfs.frontend.request.JSONRequest`
    :rtype: dict
    """

    uid = request.uid
    owner_uid = request.owner_uid
    file_id = request.file_id

    fs = Bus(request=request)

    resource = get_resource_by_office_doc_short_id(uid=uid, owner_uid=owner_uid, office_doc_short_id=file_id)

    if not resource:
        resource = fs.resource_by_file_id(uid, file_id, owner_uid)

    if not isinstance(resource, (DiskFile, SharedFile)):
        raise OfficeIsNotAllowedError()
    if isinstance(resource, SharedResource) and resource.meta['group']['rights'] != 660:
        raise OfficeIsNotAllowedError()

    office_lock_id = request.office_lock_id
    old_office_lock_id = request.old_office_lock_id

    # double lock sequence
    lock, existing_office_lock_id = _get_office_lock(fs, resource)
    if existing_office_lock_id and office_lock_id == existing_office_lock_id:
        old_office_lock_id = existing_office_lock_id

    if not old_office_lock_id:
        success, office_lock_id = _lock(fs, resource, office_lock_id)
    else:
        success, office_lock_id = _relock(fs, resource, office_lock_id, old_office_lock_id)

    return {'success': success, 'office_lock_id': office_lock_id}


def office_hancom_lock(req):
    resource_id = ResourceId(req.owner_uid, req.file_id)
    operation = AutosaveManager().start(req.uid, resource_id)
    return {
        'oid': operation.id,
        'type': operation.type,
    }


def office_hancom_unlock(req):
    operation = get_operation(req.uid, req.oid)
    AutosaveManager().stop(operation)


def office_hancom_store(req):
    operation = get_operation(req.uid, req.oid)
    store_operation = AutosaveManager().hancom_store(operation)
    return {
        'upload_url': store_operation['upload_url'],
        'oid': store_operation.id,
        'type': store_operation.type,
        'at_version': store_operation.at_version
    }


def office_unlock(request):
    """Разлочить ресурс.

    Query string аргументы:
      * uid
      * owner_uid
      * file_id
      * office_lock_id

    :type request: :class:`~mpfs.frontend.request.JSONRequest`
    :rtype: dict
    """

    uid = request.uid
    owner_uid = request.owner_uid
    file_id = request.file_id

    fs = Bus(request=request)

    resource = get_resource_by_office_doc_short_id(uid=uid, owner_uid=owner_uid, office_doc_short_id=file_id)
    if not resource:
        resource = fs.resource_by_file_id(uid, file_id, owner_uid)

    if not isinstance(resource, (DiskFile, SharedFile)):
        raise OfficeIsNotAllowedError()
    if isinstance(resource, SharedResource) and resource.meta['group']['rights'] != 660:
        raise OfficeIsNotAllowedError()

    success = False
    lock, office_lock_id = _get_office_lock(fs, resource)
    if office_lock_id == request.office_lock_id:
        try:
            fs.unset_lock(resource)
            office_lock_id, success = '', True
        except ResourceLocked:
            pass

    return {'success': success, 'office_lock_id': office_lock_id}


def office_rename(request):
    """Переименовать файл.

    Query string аргументы:
      * uid - идентификатор пользователя
      * owner_uid - идентификатор владельца файла
      * file_id - идентификатор файла
      * new_name - новое имя файла без расширения
      * office_lock_id - значение лока файла
    """
    uid = request.uid
    owner_uid = request.owner_uid
    file_id = request.file_id
    name = request.new_name

    fs = Bus(request=request)

    resource = get_resource_by_office_doc_short_id(uid=uid, owner_uid=owner_uid, office_doc_short_id=file_id)
    if resource:
        uid = owner_uid
    if not resource:
        resource = fs.resource_by_file_id(uid, file_id, owner_uid)

    if not isinstance(resource, (DiskFile, SharedFile)):
        raise OfficeIsNotAllowedError()

    if isinstance(resource, SharedFile) and resource.meta['group']['rights'] != 660:
        raise OfficeIsNotAllowedError()

    src_path = resource.id
    dst_path = make_new_path(src_path, name)
    src_rawaddress = Address.Make(uid, src_path).id
    dst_rawaddress = Address.Make(uid, dst_path).id

    if src_rawaddress == dst_rawaddress:
        return {'success': True, 'office_lock_id': ''}

    fs.check_address(dst_rawaddress, strict=True)
    fs.check_lock(dst_rawaddress)

    lock, office_lock_id = _get_office_lock(fs, resource)

    success = False
    if not office_lock_id:
        if lock:
            return {'success': False, 'office_lock_id': ''}

        fs.check_moveability(src_rawaddress, dst_rawaddress)
        fs.move_resource(uid, src_rawaddress, dst_rawaddress, force=False, lock=True)
        success = True
    else:
        fs.check_moveability(src_rawaddress, dst_rawaddress)
        fs.unset_lock(resource)
        # TODO
        # сделать rollback на unset_lock если move_resource завалился
        fs.move_resource(uid, src_rawaddress, dst_rawaddress, force=False, lock=True)

        resource = fs.resource_by_file_id(uid, file_id, owner_uid)
        _lock(fs, resource, office_lock_id)
        success = True

    return {'success': success, 'office_lock_id': office_lock_id}


def office_enable_hancom(request):
    User(request.uid).enable_hancom()


def office_disable_hancom(request):
    User(request.uid).disable_hancom()

@user_exists_and_not_blocked
def office_generate_online_editor_url(request):
    """Сгенирировать ссылку на онлайн редактор в вебе.

    ПО получает ссылку из этой ручки и пользователь открывает ее, чтобы отредактировать файл в WEB'е.

    Query string аргументы:
      * uid - идентификатор пользователя
      * path - путь до файла
      * tld - желаемый домен верхнего уровня (будет сброшен в default, если переданный не поддерживается для выдачи)
    """
    uid = request.uid
    path = request.path
    tld = request.real_tld

    if tld not in OFFICE_OPEN_DOCUMENT_BUTTON_ALLOWED_TLDS_FOR_STANDALONE_APP:
        tld = USER_DEFAULT_TLD

    raw_addr = Address.Make(uid, path).id

    editor = get_editor(request.user or uid, request)
    if not editor:
        raise OfficeIsNotAllowedError()

    resource = factory.get_resource(uid, raw_addr)
    if resource.address.storage_name not in (DISK_AREA,):
        raise OfficeStorageNotSupported()

    if isinstance(resource, SharedFile) and SharedFolderRights.is_rw(resource.meta['group']['rights']):
        raise OfficeIsNotAllowedError()

    if not editor.is_edit_possible(resource):
        raise OfficeUnsupportedExtensionError()

    client_id = 'disk'
    doc_id = path[1:]
    edit_url = build_office_online_url(client_id, doc_id, tld)
    edit_url = update_qs_params(edit_url, {'uid': uid})

    return {'edit_url': edit_url}


def _only_office_save_file(body, resource, uid, resource_id, session_id=None, subdomain=None):
    from mpfs.core.operations import manager
    parsed_url = urlparse(body['url'])
    odata = {
        'force': True,
        'provider': 'onlyoffice',
        'service_file_url': 'onlyoffice',
        'path': resource.address.id,
        'connection_id': None,
        'file_id': resource.meta['file_id'],
        'target': resource.address.id,
        'set_public': bool(resource.meta.get('public', False)),
    }

    if subdomain:
        odata['service_file_id'] = '%(uid)s:subdomain=%(subdomain)s;' \
                                   'request=%(path)s?%(query)s' % {'uid': resource_id.uid,
                                                                   'path': parsed_url.path.lstrip('/'),
                                                                   'query': parsed_url.query,
                                                                   'subdomain': subdomain}
    else:
        odata['service_file_id'] = '%(uid)s:%(path)s?%(query)s' % {'uid': resource_id.uid,
                                                                   'path': parsed_url.path.lstrip('/'),
                                                                   'query': parsed_url.query}

    if session_id is not None:
        odata['session_id'] = session_id

    store_operation = manager.create_operation(uid, 'external_copy', 'only_office', odata)
    default_log.info('Only Office external_copy operation data: %s\nOperation: %s ' % (
        to_json(odata),
        {'oid': store_operation.id, 'type': store_operation.type, 'at_version': store_operation.at_version}
    ))


def is_resource_shared(uid, file_id):
    try:
        symlink_data = Symlink.find_by_file_id(uid, file_id)
    except SymlinkNotFound:
        return
    return any(OfficeAccessStateConst.ALL == x.get_office_access_state() for x in symlink_data)


def office_only_office_callback(request):
    # see only office doc at https://helpcenter.r7-office.ru/api/editors/callback.aspx
    from mpfs.core.office.logic.only_office_utils import OnlyOfficeToken
    from mpfs.core.operations import manager
    from mpfs.core.office.logic.only_office_utils import get_oid

    default_log.info('Only Office callback raw body: %s' % request.http_req.data)
    decoded_token = OnlyOfficeToken.decode(request.token)
    uid, oid = get_oid(request.key)
    operation = manager.get_operation(uid, oid)
    body = from_json(request.http_req.data)
    body = jwt.decode(body['token'], OFFICE_ONLY_OFFICE_OUTBOX_SECRET)

    # check container that send callback is actual
    callback_subdomain = request.subdomain
    operation_subdomain = operation.data.get('subdomain')
    if operation_subdomain and not operation_subdomain == callback_subdomain:
        raise OfficeOnlyOfficeOutdatedData("Container address do not match operation container. "
                                           "subdomain from request: %s, operation data: %s, operation uid: %s, operation oid: %s" %
                                           (callback_subdomain, operation.data, operation.uid, operation.id)
                                           )

    if (operation.data['raw_resource_id'] != decoded_token['raw_resource_id']) or \
       (decoded_token['key'] != request.key) or (body['key'] != request.key):
        raise OfficeOnlyOfficeKeyMismatch(
            'KeyMismatchError: operation.data.raw_resource_id - %s\ndecoded_token.raw_resource_id - %s\n'
            'decoded_token.key - %s\nrequest.key - %s\nbody.key - %s' % (
                operation.data['raw_resource_id'], decoded_token['raw_resource_id'], decoded_token['key'], request.key,
                body['key']
            ))
    default_log.info('Only Office callback short data %s' % to_json({
        'uid': request.uid,
        'oid': oid,
        'token': decoded_token,
        'body': body,
        'subdomain': callback_subdomain}
    ))

    resource_id = ResourceId.parse(operation.data['raw_resource_id'])
    resource = factory.get_resource_by_resource_id(resource_id.uid, resource_id)

    default_log.info('Only Office callback data %s' % to_json({
        'uid': request.uid,
        'oid': oid,
        'resource_address': resource.address.id,
        'resource_id': resource_id.serialize(),
        'token': decoded_token,
        'body': body,
        'subdomain': callback_subdomain}
    ))
    parent_resource = factory.get_resource(uid, resource.visible_address.get_parent())
    parent_resource.check_rw()
    lock = LockHelper.get_lock(resource)

    if not lock:
        data = {
            "op_type": OFFICE_OP_TYPE,
            "office_online_editor_type": OnlyOfficeEditor.type_label,
            "oid": oid
        }
        LockHelper.lock(resource, data=data, time_offset=OFFICE_LOCK_TTL)
    else:
        try:
            if lock['data']['office_online_editor_type'] != OnlyOfficeEditor.type_label:
                raise ResourceLocked()
        except:
            raise ResourceLocked()
        LockHelper.update(resource, lock['data'], time_offset=OFFICE_LOCK_TTL)

    if body['status'] == 1:
        # пользователь подключается к совместному редактированию документа или отключается от него.
        operation.data['users'] = body.get('users', [])
        actors = body.get('actions', [])
        if actors:
            need_restore_executing_status = False
            for actor in actors:
                if actor['type'] == 1:
                    need_restore_executing_status = True
                    break
            if need_restore_executing_status:
                operation.set_executing()
        operation.save()
    if not is_resource_shared(resource.owner_uid, resource_id.file_id):
        users = set(operation.data.get('users', []))
        kick_unauthorized(users, resource, request.key, operation_subdomain, oid)

    if operation.is_waiting() or operation.is_done():
        operation.set_executing()
    operation.update_dtime()
    session_id = SEP.join([uid, oid])
    if experiment_manager.is_feature_active('only_office_orchestrator'):
        orchestrator_service.refresh_session(session_id)

    if body['status'] == 2:
        # файл был изменён и все вкладки закрыты. необходимо сохранить файл и закрыть операцию по его изменению
        operation.set_completed()
        LockHelper.unlock(resource)
        if experiment_manager.is_feature_active('only_office_orchestrator'):
            _only_office_save_file(body, resource, uid, resource_id, session_id=session_id,
                                   subdomain=operation_subdomain)
        else:
            _only_office_save_file(body, resource, uid, resource_id)
    elif body['status'] == 6:
        # файл был изменён и требует сохранения, но ещё редактируется. необходимо сохранить файл,
        # но не закрывать операцию по его изменению
        if experiment_manager.is_feature_active('only_office_orchestrator'):
            _only_office_save_file(body, resource, uid, resource_id, subdomain=operation_subdomain)
        else:
            _only_office_save_file(body, resource, uid, resource_id)
        operation.data['status_6_received'] = True
        operation.save()
    elif body['status'] == 4:
        actors = body.get('actions', [])
        if actors:
            removed = False
            for actor in actors:
                if actor['type'] == 0 and actor['userid'] in operation.data.get('users', []):
                    operation.data['users'].remove(actor['userid'])
                    removed = True
            if removed:
                operation.save()
        if not operation.data.get('users') and (not operation.data.get('status_6_received', False)):
            operation.data['md5'] = resource.md5()
            operation.data['sha256'] = resource.sha256()
            operation.data['size'] = resource.get_size()
            operation.set_done()
            LockHelper.unlock(resource)

    return {'error': 0}


def kick_unauthorized(users, resource, request_key, subdomain, oid):
    owner = resource.owner_uid
    unauthorized = set(users)
    try:
        # Участники ОП и Владелец всегда имеют доступ для редактирования
        if isinstance(resource, (SharedFile, GroupFile)):
            for authorized in resource.group.all_uids():
                if authorized in unauthorized:
                    unauthorized.remove(authorized)
        if owner in unauthorized:
            unauthorized.remove(owner)
        if unauthorized:
            default_log.info('Only Office drop users data %s' % to_json({
                'owners_uid': owner,
                'uids': unauthorized,
                'key': request_key,
                'subdomain': subdomain,
                'oid': oid
            }))
            only_office_balancer_service.drop_session(list(unauthorized), request_key, subdomain)
    except Exception as ex:
        error_log.exception('Error while dropping unauthorized users %s' % to_json({
            'owners_uid': owner,
            'uids': users,
            'key': request_key,
            'subdomain': subdomain,
            'oid': oid,
            'resource': resource.address.id
        }))


def office_set_editor_type(request):
    User(request.uid).set_online_editor(request.office_online_editor_type)
    return {'office_online_editor_type': request.office_online_editor_type}


def _lock(fs, resource, office_lock_id):
    """Проверить не залочен ли один из родительских ресурсов и поставить лок на ресурс.

    :type fs: :class:`~Filesystem`
    :type resource: :class:`~Resource`
    :type office_lock_id: str
    :rtype: tuple[bool, str]
    """
    success = False
    path = resource.visible_address.path
    try:
        data = {
            "op_type": OFFICE_OP_TYPE,
            "office_online_editor_type": MicrosoftEditor.type_label,
            'office_lock_id': office_lock_id,
        }
        fs.set_lock(resource, data=data, time_offset=OFFICE_LOCK_TTL)
        create_operation(resource.uid, 'office', 'locking_operation_stub', {'path': path})
        success = True
    except (ResourceLockFailed, ResourceLocked):
        lock, office_lock_id = _get_office_lock(fs, resource)

    return success, office_lock_id


def _relock(fs, resource, office_lock_id, old_office_lock_id):
    """Сменить лок на залоченном ресурсе, продлив его TTL.

    Если новый и старый локи совпадают, то просто продлится TTL.

    :type fs: :class:`~Filesystem`
    :type resource: :class:`~Resource`
    :type office_lock_id: str
    :type old_office_lock_id: str
    :rtype: tuple[bool, str]
    """
    lock, existing_office_lock_id = _get_office_lock(fs, resource)

    if not existing_office_lock_id or existing_office_lock_id != old_office_lock_id:
        return False, existing_office_lock_id

    data = lock['data']
    data['office_lock_id'] = office_lock_id
    try:
        fs.update_lock(resource, data, time_offset=OFFICE_LOCK_TTL)
    except ResourceLockFailed:
        return False, existing_office_lock_id

    return True, office_lock_id


def _get_office_lock(fs, resource):
    """Получить запись лока и идентификатор офисного лока.

    :type fs: :class:`~Filesystem`
    :type resource: :class:`~Resource`
    :rtype: tuple[None | dict, None | str]
    """
    office_lock_id = ''
    lock = fs.get_lock(resource)
    if not lock:
        return lock, office_lock_id

    office_lock_id = lock.get('data', {}).get('office_lock_id', office_lock_id)
    return lock, office_lock_id


def office_set_access_state(request):
    """Изменить разрешения для редактирования файла по ссылке.

    Query string аргументы:
      * resource_id
      * access_state

    :type request: :class:`~mpfs.frontend.request.JSONRequest`
    :rtype: dict
    """
    if request.access_state not in OfficeAccessStateConst.values():
        raise OfficeInvalidAccessStateParameter()

    editor = get_editor(request.user or request.uid, request)
    if not editor:
        raise OfficeIsNotAllowedError()

    resource_id = ResourceId.parse(request.resource_id)
    resource = factory.get_resource_by_resource_id(request.uid, resource_id)
    if not resource.is_fully_public():
        raise OfficeIsNotAllowedError()

    if (isinstance(resource, SharedResource) and
            not SharedFolderRights.is_rw(resource.meta['group']['rights'])):
        raise OfficeFileIsReadOnlyError()

    ext = os.path.splitext(resource.name)[1][1:].lower()
    editor.check_action_possible(ext, resource.size)

    # не юзаем resource.get_symlink_obj()
    # т.к. он зачем-то получает symlink на parent, а не на сам ресурс
    # даже если симлинк на ресурс есть
    symlink = Symlink.find(resource.address)
    if symlink.get_office_access_state() == request.access_state:
        return {}

    check_and_change_editor_to_only_office(user=request.user, editor=editor, resource=resource,
                                           office_selection_strategy=request.set_office_selection_strategy,
                                           region=request.region)
    # Если залочен редактированием в MSO, то не выставляем доступ
    lock = Bus().get_lock(resource)
    if (lock and
            lock.get('data', {}).get('office_online_editor_type') == MicrosoftEditor.type_label):
        raise OfficeIsNotAllowedError

    data_to_update = {'office_access_state': request.access_state}

    if symlink.get_office_doc_short_id() is None:
        data_to_update['office_doc_short_id'] = generate_office_doc_short_id()

    if symlink.get_read_only() is True and request.access_state == OfficeAccessStateConst.ALL:
        symlink.update({'read_only': None})

    symlink.update_office_fields(data_to_update)
    sync_office_fields_from_link_data(resource.owner_uid, resource.meta['file_id'])
    Publicator(request=request).notify_xiva(resource, 'office_access_state_changed')

    return {}


def office_get_access_state(request):
    """Получить разрешения для редактирования файла по ссылке.

    Query string аргументы:
      * resource_id

    :type request: :class:`~mpfs.frontend.request.JSONRequest`
    :rtype: dict
    """
    sharing_url_addr = SharingURLAddress.parse(request.office_doc_id)
    owner_uid = SharingURLAddressHelper.decrypt_uid(sharing_url_addr.encrypted_uid)
    symlink_active = True
    try:
        symlinks = Symlink.find_by_office_doc_short_id(owner_uid, sharing_url_addr.office_doc_short_id)
    except SymlinkNotFound:
        symlink_active = False
        symlinks = []
    has_all_access = any(symlink.get_office_access_state() == OfficeAccessStateConst.ALL
                         for symlink in symlinks)
    if has_all_access:
        return {'office_access_state': OfficeAccessStateConst.ALL, 'is_public': symlink_active}
    else:
        return {'office_access_state': OfficeAccessStateConst.DISABLED, 'is_public': symlink_active}


@user_exists_and_not_blocked
def office_info_by_office_doc_short_id(request):
    """Получить информацию о ресурсе по uid + office_doc_short_id.

    Query string аргументы:
      * office_doc_short_id

    :type request: :class:`~mpfs.frontend.request.JSONRequest`
    :rtype: dict
    """
    fs = Bus(request=request)

    owner_uid = request.owner_uid
    if not owner_uid:
        owner_uid = request.uid

    resource = factory.get_resource_by_uid_and_office_doc_short_id(
        uid=request.uid,
        owner_uid=owner_uid,
        office_doc_short_id=request.office_doc_short_id,
        skip_access_check=OFFICE_SHARING_URL_SKIP_ACCESS_CHECK,
    )

    resource.set_request(request)
    filtered_resources = fs.filter_by_blocked_hids([resource])

    if not filtered_resources:
        raise ResourceNotFound()

    return filtered_resources[0].info()


@user_exists_and_not_blocked
def bulk_info_by_office_online_sharing_urls(req):
    """Получить информацию о ресурсах по office_sharing_urls.

    :type req: :class:`~mpfs.frontend.request.JSONRequest`
    :rtype: dict
    """
    raw_office_online_sharing_urls = from_json(req.http_req.data)
    if not isinstance(raw_office_online_sharing_urls, list):
        raise BadArguments()

    fs = Bus(request=req)
    resources = []
    for raw_online_sharing_url in raw_office_online_sharing_urls:
        office_doc_id = raw_online_sharing_url.rsplit('/', 1)[-1]
        sharing_url_addr = SharingURLAddress.parse(office_doc_id)
        owner_uid = SharingURLAddressHelper.decrypt_uid(sharing_url_addr.encrypted_uid)
        try:
            resource = factory.get_resource_by_uid_and_office_doc_short_id(
                uid=req.uid,
                owner_uid=owner_uid,
                office_doc_short_id=sharing_url_addr.office_doc_short_id,
                access_check_func=all
            )
        except ResourceNotFound:
            continue

        if resource is None:
            continue
        resource.set_request(req)
        append_meta_to_office_files(resource, req)
        owner_editor = get_editor(owner_uid, req)
        if not owner_editor or owner_editor.type_label != OnlyOfficeEditor.type_label:
            continue

        if 'office_online_sharing_url' in resource.meta:
            # Если office_doc_id отличается от того, по которому нашли ресурс, значит:
            # 1. есть конфикт ссылок
            # 2. ресурс имеет поля другой ссылки (не по которой мы нашли ресурс)
            # в этом случае не возвращаем ресурс (если придут по ссылке, которая лежит в ресурсе - тогда вернем)
            resource_office_doc_id = resource.meta['office_online_sharing_url'].rsplit('/', 1)[-1]
            if resource_office_doc_id != office_doc_id:
                continue

        resource.meta.pop('office_online_url', None)
        resource.load_views_counter()
        new_path = '%s:/%s' % (str(sharing_url_addr.office_doc_short_id), resource.name)
        resource.id = new_path
        resource.path = new_path
        if isinstance(resource, MPFSFile):
            resource.setup_previews(PUBLIC_UID)
        resources.append(resource)

    filtered_resources = fs.filter_by_blocked_hids(resources)
    return filtered_resources


def office_get_file_filters(request):
    """Получить информацию о фильтрах на открытие файла.

    :type request: :class:`~mpfs.frontend.request.JSONRequest`
    :rtype: dict
    """
    return OnlyOfficeEditor.get_file_filters()


@user_exists_and_not_blocked
def office_get_file_urls(request):
    """Получить URL на редактирование и просмотр по resource_id файла.

    Query string аргументы:
      * resource_id

    :type request: :class:`~mpfs.frontend.request.JSONRequest`
    :rtype: dict
    """
    fs = Bus(request=request)
    response = {}
    uid = request.uid
    tld = request.real_tld
    if tld not in OFFICE_OPEN_DOCUMENT_BUTTON_ALLOWED_TLDS_FOR_STANDALONE_APP:
        tld = USER_DEFAULT_TLD

    resource_id = ResourceId.parse(request.resource_id)
    resource = factory.get_resource_by_resource_id(request.uid, resource_id)

    filtered_resources = fs.filter_by_blocked_hids([resource])
    if not filtered_resources:
        raise ResourceNotFound()
    resource = filtered_resources[0]

    file_filters = OnlyOfficeEditor.get_file_filters()
    if resource.visible_address.ext.lower() not in file_filters['all_supported_types']:
        raise OfficeUnsupportedExtensionError

    file_url = 'ya-disk://%s' % resource.visible_address.path
    response['view_url'] = DOCVIEWER_URL % {'tld': tld,
                                            'qs': urlencode({'url': file_url, 'name': resource.name, 'uid': uid})}

    edit_url = _get_edit_url(resource, request, uid, tld)
    if edit_url:
        response['edit_url'] = edit_url

    return response


def _get_edit_url(resource, request, uid, tld):
    user = User(uid)
    editor = get_editor(user, request)

    try:
        check_country_for_only_office(uid, editor, resource)
    except OfficeRegionNotSupportedForSharedEditError:
        return

    owner_editor = None
    if FEATURE_TOGGLES_EDITOR_BY_OWNER_ENABLED and isinstance(resource, SharedFile):
        if not SharedFolderRights.is_rw(resource.meta['group']['rights']):
            return

        owner_editor = get_editor(resource.owner_uid, request)
        # Юзер-инициатор запроса не может перевести Владельца на ОО, поэтому ссылку на редактирование не можем отдавать
        if owner_editor is not OnlyOfficeEditor:
            return
    elif editor is not OnlyOfficeEditor and not check_user_country_match(uid, 'RU'):
        # Нет смысла отдавать ссылку на редактирование если нет возможности перевести пользователя на OO
        return

    editor = owner_editor or editor
    if editor and editor.is_edit_possible(resource):
        client_id = 'disk'
        document_id = resource.visible_address.path
        if document_id.startswith('/'):
            document_id = document_id[1:]
        edit_url = build_office_online_url(client_id, document_id, tld)
        return edit_url


@user_exists_and_not_blocked
def office_set_selection_strategy(request):
    """Изменить стратегию выбора редактора.

    Query string аргументы:
      * selection_strategy - новая стратегия выбора редактора

    :type request: :class:`~mpfs.frontend.request.JSONRequest`
    :rtype: dict
    """
    if request.selection_strategy not in Editor.ALL_STRATEGIES:
        raise OfficeInvalidSelectionStrategyParameter

    if request.user.get_office_selection_strategy() == request.selection_strategy:
        return {}

    if not check_user_country_match(request.user.uid, 'RU'):
        return {}

    if LockHelper.is_lock_with_data_exist(request.user.uid,
                                          data_field_name='office_online_editor_type',
                                          value='only_office'):
        raise OfficeEditorNotSupportedForSharedEditError

    request.user.set_office_selection_strategy(request.selection_strategy)

    return {}


def office_switch_to_onlyoffice(request):
    """Переключает юзера на OnlyOffice"""
    uid = request.uid

    result = {'switched': False,
              'should_show_onboarding': False}
    user = User(uid)
    editor = get_editor(request.user or request.uid, request)

    is_force_oo = user.get_office_selection_strategy() == OnlyOfficeEditor.STRATEGY_FORCE_OO
    is_old = is_old_mso_user_by_last_edit_date(user.get_last_mso_usage())
    is_mso = get_editor(user, skip_force_oo=True) is MicrosoftEditor
    is_not_oo = not get_editor(user) is OnlyOfficeEditor

    result['return_old_editor'] = is_force_oo and is_old and is_mso
    result['use_new_editor'] = is_not_oo
    if editor:
        result['editnew'] = True
        result['office_online_editor_type'] = editor.type_label
    else:
        result['editnew'] = False
        result['office_online_editor_type'] = None

    if editor and editor is OnlyOfficeEditor:
        return result

    if not check_user_country_match(uid, 'RU'):
        return result

    is_user_used_mso = is_old_mso_user_by_last_edit_date(user.get_last_mso_usage())
    if is_user_used_mso:
        return result

    user_saw_alert = user.get_office_selection_strategy() == Editor.STRATEGY_DEFAULT
    if user_saw_alert:
        return result

    # переключаем юзера на OO
    user.set_office_selection_strategy(Editor.STRATEGY_FORCE_OO)
    result['switched'] = True
    result['should_show_onboarding'] = True
    result['editnew'] = True
    result['office_online_editor_type'] = OnlyOfficeEditor.type_label
    result['return_old_editor'] = is_old and is_mso
    result['use_new_editor'] = False

    # запоминаем что переключение состоялось
    # Если юзер вернется на MSO, то этот сценарий перехода на OO перестанет для него триггериться
    user.set_setting(SettingsVerstkaEditorConst.MIGRATED_TO_ONLY_OFFICE_EDITOR_ON_OPEN_CLIENT, '1',
                     namespace='verstka')

    return result
