# -*- coding: utf-8 -*-
import time

import mpfs.engine.process

from mpfs.config import settings
from mpfs.common.errors import StorageInitUser
from mpfs.common.errors.common import APIError
from mpfs.core.address import Address
from mpfs.core.bus import Bus
from mpfs.core.operations import manager
from mpfs.core.factory import get_resource_by_resource_id
from mpfs.core.user.constants import DEFAULT_LOCALE, PUBLIC_UID
from mpfs.core.office.logic.base_editor import Editor
from mpfs.core.services.hancom_service import HancomService
from mpfs.core.filesystem.helpers.lock import LockHelper
from mpfs.core.office.errors import OfficeNoHandlerFoundError, OfficeIsNotAllowedError, OfficeUnsupportedExtensionError
from mpfs.core.office.errors import HancomBadLockError, HancomEditNotFinished, HancomLockNotFound

default_log = mpfs.engine.process.get_default_log()

OFFICE_HANCOM_LOCK_TTL = 1800  # s
OFFICE_HANCOM_ENABLE_FOR_ALL = settings.office['hancom']['open_document_button']['enable_for_all']
OFFICE_HANCOM_ENABLE_FOR_UIDS = settings.office['hancom']['open_document_button']['enable_for_uids']
OFFICE_HANCOM_DISABLED_TLDS = settings.office['hancom']['disabled_tlds']


class HancomEditor(Editor):
    type_label = 'hancom'
    _discovery = None
    """Словарь вида `extension: hancom_app_name`"""
    hancom_service = HancomService()

    @classmethod
    def is_user_allowed(cls, user_or_uid, request=None):
        """
        :type user_or_uid: str | :class:`~mpfs.core.user.common.CommonUser`
        :rtype: bool
        """
        if not user_or_uid:
            return False
        # циклический импорт
        from mpfs.core.user.base import User
        from mpfs.core.user.common import CommonUser
        if isinstance(user_or_uid, CommonUser):
            user = user_or_uid
        else:
            if user_or_uid == PUBLIC_UID:
                return False
            try:
                user = User(user_or_uid)
            except StorageInitUser:
                return False

        return user.is_hancom_enabled()

    @classmethod
    def is_tld_forbidden(cls, tld):
        return tld in OFFICE_HANCOM_DISABLED_TLDS

    def handle_wopitest(cls, *args, **kwargs):
        raise OfficeNoHandlerFoundError()

    @classmethod
    def get_is_convert_required_new_ext_app_name(cls, ext):
        if not cls.hancom_service.is_extension_supported(ext):
            raise OfficeUnsupportedExtensionError()
        return False, ext, None

    @classmethod
    def get_edit_data(cls, uid, resource, post_message_origin=None, locale=DEFAULT_LOCALE, action='edit', request=None):
        if not cls.is_edit_possible(resource):
            raise OfficeUnsupportedExtensionError()
        from mpfs.core.user.base import User
        user_info = User(uid).get_user_info()
        editor_name = user_info.get('display_name', user_info.get('login', 'editor'))
        extension = resource.visible_address.ext
        action_url = cls.hancom_service.get_edit_url(uid, resource.resource_id, resource.visible_address.name,
                                                     extension, locale, editor_name=editor_name)
        return {
            'action_url': action_url,
            'office_online_editor_type': cls.type_label
        }

    @classmethod
    def is_edit_possible(cls, resource):
        if resource.type != 'file':
            return False
        ext = resource.visible_address.ext
        return cls.hancom_service.is_extension_supported(ext)

    @classmethod
    def check_action_possible(cls, ext, size, action='edit'):
        if not cls.hancom_service.is_extension_supported(ext):
            raise OfficeUnsupportedExtensionError()


class AutosaveManager(object):
    def __init__(self):
        self.lock_helper = LockHelper()
        self.hancom_service = HancomService()

    def start(self, uid, resource_id):
        resource = get_resource_by_resource_id(uid, resource_id)
        if not HancomEditor.is_edit_possible(resource):
            raise OfficeUnsupportedExtensionError()
        if resource.is_shared and not resource.link.is_rw():
            raise OfficeIsNotAllowedError()

        path = resource.visible_address.path
        lock = self.lock_helper.get_lock(resource)
        if lock is None:
            # нам нужно сохранить лок с oid преждем чем запустить операцию, чтобы
            # не было race condition
            oid = manager.generate_operation_id(uid)
            self.lock_helper.lock(
                resource, data={'oid': oid, 'last_downloaded_modification_time': 0, 'type': 'hancom_edit'},
                time_offset=OFFICE_HANCOM_LOCK_TTL)
            operation = manager.create_operation(uid, 'hancom', 'autosave',
                                                 {'resource_id': resource.resource_id.serialize(), 'id': oid})
        else:
            self._check_is_hancom_lock(lock)
            oid = lock['data']['oid']
            operation = manager.get_operation(uid, oid)
        return operation

    def stop(self, operation):
        try:
            resource, lock = self._get_resource_and_lock(operation)
        except HancomLockNotFound:
            pass
        else:
            self.lock_helper.unlock(resource)
        operation.set_completed()

    def hancom_store(self, operation):
        resource, lock = self._get_resource_and_lock(operation)
        # Hancom гарантирует, что дергает store только при завершении редактирования, поэтому ставим в таймстемп лока
        # текущее время, которое будет заведомо больше, чем таймстемпы при автосохранении, чтобы оно не успело записать
        # устаревшую версию между вызовами office_hancom_store и office_hancom_unlock
        lock['data']['last_downloaded_modification_time'] = int(time.time() * 10 ** 3)
        self.lock_helper.update(resource, data=lock['data'])
        changes = None
        if resource.meta.get('public', False):
            changes = {'public': True}
        return Bus().store(
            resource.visible_address.uid, resource.visible_address.id, force=True,
            skip_self_lock=True, skip_check_space=True,
            oper_type='hancom', oper_subtype='overwrite', changes=changes
        )

    def check_changes(self, operation):
        resource, lock = self._get_resource_and_lock(operation)
        extension = resource.visible_address.ext.lower()
        file_id = resource.meta['file_id']
        try:
            last_modified, continue_polling = self.hancom_service.get_document_status(extension, file_id)
        except APIError as e:
            if e.status_code >= 500:
                # будем полить до тех пор, пока лок не протухнет
                raise HancomEditNotFinished()
            else:
                self.lock_helper.unlock(resource)
                raise

        if not continue_polling and lock['data']['last_downloaded_modification_time'] >= last_modified:
            # С последней записи изменений не было и не будет, снимаем лок и выходим
            self.lock_helper.unlock(resource)
            return

        if lock['data']['last_downloaded_modification_time'] < last_modified:
            download_url = self.hancom_service.get_download_url(extension, file_id)
            service_file_id = '%(uid)s:%(app_type)s/%(document_id)s' % {
                'uid': resource.visible_address.uid,
                'app_type': self.hancom_service.get_app_alias(extension),
                'document_id': file_id,
            }
            odata = {
                'connection_id': '',
                'target': Address.Make(operation.uid, resource.visible_address.path).id,
                'file_id': file_id,
                'service_file_url': download_url,
                'provider': 'hancom',
                'hancom': {
                    'modification_time': last_modified,
                },
                'service_file_id': service_file_id,
                'force': 1,
            }
            if resource.meta.get('public', False):
                odata['changes'] = {'public': True}
            if not continue_polling:
                odata['hancom']['final_modification_time'] = last_modified
            manager.create_operation(operation.uid, 'external_copy', 'hancom_auto_download', odata)
            lock['data']['last_downloaded_modification_time'] = last_modified

        self.lock_helper.update(resource, data=lock['data'], time_offset=OFFICE_HANCOM_LOCK_TTL)
        if continue_polling:
            raise HancomEditNotFinished()

    def is_updating_needed(self, edited_resource, new_modification_time):
        lock = self.lock_helper.get_lock(edited_resource)
        if lock is None:
            default_log.info('Hancom lock not found, probably, was removed due to finished process')
            return False
        if lock['data']['last_downloaded_modification_time'] > new_modification_time:
            return False
        return True

    def unlock_by_operation_data(self, resource, operation_data):
        if 'hancom' in operation_data and 'final_modification_time' in operation_data['hancom']:
            self.lock_helper.unlock(resource)

    def _get_resource_and_lock(self, operation):
        from mpfs.core.office.operations import HancomAutosaveOperation
        if not isinstance(operation, HancomAutosaveOperation):
            raise TypeError("Expect `HancomAutosaveOperation`. Got: %s" % type(operation))
        resource = get_resource_by_resource_id(operation.uid, operation.get_resource_id())
        lock = self.lock_helper.get_lock(resource)
        if not lock:
            raise HancomLockNotFound("address: %r. resource_id: %r" % (resource.visible_address.id, operation.get_resource_id()))
        self._check_is_hancom_lock(lock)
        if lock['data']['oid'] != operation.id:
            raise HancomBadLockError("oid in lock missmatch. OID: %r. Lock: %r" % (operation.id, lock))
        return resource, lock

    def _check_is_hancom_lock(self, lock):
        if (not isinstance(lock, dict) or
                'data' not in lock or
                lock['data'].get('type', None) != 'hancom_edit' or
                not lock['data'].get('oid', None)):
            raise HancomBadLockError("Lock: %r" % lock)
