# -*- coding: utf-8 -*-
import os
import urllib
from urlparse import parse_qs, urlparse

from mpfs.common.errors import StorageInitUser
from mpfs.config import settings
from mpfs.core import factory
from mpfs.core.address import Address
from mpfs.core.user.constants import DEFAULT_LOCALE, PUBLIC_UID
from mpfs.core.services.discovery_service import DiscoveryService
from mpfs.core.services.passport_service import Passport
from mpfs.core.office import auth
from mpfs.core.office.logic.base_editor import Editor
from mpfs.core.office.util import (make_resource_id, make_session_context,
                                   check_office_online_size_limits, check_convert_size_limits, OfficeDocShortID)
from mpfs.core.office.errors import OfficeNoHandlerFoundError, OfficeError


passport = Passport()


OFFICE_TURN_OFF_FOR_ALL = settings.office['open_document_button']['turn_off_for_all']
OFFICE_DISABLE_UID_CHECK = settings.office['open_document_button']['disable_uid_check']
OFFICE_AVAILABLE_FOR_UIDS = settings.office['open_document_button']['available_for_uids']
OFFICE_AVAILABLE_FOR_YATEAM = settings.office['open_document_button']['available_for_yateam']
OFFICE_DISABLE_FOR_B2B = settings.office['open_document_button']['disable_for_b2b']
FEATURE_TOGGLES_CHECK_OFFICE_BY_DOMAIN_ENABLED = settings.feature_toggles['check_office_by_domain_enabled']
OFFICE_WOPI_RESOURCE_URL_TEMPLATE = settings.office['wopi_resource_url_template']


# https://st.yandex-team.ru/DOCVIEWER-1797
OFFICE_CONVERT_MAP = {
    'doc': 'docx',
    'xls': 'xlsx',
    'ppt': 'pptx',
}


class MicrosoftEditor(Editor):
    type_label = 'microsoft_online'

    text_types = {'docx', 'docm', 'dotm', 'dotx'}
    text_type = 'text'

    spreadsheet_types = {'xlsx', 'xltx', 'xlsb', 'xlsm'}
    spreadsheet_type = 'spreadsheet'

    presentation_types = {'pptx', 'ppsx', 'potm', 'potx', 'ppsm', 'pptm', 'thmx'}
    presentation_type = 'presentation'

    @classmethod
    def _get_document_type(cls, ext):
        if ext in cls.text_types:
            return cls.text_type
        elif ext in cls.spreadsheet_types:
            return cls.spreadsheet_type
        elif ext in cls.presentation_types:
            return cls.presentation_type
        # Не бросаем исключение, т.к. для MSO это вспомогательная информация
        return cls.text_type

    @classmethod
    def get_edit_data(cls, uid, resource, post_message_origin=None, locale=DEFAULT_LOCALE, action='edit', request=None,
                      sharing_url_address_schema=False):
        session_context = make_session_context(post_message_origin, locale)

        ext = resource.visible_address.ext.lower()
        address = resource.visible_address
        app, url = DiscoveryService().get_app_url(action, ext, locale=locale)

        check_office_online_size_limits(app, action, resource.size)

        action_url, resource_url, resource_id = cls._get_resource_params(
            uid, address, url, session_context,
            sharing_url_address_schema=sharing_url_address_schema
        )
        access_token, access_token_expires = auth.generate_access_token(uid, resource_id)

        return {
            'action_url': action_url,
            'access_token': access_token,
            'access_token_ttl': access_token_expires,
            'resource_url': resource_url,
            'resource_id': resource_id,
            'resource_path': resource.id,
            'office_online_editor_type': cls.type_label,
            'editor_config': {'documentType': cls._get_document_type(ext)}
        }

    @classmethod
    def get_app_url(cls, action, ext, locale=DEFAULT_LOCALE):
        return DiscoveryService().get_app_url(action, ext, locale=locale)

    @classmethod
    def is_user_allowed(cls, user_or_uid, request=None):
        """
        Проверить, доступна ли пользователю функциональность Office Online в соответствии с настройками.
        См. "office->open_document_button".

        :type user_or_uid: str | :class:`~mpfs.core.user.common.CommonUser`
        :rtype: bool
        """
        from mpfs.core.user.common import CommonUser  # цикл. импорт
        from mpfs.core.user.base import User

        if user_or_uid is None or user_or_uid == PUBLIC_UID:
            return OFFICE_DISABLE_UID_CHECK
        elif isinstance(user_or_uid, basestring):
            uid = user_or_uid
            try:
                user = User(user_or_uid)
            except StorageInitUser:
                return OFFICE_DISABLE_UID_CHECK
        else:
            user = user_or_uid
            uid = getattr(user, 'uid', None)

        if OFFICE_TURN_OFF_FOR_ALL or uid is None:
            return False

        if isinstance(user, CommonUser):
            if not cls._is_user_allowed(user):
                return False

        if OFFICE_DISABLE_UID_CHECK:
            return True
        return cls._is_uid_allowed(uid)

    @classmethod
    def is_edit_possible(cls, resource):
        """Проверить, доступно ли для файла редактирование в Office Online:

        - файл должен быть поддерживаемого расширения (проверяется также возможность сконвертировать его в поддерживаемый
        формат)
        - размер файла должен удовлетворять ограничениям
        """
        if resource.type != 'file':
            return False
        extension = resource.visible_address.ext
        if not extension:
            return False

        try:
            cls.check_action_possible(extension, resource.get_size())
        except OfficeError:
            return False

        return True

    @classmethod
    def get_is_convert_required_new_ext_app_name(cls, ext):
        try:
            app, _ = DiscoveryService().get_app_url('edit', ext)
        except OfficeError:
            new_ext = OFFICE_CONVERT_MAP.get(ext, None)
            app, _ = DiscoveryService().get_app_url('edit', new_ext)
            return new_ext != ext, new_ext, app
        else:
            return False, ext, app

    @classmethod
    def handle_wopitest(cls, uid, path, action, locale):
        ext = os.path.splitext(path)[1][1:].lower()
        if ext != 'wopitest':
            raise OfficeNoHandlerFoundError()

        discovery_service = DiscoveryService()
        app, url = discovery_service.get_app_url(action, ext, locale=locale)

        address = Address.Make(uid, path)
        action_url, resource_url, resource_id = cls._get_resource_params(uid, address, url, session_context='')
        access_token, access_token_expires = auth.generate_access_token(uid, resource_id)

        if action == 'getinfo':
            return discovery_service.get_wopi_test_info(action_url, access_token, access_token_expires)

        return {
            'action_url': action_url,
            'access_token': access_token,
            'access_token_ttl': access_token_expires,
            'resource_url': resource_url,
            'resource_id': resource_id,
        }

    @classmethod
    def check_action_possible(cls, ext, size, action='edit'):
        """
        Проверить, доступно ли для файла редактирование в Office Online.

        - расширение должно быть поддерживаемым
        - размер файла должен удовлетворять ограничениям

        :type ext: str
        :type size: int
        :type action: str
        :raises OfficeUnsupportedActionError:
        :raises OfficeUnsupportedExtensionError:
        :raises OfficeFileTooLargeError:
        """
        ext = ext.lower()  # расширение может быть uppercase, но нужно считать его валидным

        new_ext = ext
        try:
            app, _ = DiscoveryService().get_app_url(action, ext)
        except OfficeError:
            new_ext = OFFICE_CONVERT_MAP.get(ext, None)
            app, _ = DiscoveryService().get_app_url(action, new_ext)

        if new_ext == ext:
            check_office_online_size_limits(app, action, size)
        else:
            check_convert_size_limits(ext, size)

    @classmethod
    def _is_user_allowed(cls, user):

        if OFFICE_DISABLE_FOR_B2B and user.is_b2b():
            return False
        if user.get_online_editor() is not None:
            return True
        if FEATURE_TOGGLES_CHECK_OFFICE_BY_DOMAIN_ENABLED:
            if user.is_b2b() or user.is_pdd():
                return user.is_domain_allowed_for_office()
        online_editor_enabled = user.get_online_editor_enabled()
        return online_editor_enabled or (online_editor_enabled is None)

    @classmethod
    def _is_uid_allowed(cls, uid):
        if uid in OFFICE_AVAILABLE_FOR_UIDS:
            return True

        if OFFICE_AVAILABLE_FOR_YATEAM and uid.isdigit():
            has_staff = False
            try:
                user_info = passport.userinfo(uid=uid)
                has_staff = user_info.get('has_staff', has_staff)
            except Exception:
                pass
            return has_staff

        return False

    @classmethod
    def _get_resource_params(cls, uid, address, url, session_context, sharing_url_address_schema=False):
        resource = factory.get_resource(uid, address)
        file_id = resource.meta['file_id']
        if (sharing_url_address_schema and
                resource.office_doc_short_id):
            file_id = OfficeDocShortID(resource.office_doc_short_id).serialize()

        owner_uid = uid if not resource.is_shared else resource.link.group.owner
        resource_id = make_resource_id(resource.owner_uid, file_id)
        resource_url = OFFICE_WOPI_RESOURCE_URL_TEMPLATE % {'resource_id': resource_id}

        query = parse_qs(urlparse(url).query)
        query['WOPISrc'] = resource_url
        if session_context:
            query['sc'] = session_context
        action_url = url.split('?', 1)[0] + '?' + urllib.urlencode(query, doseq=1)

        return action_url, resource_url, resource_id
