# -*- coding: utf-8 -*-
import base64
import hmac
import traceback
import urllib2
from datetime import datetime
import jwt
from hashlib import md5
import urllib
from urlparse import urlunsplit
import mpfs.engine.process
from mpfs.common.errors import StorageInitUser, ResourceLocked, ResourceNotFound, OrchestratorNotFoundError, \
    OrchestratorSessionInactiveError, OverdraftUserPublicLinkDisabled
from mpfs.common.errors.share import GroupNoPermit
from mpfs.common.static import codes
from mpfs.common.util import to_json, ctimestamp
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.config import settings
from mpfs.core import factory
from mpfs.core.filesystem.helpers.lock import LockHelper
from mpfs.core.office.logic.only_office_utils import (
    make_key, OnlyOfficeToken, convert_url_for_oo_proxy, SEP,
    check_country_for_only_office
)
from mpfs.core.office.static import AVAILABLE_TO_DISABLE_DOCUMENT_PERMISSIONS, DocumentPermissionConst
from mpfs.core.services.directory_service import DirectoryService
from mpfs.core.services.orchestrator_service import orchestrator_service
from mpfs.core.services.only_office_balancer_service import only_office_balancer_service
from mpfs.core.user.anonymous import AnonymousUser, AnonymousUID
from mpfs.core.user.base import User
from mpfs.core.user.common import CommonUser

from mpfs.core.office.util import check_office_online_size_limits, OFFICE_SIZE_LIMITS
from mpfs.core.office.errors import OfficeNoHandlerFoundError, OfficeError, OfficeUnsupportedExtensionError, \
    OfficeOnlyOfficeUserLimitReached, OfficeOnlyOfficeFilesLimitReached, OfficeOnlyOfficePublicFilesLimitReached

from mpfs.core.office.logic.base_editor import Editor
from mpfs.core.operations import manager
from mpfs.core.user.constants import DEFAULT_LOCALE, PUBLIC_UID
from mpfs.core.yateam.logic import is_yateam_subtree, user_has_nda_rights
from mpfs.core.services.passport_service import passport

default_log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

OFFICE_ONLY_OFFICE_REST_API_HOST = settings.office['only_office']['rest_api_host']
OFFICE_ONLY_OFFICE_ENABLED = settings.office['only_office']['enabled']
OFFICE_ONLY_OFFICE_ENABLED_FOR_YANDEX_NETS = settings.office['only_office']['enabled_for_yandex_nets']
OFFICE_ONLY_FILES_PER_USER_LIMIT = settings.office['only_office']['files_per_user_limit']
OFFICE_ONLY_SHARED_FILES_PER_USER_LIMIT = settings.office['only_office']['shared_files_per_user_limit']
OFFICE_ONLY_USERS_PER_FILE_LIMIT = settings.office['only_office']['users_per_file_limit']
OFFICE_ONLY_OFFICE_INBOX_SECRET = settings.office['only_office']['inbox_secret']
OFFICE_ONLY_OFFICE_ORCHESTRATOR_SECRET = settings.office['only_office']['orchestrator_secret']
OFFICE_ONLY_OFFICE_BALANCER_URL = settings.office['only_office']['balancer_url']
OFFICE_ONLY_OFFICE_SUPPORTED_LOCALES = settings.office['only_office']['supported_locales']
OFFICE_ONLY_OFFICE_LOGO = settings.office['only_office']['logo']
FEATURE_TOGGLES_ONLYOFFICE_EDITOR_NEW_LOGO_ENABLED = settings.feature_toggles['onlyoffice_editor_new_logo_enabled']
DOCS_VIEW_EXTENSIONS = settings.docs['view_extensions']
VIEW_ACTION = 'view'
OFFICE_ONLY_OFFICE_OPERATION_STALE_TIMEOUT = settings.office['only_office']['operation_stale_timeout']

# https://st.yandex-team.ru/DOCVIEWER-1797
OFFICE_CONVERT_MAP = {
    'doc': 'docx',
    'xls': 'xlsx',
    'ppt': 'pptx',
    'rtf': 'docx',
    'odt': 'docx',
    'ods': 'xlsx',
    'odp': 'pptx',
}


def generate_orchestrator_sign(container):
    salted_container = container + OFFICE_ONLY_OFFICE_ORCHESTRATOR_SECRET
    sign = md5(salted_container).hexdigest()
    return sign


def need_new_operation(resource, operation, key, session_id):
    if not operation.data.get('subdomain'):
        return True

    # Если файл изменился вне рамках операции - нужна новая сессия
    if operation.state == codes.DONE:
        is_same_file = (resource.md5() == operation.data.get('md5') and
                        resource.sha256() == operation.data.get('sha256') and
                        resource.get_size() == operation.data.get('size'))
        if not is_same_file:
            return True

    try:
        error = only_office_balancer_service.info(key, operation.data['subdomain']).json()['error']
    except Exception:
        error_log.exception('Failed to get OO info from container. Trying to check with Orchestrator.')
        try:
            orchestrator_service.get_session(session_id)
        except OrchestratorNotFoundError:
            return True
        return False

    if error != 0:
        return True

    return False


class OnlyOfficeEditor(Editor):
    type_label = 'only_office'

    # https://api.onlyoffice.com/editors/config/#documentType
    text_types = {'docx'}
    text_type = 'text'
    text_app = 'Word'

    spreadsheet_types = {'xlsx'}
    spreadsheet_type = 'spreadsheet'
    spreadsheet_app = 'Excel'

    presentation_types = {'pptx'}
    presentation_type = 'presentation'
    presentation_app = 'PowerPoint'

    supported_types = text_types | spreadsheet_types | presentation_types
    @classmethod
    def is_user_allowed(cls, user_or_uid, request=None, skip_force_oo=False):
        from mpfs.core.degradations import is_allow_only_office

        if not OFFICE_ONLY_OFFICE_ENABLED:
            return False
        if isinstance(user_or_uid, CommonUser):
            user = user_or_uid
        else:
            if user_or_uid == PUBLIC_UID:
                return False
            try:
                user = User(user_or_uid)
            except StorageInitUser:
                return False
        if isinstance(user, AnonymousUser):
            return False

        if not is_allow_only_office(user_or_uid) or request is not None and not is_allow_only_office(request.uid):
            return False

        if not skip_force_oo and user.get_office_selection_strategy() == cls.STRATEGY_FORCE_OO:
            return True
        if user.get_only_office_enabled():
            return True
        if user.get_online_editor() != cls.type_label:
            return False
        if request:
            if OFFICE_ONLY_OFFICE_ENABLED_FOR_YANDEX_NETS:
                return bool(getattr(request, 'is_yandex_nets', 0))
            return True
        return False

    @classmethod
    def is_edit_possible(cls, resource):
        if resource.type != 'file':
            return False
        extension = resource.visible_address.ext
        if not extension:
            return False
        try:
            cls.check_action_possible(extension, resource.get_size())
        except OfficeError:
            return False

        return True

    @classmethod
    def check_users_limit_reached(cls, resource, uid, operation):
        if operation.uid == uid:
            return

        active_users = operation.data.get('users', [])
        if uid not in active_users:
            cls._raise_if_limit_reached(len(active_users), OFFICE_ONLY_USERS_PER_FILE_LIMIT,
                                        'OFFICE_ONLY_USERS_PER_FILE_LIMIT',
                                        OfficeOnlyOfficeUserLimitReached, resource)

    @classmethod
    def check_open_files_limit_reached(cls, resource, actor_user):
        raw_resource_id = resource.resource_id.serialize()

        def get_other_office_operations(user_uid):
            timeout = OFFICE_ONLY_OFFICE_OPERATION_STALE_TIMEOUT
            return [op for op in manager.get_active_operations(user_uid)
                    if op['type'] == 'only_office'
                    and (datetime.now() - datetime.fromtimestamp(op['mtime'])).total_seconds() < timeout
                    and op['data']['raw_resource_id'] != raw_resource_id]

        if not isinstance(actor_user, AnonymousUser):
            active_operations_count = len([op for op in get_other_office_operations(actor_user.uid)
                                           if 'users' not in op['data']
                                           or actor_user.uid in op['data']['users']])
            cls._raise_if_limit_reached(active_operations_count, OFFICE_ONLY_FILES_PER_USER_LIMIT,
                                         'OFFICE_ONLY_FILES_PER_USER_LIMIT',
                                         OfficeOnlyOfficeFilesLimitReached, resource)

        if resource.is_shared and resource.resource_id.uid != actor_user.uid:
            active_operations_count = len(get_other_office_operations(resource.resource_id.uid))
            cls._raise_if_limit_reached(active_operations_count, OFFICE_ONLY_SHARED_FILES_PER_USER_LIMIT,
                                         'OFFICE_ONLY_SHARED_FILES_PER_USER_LIMIT',
                                         OfficeOnlyOfficePublicFilesLimitReached, resource)

    @classmethod
    def check_action_possible(cls, ext, size, action='edit'):
        ext = ext.lower()  # расширение может быть uppercase, но нужно считать его валидным
        if ext not in cls.supported_types:
            if ext not in OFFICE_CONVERT_MAP:
                raise OfficeUnsupportedExtensionError()
            else:
                return check_office_online_size_limits(cls._get_document_app(OFFICE_CONVERT_MAP[ext]), action, size,
                                                       editor_type=cls.type_label)

        return check_office_online_size_limits(cls._get_document_app(ext), action, size, editor_type=cls.type_label)

    @classmethod
    def get_edit_data(cls, uid, resource, post_message_origin=None, locale=DEFAULT_LOCALE, action='edit', request=None,
                      sharing_url_address_schema=False):
        region = ''
        if request and hasattr(request, 'region'):
            region = request.region
        check_country_for_only_office(uid=uid, editor=cls, resource=resource, region=region)

        import mpfs.common.static.codes as codes

        from mpfs.core.services.only_office_balancer_service import only_office_balancer_service
        raw_resource_id = resource.resource_id.serialize()
        data = {
            'raw_resource_id': raw_resource_id,
            'path': resource.visible_address.path,
            'uid': resource.resource_id.uid,
            'actor_uid': uid
        }
        ext = resource.visible_address.ext.lower()
        cls.check_action_possible(ext, resource.size, action)
        user = User(uid)
        cls.check_open_files_limit_reached(resource, user)
        lock = LockHelper.get_lock(resource)
        if lock:
            try:
                if lock['data']['office_online_editor_type'] != cls.type_label:
                    default_log.info(
                        'Only Office changed action from %(old_action)s to %(new_action)s: '
                        'file %(file_data)s locked by %(lock_data)s' % {
                            'old_action': action,
                            'new_action': VIEW_ACTION,
                            'file_data': to_json(data),
                            'lock_data': to_json(lock['data'])
                    })
                    action = VIEW_ACTION
            except:
                raise ResourceLocked()
        if (is_yateam_subtree(resource.path) and user_has_nda_rights(resource.owner_uid)) and (not user_has_nda_rights(uid)):
            raise ResourceNotFound('Tried to access YaTeam directory without permission')
        owner_user = User(resource.owner_uid)
        experiment_manager.update_context(uid=resource.owner_uid)
        if owner_user.is_in_overdraft_for_restrictions() and \
           experiment_manager.is_feature_active('new_overdraft_strategy'):
            raise OverdraftUserPublicLinkDisabled()
        parent_resource = factory.get_resource(uid, resource.visible_address.get_parent())
        try:
            parent_resource.check_rw()
        except GroupNoPermit:
            default_log.info(
                'Only Office changed action from %(old_action)s to %(new_action)s: '
                'no write permission to file %(file_data)s' % {
                    'old_action': action,
                    'new_action': VIEW_ACTION,
                    'file_data': to_json(data)})
            action = VIEW_ACTION

        operation = manager.create_operation(resource.resource_id.uid, 'only_office', 'session', odata=data, additional_statuses=[codes.DONE])
        cls.check_users_limit_reached(resource, uid, operation)

        key = make_key(resource.resource_id.uid, operation.id)
        session_id = SEP.join([operation.uid, operation.id])
        if need_new_operation(resource, operation, key, session_id):
            try:
                only_office_balancer_service.drop_session(
                    operation.data.get('users', []), key, operation.data.get('subdomain')
                )
            except Exception:
                error_log.exception('Failed to drop users from OO session')
            operation.set_completed()
            orchestrator_service.delete_session(session_id)

            operation = manager.create_operation(resource.resource_id.uid, 'only_office', 'session', odata=data)
            key = make_key(resource.resource_id.uid, operation.id)
        else:
            operation.data['path'] = resource.path
            operation.set_executing()

        # работа с оркестратором
        subdomain = None
        if experiment_manager.is_feature_active('only_office_orchestrator'):
            try:
                organizations = DirectoryService().get_organizations_by_uids([resource.owner_uid]).get(resource.owner_uid, [])
                if organizations:
                    group_id = organizations[0].get('id') or ''
                else:
                    group_id = ''
            except Exception:
                error_log.exception('Failed to get organizations from DirectoryService')
                group_id = user.b2b_key or ''
            if group_id and not user.is_paid():
                group_id = 'free_b2b'
            if user.is_yateam():
                group_id = 'yandexoid'

            orchestrator_session_id = SEP.join([operation.uid, operation.id])
            try:
                session_obj = orchestrator_service.create_session(orchestrator_session_id, group_id)
            except OrchestratorSessionInactiveError:
                operation.set_failed({'message': 'Only Office operation out of date'})
                operation = manager.create_operation(resource.resource_id.uid, 'only_office', 'session', odata=data)
                key = make_key(resource.resource_id.uid, operation.id)
                orchestrator_session_id = SEP.join([operation.uid, operation.id])
                session_obj = orchestrator_service.create_session(orchestrator_session_id, group_id)

            if session_obj:
                container = session_obj['container']
                operation.data['container'] = container
                hmac_object = hmac.new(OFFICE_ONLY_OFFICE_ORCHESTRATOR_SECRET, '', md5)
                hmac_object.update(container)
                endpoint_sign = generate_orchestrator_sign(container)
                container = container.replace('.yp-c.yandex.net', '').replace('-', '_').replace(':', '_').replace('.', '_')
                subdomain = '_'.join([container, endpoint_sign])
                operation.data['subdomain'] = subdomain
                operation.save()
                default_log.info('Only Office operation.data: %s' % operation.data)
        mpfs_token = OnlyOfficeToken.encode(key, raw_resource_id)
        uid_oid = '%s:%s' % (operation.uid, operation.id)
        url_qs = urllib.urlencode({'token': mpfs_token, 'uid': resource.resource_id.uid, 'oid': uid_oid, 'subdomain': subdomain})
        callback_url = urlunsplit(
            ('https', OFFICE_ONLY_OFFICE_REST_API_HOST, '/v1/disk/only-office/%s' % key, url_qs, '')
        )
        callback_url = convert_url_for_oo_proxy(callback_url)
        author_user = User(resource.resource_id.uid)

        locale = 'default'
        if (request is not None and
                hasattr(request, 'locale') and
                request.locale in OFFICE_ONLY_OFFICE_SUPPORTED_LOCALES):
            locale = request.locale

        document_type = cls._get_document_type(ext)
        if FEATURE_TOGGLES_ONLYOFFICE_EDITOR_NEW_LOGO_ENABLED:
            logo_image = OFFICE_ONLY_OFFICE_LOGO['image'][locale]
            logo_image_embedded = OFFICE_ONLY_OFFICE_LOGO['image_embedded'][locale]
            logo_url = OFFICE_ONLY_OFFICE_LOGO['url'][document_type]
        else:
            logo_image = 'https://yastatic.net/s3/editor/_/disk_white_36.png'
            logo_image_embedded = 'https://yastatic.net/s3/editor/_/disk_white_36.png'
            logo_url = 'https://disk.yandex.ru/client/recent'

        document_permissions = {
            DocumentPermissionConst.COMMENT: True,
            DocumentPermissionConst.DOWNLOAD: True,
            DocumentPermissionConst.EDIT: True,
            DocumentPermissionConst.FILL_FORMS: True,
            DocumentPermissionConst.PRINT: True,
            DocumentPermissionConst.REVIEW: True
        }
        if (request is not None and
                hasattr(request, 'disable_document_permissions') and
                isinstance(request.disable_document_permissions, basestring)):
            should_disable_permissions = [doc_permission.strip()
                                          for doc_permission in request.disable_document_permissions.split(',')
                                          if doc_permission in AVAILABLE_TO_DISABLE_DOCUMENT_PERMISSIONS]
            for doc_permission in should_disable_permissions:
                document_permissions[doc_permission] = False

        oo_config = {
            'document': {
                'fileType': ext,
                'key': key,
                'permissions': document_permissions,
                'title': resource.visible_address.name,
                'url': convert_url_for_oo_proxy(resource.meta['file_url']),
                'info': {"folder": resource.address.parent_path}
            },
            'documentType': document_type,
            'editorConfig': {
                'callbackUrl': callback_url,
                'mode': action,
                "lang": user.get_supported_locale(),
                'user': {
                    'id': uid,
                    'name': user.get_user_info()['username']
                },
                'customization': {
                    "autosave": True,
                    "chat": False,  # отключен чат
                    "commentAuthorOnly": False,
                    "compactToolbar": False,
                    "feedback": {
                        "url": 'https://forms.yandex.ru/surveys/8167/',  #ссылка на нашу поддержку
                        "visible": True
                    },
                    "forcesave": True,
                    "help": True,
                    "showReviewChanges": False,
                    "zoom": 100,
                    "loaderName": u"Яндекс.Диск",
                    "logo": {
                        "image": logo_image,
                        "imageEmbedded": logo_image_embedded,
                        "url": logo_url
                    },
                },
            }}
        date_created = resource.ctime
        if date_created:
            oo_config['document']['info']['created'] = datetime.fromtimestamp(date_created).strftime("%Y-%m-%d %I:%M %p")

        user_info = author_user.get_user_info()
        if 'public_name' in user_info and user_info['public_name']:
            oo_config['document']['info']['author'] = user_info['public_name']
        if request is not None:
            if hasattr(request, 'locale'):
                oo_config['editorConfig']['lang'] = request.locale
            if hasattr(request, 'oo_height'):
                oo_config['height'] = request.oo_height
            if hasattr(request, 'oo_width'):
                oo_config['width'] = request.oo_width
            if hasattr(request, 'oo_type') and request.oo_type:
                oo_config['type'] = request.oo_type
            if hasattr(request, 'oo_type'):
                oo_config['type'] = request.oo_type
        oo_config['token'] = jwt.encode(oo_config, OFFICE_ONLY_OFFICE_INBOX_SECRET)
        result = {
            'office_online_editor_type': cls.type_label,
            'editor_config': oo_config,
            'resource_path': resource.visible_address.name,
        }
        if OFFICE_ONLY_OFFICE_BALANCER_URL and subdomain:
            result['balancer_url'] = OFFICE_ONLY_OFFICE_BALANCER_URL % subdomain
        default_log.info('Only Office action: %(action)s by %(uid)s. config: %(config)s\ndata: %(data)s ' % {
            'action': action,
            'uid': uid,
            'config': to_json(result),
            'data': to_json({
                'oid': operation.id,
                'resource_address': resource.address.id,
                'resource_id': resource.resource_id.serialize()
            })
        })
        return result

    @classmethod
    def handle_wopitest(cls, uid, path, action, locale):
        raise OfficeNoHandlerFoundError()

    @classmethod
    def get_is_convert_required_new_ext_app_name(cls, ext):
        if ext in OFFICE_CONVERT_MAP:
            new_ext = OFFICE_CONVERT_MAP[ext]
            app = cls._get_document_app(new_ext)
            return True, new_ext, app
        else:
            app = cls._get_document_app(ext)
            return False, ext, app

    @classmethod
    def _get_document_type(cls, ext):
        if ext in cls.text_types:
            return cls.text_type
        elif ext in cls.spreadsheet_types:
            return cls.spreadsheet_type
        elif ext in cls.presentation_types:
            return cls.presentation_type
        raise OfficeUnsupportedExtensionError("Unsupported extension")

    @classmethod
    def _get_document_app(cls, ext):
        if ext in cls.text_types:
            return cls.text_app
        elif ext in cls.spreadsheet_types:
            return cls.spreadsheet_app
        elif ext in cls.presentation_types:
            return cls.presentation_app
        raise OfficeUnsupportedExtensionError("Unsupported extension")

    @classmethod
    def _raise_if_limit_reached(cls, value, limit, limit_name, error, resource):
        if value >= limit:
            default_log.info('Only Office:: limit %s reached. Limit=%d, Value=%d'
                             % (limit_name, limit, value))

            ext = resource.visible_address.ext.lower()
            try:
                doc_type = cls._get_document_type(ext)
            except OfficeUnsupportedExtensionError:
                doc_type = cls.text_type
            data = {
                'short_url': resource.meta.get('short_url'),
                'public_hash': resource.meta.get('public_hash'),
                'name': resource.name,
                'documentType': doc_type
            }
            data = {x: data[x] for x in data if data[x]}

            raise error(data=data)

        default_log.info('Only Office:: limit %s not reached. Limit=%d, Value=%d'
                         % (limit_name, limit, value))

    @classmethod
    def _get_supported_types(cls, supported_types):
        return set(key for key, value in OFFICE_CONVERT_MAP.items() if value in supported_types) | supported_types

    @classmethod
    def get_file_filters(cls):
        text_types = cls._get_supported_types(cls.text_types)
        presentation_types = cls._get_supported_types(cls.presentation_types)
        spreadsheet_types = cls._get_supported_types(cls.spreadsheet_types)
        all_supported_types = text_types | presentation_types | spreadsheet_types | set(DOCS_VIEW_EXTENSIONS)
        size_limits = settings.office['only_office']['size_limits']
        return {
            'text': {
                'supported_types': text_types,
                'size_limit': size_limits.get(cls.text_app, {}).get('edit')
            },
            'presentation': {
                'supported_types': presentation_types,
                'size_limit': size_limits.get(cls.presentation_app, {}).get('edit')
            },
            'spreadsheet': {
                'supported_types': spreadsheet_types,
                'size_limit': size_limits.get(cls.spreadsheet_app, {}).get('edit')
            },
            'all_supported_types': all_supported_types
        }
