# -*- coding: utf-8 -*-
import base64
import jwt
import urllib
import datetime
from urlparse import urlunsplit, urlparse, parse_qs

from mpfs.common.util import to_json, from_json

from mpfs.config import settings
from mpfs.core.bus import Bus
from mpfs.core.office.logic.base_editor import Editor
from mpfs.core.office.logic.microsoft import MicrosoftEditor
from mpfs.core.office.errors import (
    OfficeEditorNotSupportedForSharedEditError,
    OfficeEditorLockedForSharedEditError,
    OfficeIsNotAllowedError,
    OfficeEditorOnPlatformForbidden,
    OfficeRegionNotSupportedForSharedEditError)
from mpfs.core.office.util import check_user_country_match
from mpfs.core.office.static import OfficeSourceConst, SKIP_FORCE_OO, SettingsVerstkaEditorConst
from mpfs.core.social.publicator import Publicator
from mpfs.core.user.anonymous import AnonymousUser, AnonymousUID
from mpfs.core.user.base import User

OFFICE_ONLY_OFFICE_LOCAL_PROXY = settings.office['only_office']['local_proxy']
OFFICE_ONLY_OFFICE_MPFS_TOKEN = settings.office['only_office']['mpfs_token_secret']
OFFICE_SHARED_EDIT_MIGRATE_USERS_TO_ONLYOFFICE = settings.office['shared_edit']['migrate_users_to_onlyoffice']
FEATURE_TOGGLES_RESTRICT_ONLYOFFICE_BY_REGION = settings.feature_toggles['restrict_onlyoffice_by_region']
FEATURE_TOGGLES_RESTRICT_ONLYOFFICE_BY_REGION = settings.feature_toggles['restrict_onlyoffice_by_region']
SEP = ':'


def make_key(uid, oid):
    return base64.b64encode(SEP.join([uid, oid]))


def get_oid(key):
    return base64.b64decode(key).split(SEP, 1)


class OnlyOfficeToken(object):

    @staticmethod
    def encode(key, raw_resource_id):
        return jwt.encode({'key': key, 'raw_resource_id': raw_resource_id}, OFFICE_ONLY_OFFICE_MPFS_TOKEN)

    @staticmethod
    def decode(token):
        return jwt.decode(token, OFFICE_ONLY_OFFICE_MPFS_TOKEN)


def convert_url_for_oo_proxy(url):
    parsed_url = urlparse(url)
    parsed_qs = parse_qs(parsed_url.query.encode('utf8'))
    qs = {}
    for k, v in parsed_qs.iteritems():
        qs[k] = v[0]
    qs['host'] = parsed_url.netloc
    return urlunsplit(('http', OFFICE_ONLY_OFFICE_LOCAL_PROXY, parsed_url.path, urllib.urlencode(qs), ''))


def _raise_error_to_show_alert(user, error):
    if user.get_office_selection_strategy() is None:
        # Выставляем в default, чтобы понимать, что юзер видел alert про смену редактора
        user.set_office_selection_strategy(Editor.STRATEGY_DEFAULT)

    raise error


def is_old_mso_user_by_last_edit_date(last_mso_usage):
    if not last_mso_usage:
        return False
    today = datetime.date.today()
    last_mso_usage = datetime.datetime.strptime(last_mso_usage, "%Y-%m-%d").date()
    return (today - last_mso_usage).days <= settings.office['shared_edit']['old_user_threshold_days']


def check_and_change_editor_to_only_office(user, editor, resource, office_selection_strategy=None, source=None,
                                           region=None):
    if not OFFICE_SHARED_EDIT_MIGRATE_USERS_TO_ONLYOFFICE:
        return editor

    data_for_region_not_supported = {}
    if resource.get_short_url():
        data_for_region_not_supported['short_url'] = resource.get_short_url()

    if isinstance(user, AnonymousUser):
        if region and region.lower() != 'ru':
            raise OfficeRegionNotSupportedForSharedEditError(data=data_for_region_not_supported)
        return editor

    # без редактора показываем DV
    if not editor:
        public_hash = resource.meta.get('public_hash')
        if public_hash:
            data = {'public_hash': public_hash}
        else:
            data = {'path': resource.address.path}
        data['name'] = resource.name
        raise OfficeIsNotAllowedError(data=data)

    from mpfs.core.office.logic.only_office import OnlyOfficeEditor
    is_user_confirmed_force_oo = office_selection_strategy == Editor.STRATEGY_FORCE_OO
    if editor.type_label == OnlyOfficeEditor.type_label:
        return editor

    # Совместное Редактирование только для RU
    if not check_user_country_match(user.uid, 'RU'):
        public_hash = resource.meta.get('public_hash')
        if public_hash:
            data_for_region_not_supported['public_hash'] = public_hash
        else:
            data_for_region_not_supported['path'] = resource.address.path
        raise OfficeRegionNotSupportedForSharedEditError(data=data_for_region_not_supported)

    # Из DOCsов можно скипнуть (из Диска - нет)
    if (source == OfficeSourceConst.DOCS and
            office_selection_strategy == SKIP_FORCE_OO):
        return editor

    # Из DOCsов только раз пытаемся
    if source == OfficeSourceConst.DOCS:
        already_tried = user.settings.get(SettingsVerstkaEditorConst.TRIED_TO_MIGRATE_TO_ONLY_OFFICE_EDITOR_ON_EDIT_FILE_FROM_DOCS,
                                          namespace='verstka')
        if already_tried:
            if not is_user_confirmed_force_oo:
                return editor
        else:
            # Запоминаем что для DOCSов уже пытались
            user.set_setting(SettingsVerstkaEditorConst.TRIED_TO_MIGRATE_TO_ONLY_OFFICE_EDITOR_ON_EDIT_FILE_FROM_DOCS, '1',
                             namespace='verstka')

    # На touch'е можно скипнуть, тогда откроется public
    if (source == OfficeSourceConst.MOBILE and
            office_selection_strategy == SKIP_FORCE_OO):
        public_hash = resource.meta.get('public_hash')
        if public_hash:
            data = {'public_hash': public_hash}
        else:
            data = {'path': resource.address.path}
        raise OfficeEditorOnPlatformForbidden(data=data)

    data = {'documentType': editor._get_document_type(resource.visible_address.ext.lower())}
    if resource.get_short_url():
        data['short_url'] = resource.get_short_url()
    confirm_error = OfficeEditorNotSupportedForSharedEditError(data=data)
    # Смотрим, можем ли переключить юзера на ОО без алертов

    if settings.office['shared_edit']['show_alert_to_everyone'] and not is_user_confirmed_force_oo:
        _raise_error_to_show_alert(user, confirm_error)

    # Если залочен редактированием в MSO -> alert
    lock = Bus().get_lock(resource)
    if (lock and
            lock.get('data', {}).get('office_online_editor_type') == MicrosoftEditor.type_label):
        if office_selection_strategy is not None:
            confirm_error = OfficeEditorLockedForSharedEditError(data=data)
        _raise_error_to_show_alert(user, confirm_error)

    # default-стратегия = юзер видел alert
    # не выставленная стретегия (None) или выставленнеая в другое значение = не видел
    if user.get_office_selection_strategy() == Editor.STRATEGY_DEFAULT and not is_user_confirmed_force_oo:
        raise confirm_error

    # проставляем показ онбоардинга, если юзер не видел алертов
    last_mso_usage = user.get_last_mso_usage()
    if (source != OfficeSourceConst.DOCS and
            user.get_office_selection_strategy() is None and
            is_old_mso_user_by_last_edit_date(last_mso_usage)):
        user.set_setting('shouldShowShareEditOnboarding', '1', namespace='verstka')

    # переключаем юзера на OO
    user.set_office_selection_strategy(Editor.STRATEGY_FORCE_OO)

    return OnlyOfficeEditor


def check_and_change_editor_to_only_office_for_docs(user, editor):
    if not OFFICE_SHARED_EDIT_MIGRATE_USERS_TO_ONLYOFFICE:
        return editor
    if not editor:
        return editor

    from mpfs.core.office.logic.only_office import OnlyOfficeEditor
    if editor.type_label == OnlyOfficeEditor.type_label:
        return editor

    already_tried = user.settings.get(SettingsVerstkaEditorConst.TRIED_TO_MIGRATE_TO_ONLY_OFFICE_EDITOR_ON_OPEN_DOCS,
                                      namespace='verstka')
    if already_tried:
        return editor
    else:
        # помечаем что уже пытались
        user.set_setting(SettingsVerstkaEditorConst.TRIED_TO_MIGRATE_TO_ONLY_OFFICE_EDITOR_ON_OPEN_DOCS, '1',
                         namespace='verstka')

    # default-стратегия = юзер видел alert
    # не выставленная стретегия (None) или выставленнеая в другое значение = не видел
    if user.get_office_selection_strategy() == Editor.STRATEGY_DEFAULT:
        return editor

    # переключаем юзера на OO
    user.set_office_selection_strategy(Editor.STRATEGY_FORCE_OO)

    return OnlyOfficeEditor


def get_public_info_for_sharing_url(resource, request_uid):
    short_url = resource.get_short_url()
    if short_url:
        # если файл со ссылкой - проверить не нужно ли сделать ее nda-ссылкой
        short_url = Publicator.get_short_url(resource)
    return {'short_url': short_url,
            'is_owner': resource.owner_uid == request_uid,
            'office_access_state': resource.office_access_state,
            'disk_resource_id': resource.resource_id.serialize()}


def check_country_for_only_office(uid, editor, resource, region=''):
    from mpfs.core.office.util import get_editor
    from mpfs.core.office.logic.only_office import OnlyOfficeEditor
    # OO только для RU
    if AnonymousUID.is_anonymous_uid(uid) or editor is not OnlyOfficeEditor:
        return

    user = User(uid)

    # для яндексоидов не ограничиваем
    if user.is_yateam():
        return

    default_editor = get_editor(user, skip_force_oo=True)
    if default_editor and default_editor.type_label == OnlyOfficeEditor.type_label:
        return

    data_for_region_not_supported = {}
    if resource.get_short_url():
        data_for_region_not_supported['short_url'] = resource.get_short_url()

    is_allowed_region = True
    if (FEATURE_TOGGLES_RESTRICT_ONLYOFFICE_BY_REGION['enabled'] and
            region and
            region.lower() not in FEATURE_TOGGLES_RESTRICT_ONLYOFFICE_BY_REGION['allowed_regions']):
        is_allowed_region = False

    if not is_allowed_region or not check_user_country_match(uid, 'RU'):
        public_hash = resource.meta.get('public_hash')
        if public_hash:
            data_for_region_not_supported['public_hash'] = public_hash
        else:
            data_for_region_not_supported['path'] = resource.address.path
        raise OfficeRegionNotSupportedForSharedEditError(data=data_for_region_not_supported)
