# -*- coding: utf-8 -*-
from mpfs.common.errors import ResourceNotFound, SymlinkNotFound
from mpfs.config import settings
from mpfs.core.filesystem.dao.file import FileDAO
from mpfs.core.filesystem.symlinks import Symlink
from mpfs.core.office.util import OfficeDocShortID, generate_office_doc_short_id
from mpfs.engine.process import get_error_log


OFFICE_SHARING_URL_SKIP_ACCESS_CHECK = settings.office['sharing_url']['skip_access_check']

error_log = get_error_log()


def sync_office_fields_from_link_data(uid, file_id):
    from mpfs.core.factory import get_not_removed_resource_by_file_id
    from mpfs.core.filesystem.helpers.lock import LockHelper
    try:
        resource = get_not_removed_resource_by_file_id(uid, file_id)
    except ResourceNotFound:
        error_log.exception('Failed to find resource, while syncing office fields: %s:%s' % (uid, file_id))
        return

    FileDAO().update_office_fields_from_link_data(uid, file_id)
    lock = LockHelper.get_lock(resource)
    if lock:
        from mpfs.core.queue import mpfs_queue
        job_data = {
            'uid': uid,
            'file_id': file_id,
        }
        deduplication_id = '%s_%s_%s' % ('sync_office_fields_from_link_data', uid, file_id)
        mpfs_queue.put(job_data, 'sync_office_fields_from_link_data', deduplication_id=deduplication_id, delay=30)


def sync_public_fields_from_link_data(uid, file_id):
    from mpfs.core.factory import get_not_removed_resource_by_file_id
    try:
        resource = get_not_removed_resource_by_file_id(uid, file_id)
    except ResourceNotFound:
        error_log.exception('Failed to find resource, while syncing public fields: %s:%s' % (uid, file_id))
        return

    try:
        symlinks = Symlink.find_by_file_id(uid, file_id)
        symlink = symlinks[0]
    except Exception:
        symlink = None

    if symlink and not resource.is_fully_public():
        try:
            resource.make_public(symlink)
        except Exception:
            pass
    elif not symlink and resource.is_fully_public():
        # reset
        resource.make_private()


def get_resource_by_office_doc_short_id(uid, owner_uid, office_doc_short_id):
    if OfficeDocShortID.is_office_doc_short_id(office_doc_short_id):
        parsed = OfficeDocShortID.parse(office_doc_short_id)
        if not parsed:
            return

        from mpfs.core import factory
        return factory.get_resource_by_uid_and_office_doc_short_id(uid=uid, owner_uid=owner_uid,
                                                                   office_doc_short_id=parsed.office_doc_short_id,
                                                                   skip_access_check=OFFICE_SHARING_URL_SKIP_ACCESS_CHECK)


def generate_and_set_office_doc_short_id(resource):
    symlink = Symlink.find(resource.address)
    if symlink.get_office_doc_short_id() is not None:
        return symlink.get_office_doc_short_id()

    office_doc_short_id = generate_office_doc_short_id()

    symlink.update_office_fields({'office_doc_short_id': office_doc_short_id})
    sync_office_fields_from_link_data(resource.owner_uid, resource.meta['file_id'])
    return office_doc_short_id
