# -*- coding: utf-8 -*-

from datetime import datetime
from time import time

import mpfs.engine.process
import mpfs.core.metastorage.control
from mpfs.common.util import hashed
from mpfs.common.util.experiments.logic import experiment_manager

from mpfs.config import settings
from mpfs.core.address import ResourceId
from mpfs.common.static.tags import ERROR, CODE, MESSAGE
from mpfs.core.address import Address
from mpfs.core.bus import Bus
from mpfs.core.factory import get_resource
from mpfs.core.filesystem.cleaner.models import DeletedStid, DeletedStidSources
from mpfs.core.filesystem.helpers.lock import LockHelper
from mpfs.core.filesystem.quota import Quota
from mpfs.core.filesystem.resources.base import Resource
from mpfs.core.office.errors import (
    HancomEditNotFinished,
    HancomLockNotFound,
    HancomOutdatedModification,
    OfficeEmptyFileNotCreatedError,
    OfficeFileTooLargeError,
    OfficeNoFreeSpaceKladunError,
    OfficeSizeConstraintFailedKladunError,
    OfficeStorageNotSupported,
)
from mpfs.core.office.events import OfficeFileConvertedEvent
from mpfs.core.office.util import check_office_online_size_limits, check_convert_size_limits
from mpfs.core.operations.base import Operation, SaveOnDiskOperation
from mpfs.core.operations.filesystem.copy import BaseExternalCopyWebDisk
from mpfs.core.operations.filesystem.store import StoreDisk, StoreDiskMixin
from mpfs.core.services import kladun_service
from mpfs.core.services.orchestrator_service import orchestrator_service
from mpfs.core.queue import mpfs_queue
from mpfs.common.errors import ResourceNotFound, NoFreeSpace
from mpfs.common.static import codes
from mpfs.common.static.codes import EXECUTING
from mpfs.common.static.messages import titles
from mpfs.common.static.tags import SUCCESS, INITIAL


# sec
OFFICE_LOCK_POLL_INTERVAL = settings.operations['office']['locking_operation_stub']['lock_poll_interval']

log = mpfs.engine.process.get_default_log()


class HancomAutosaveOperation(Operation):
    type = 'hancom'
    subtype = 'autosave'

    def _process(self, *args, **kwargs):
        from mpfs.core.office.logic.hancom import AutosaveManager
        manager = AutosaveManager()
        try:
            manager.check_changes(self)
        except HancomEditNotFinished:
            self.reenque(OFFICE_LOCK_POLL_INTERVAL)
        except Exception:
            mpfs.engine.process.get_error_log().exception("Got unexpected exception during lock update.")
            raise
        else:
            self.set_completed()

    def get_resource_id(self):
        return ResourceId.parse(self.data['resource_id'])


class HancomAutoDownloadToDisk(BaseExternalCopyWebDisk):
    subtype = 'hancom_auto_download'
    keep_lock = True

    def __init__(self, **data):
        from mpfs.core.office.logic.hancom import AutosaveManager  # Циклический импорт
        super(HancomAutoDownloadToDisk, self).__init__(**data)
        self.manager = AutosaveManager()

    def handle_callback_1(self):
        edited_resource = self.__get_to_be_updated_resource()
        if edited_resource is None:
            log.info('No need to update file on first callback')
            # Говорим кладуну, что якобы схардлинкали, чтобы он не заливал файлы в сторедж
            self.data['hardlinked'] = True
            self.post_process_all()
            self.set_completed()
            return
        super(HancomAutoDownloadToDisk, self).handle_callback_1()
        self.manager.unlock_by_operation_data(edited_resource, self.data)

    def handle_callback_2_and_3(self):
        edited_resource = self.__get_to_be_updated_resource()
        if edited_resource is None:
            log.info('No need to update file on second callback')
            # Кладун зря загрузил бинарь, ставим на чистку
            if self.data['filedata']['meta']['file_mid']:
                DeletedStid.controller.bulk_create(
                    [DeletedStid(stid=self.data['filedata']['meta']['file_mid'],
                                 stid_source=DeletedStidSources.KLADUN_CLEAN_UP)])
            return
        super(HancomAutoDownloadToDisk, self).handle_callback_2_and_3()
        self.manager.unlock_by_operation_data(edited_resource, self.data)

    def _check_available_space(self, uid, path, size):
        # Должны загружать файл при превышении места в случае автосохранения
        pass

    def __get_to_be_updated_resource(self):
        """
        Возвращает ресурс, который мы долнжы обновить. Если обновление не требуется - вернем None
        """
        if 'hancom' not in self.data:
            log.info('No Hancom data in operation')
            return None
        try:
            edited_resource = get_resource(self.uid, self.get_path())
        except ResourceNotFound:
            return None

        if self.manager.is_updating_needed(edited_resource, self.data['hancom']['modification_time']):
            return edited_resource
        return None



class OfficeLockingOperationStub(Operation):
    """Операция, которая эмулирует операции файловой системы,
    работающие после установки лока файловой системы. Нужна для синхронизации с ПО.

    Операция этого типа создается вместе с созданием офисного лока
    на файл и активна пока жив лок.

    ..warning::
        В поле ```data``` операции обязательно должно содержаться поле `path`, т.е.
        путь к ресурсу от лица того, кто ставил лок.
        Но офисный лок всегда должен ставиться по пути владельца.

    """
    type = 'office'
    subtype = 'locking_operation_stub'

    def __init__(self, **data):
        self.id = None
        self.uid = None
        self.data = None
        super(OfficeLockingOperationStub, self).__init__(**data)
        self.path = self.data['path']

    def _process(self, *args, **kwargs):
        try:
            lock = LockHelper().get_lock(Address.Make(self.uid, self.path).id)
        except ResourceNotFound:
            self.set_completed()
            return

        if not (lock and 'office_lock_id' in lock.get('data', {})):
            self.set_completed()
            return

        self.reenque(OFFICE_LOCK_POLL_INTERVAL)


class OfficeOverwrite(StoreDisk):
    """Операция для перезаписи файла Офисом.
    """

    type = 'office'
    subtype = 'overwrite'

    def __init__(self, *args, **kwargs):
        super(OfficeOverwrite, self).__init__(*args, **kwargs)
        self.keep_lock = True
        # Сохранить симлинки, чтобы значение `public_hash` сохранялось
        # после перегенерации.
        if self.data.get('set_public', False):
            self.keep_symlinks = True

    def _check_available_space(self, uid, path, size):
        try:
            Bus().check_available_space(uid, path, size)
        except NoFreeSpace:
            log.info('[office_overwrite] Not enough space: uid=%s, size=%s. Continue' % (uid, size))


class HancomOverwrite(OfficeOverwrite):
    type = 'hancom'
    subtype = 'overwrite'


class OnlyOfficeProcess(Operation):
    type = 'only_office'
    subtype = 'session'

    def get_uniq_id(self):
        return hashed(self.data['raw_resource_id'])


class OfficeConvertFile(StoreDiskMixin, SaveOnDiskOperation):
    type = 'office'
    subtype = 'convert'
    kladun = kladun_service.OfficeConvertAndUploadToDisk()

    errors_map = {
        codes.OFFICE_NO_FREE_SPACE_KLADUN_ERROR: codes.NO_FREE_SPACE,
        codes.OFFICE_SIZE_CONSTRAINT_FAILED_KLADUN_ERROR: codes.OFFICE_FILE_TOO_LARGE,
    }

    def get_status_dict(self):
        result = super(OfficeConvertFile, self).get_status_dict()
        if self.is_failed():
            code = self.data.get('error', {}).get('code', None)
            result_code = self.errors_map.get(code, code)

            if result_code is not None:
                if CODE in result[ERROR]:
                    result[ERROR][CODE] = result_code
                if MESSAGE in result[ERROR]:
                    result[ERROR][MESSAGE] = titles.get(result_code, '')
        return result

    @classmethod
    def Create(cls, uid, odata, **kw):
        odata_required_keys = {'id', 'service_id', 'service_file_id', 'service_file_size', 'service_file_ext',
                               'dst_address', 'app', 'action', 'connection_id'}
        missed_required_keys = odata_required_keys - odata.viewkeys()
        if missed_required_keys:
            raise KeyError('Missed required params: %s' % list(missed_required_keys))

        fs = Bus(connection_id=odata['connection_id'])
        service_id = odata['service_id']
        service_file_id = odata['service_file_id']
        service_file_size = odata['service_file_size']
        service_file_ext = odata['service_file_ext']
        dst_rawaddr = odata['dst_address'].id
        fs.check_target(dst_rawaddr)
        fs.check_available_space(uid, dst_rawaddr)
        free_space = Quota().free_with_shared_support(address=Address(dst_rawaddr).get_parent())
        file_id = Resource.generate_file_id(uid, dst_rawaddr)

        query_args = {
            'uid': uid,
            'oid': odata['id'],
            'file-id': file_id,
            'path': dst_rawaddr,
            'max-file-size': free_space
        }

        if service_id in ('disk', 'public'):
            query_args['source-service'] = 'mulca'
            query_args['service-file-id'] = service_file_id
        elif service_id == 'mail':
            query_args['source-service'] = 'mail2'
            query_args['service-file-id'] = service_file_id
        elif service_id == 'browser':
            query_args['source-service'] = service_id
            query_args['service-file-id'] = service_file_id
        elif service_id == 'web':
            query_args['source-service'] = service_id
            query_args['service-file-url'] = service_file_id
        else:
            raise OfficeStorageNotSupported()

        check_convert_size_limits(service_file_ext, service_file_size)

        int_status_url = cls.post_request_to_kladun(query_args)
        odata['status_url'] = int_status_url
        odata['state'] = EXECUTING
        odata['file_id'] = file_id
        odata['path'] = dst_rawaddr
        odata['free_space'] = free_space

        operation = super(OfficeConvertFile, cls).Create(uid, odata, **kw)
        return operation

    @classmethod
    def prepare_arguments(cls, uid, odata, **kw):
        cls.check_parent_rw(odata['path'])

    def _check_file_size(self):
        try:
            _, _, size = self._get_sha_size_md5()
            app, action = self.data['app'], self.data['action']
            check_office_online_size_limits(app, action, size)
        except OfficeFileTooLargeError:
            raise OfficeSizeConstraintFailedKladunError()

        if int(self.data['free_space']) < int(size):
            raise OfficeNoFreeSpaceKladunError()

    def handle_callback_1(self):
        # сначала надо проверить размер получившегося файла, если он больше ограничения, то фейлим операцию
        self._check_file_size()
        super(OfficeConvertFile, self).handle_callback_1()

    def handle_callback_2_and_3(self):
        # сначала надо проверить размер получившегося файла, если он больше ограничения, то фейлим операцию
        self._check_file_size()
        super(OfficeConvertFile, self).handle_callback_2_and_3()

    def set_completed(self):
        super(OfficeConvertFile, self).set_completed()
        connection_id = self.data['connection_id']
        key = self.data['dst_address']['path']
        OfficeFileConvertedEvent(uid=self.uid, connection_id=connection_id, key=key).send()


class CopyExternalToDisk(StoreDiskMixin, SaveOnDiskOperation):
    """Базовый класс для операций перекладки файла из разных внешних источников.

    Источники:
        * Публичная страница
        * Почтовый аттач

    """
    type = 'office'
    kladun = kladun_service.Capture()

    @staticmethod
    def _get_source_service():
        raise NotImplementedError()

    @classmethod
    def Create(cls, uid, odata, **kw):
        file_id = Resource.generate_file_id(uid, odata['target'])
        post_data = {
            'uid': uid,
            'file-id': file_id,
            'path': odata['target'],
            'source-service': cls._get_source_service(),
            'service-file-id': odata['service_file_id'],
            'free_space': odata['free_space'],
            'oid': odata['id'],
        }

        int_status_url = cls.post_request_to_kladun(post_data, **kw)
        odata['status_url'] = int_status_url
        odata['file_id'] = file_id
        odata['state'] = EXECUTING
        odata['path'] = odata['target']

        operation = super(CopyExternalToDisk, cls).Create(uid, odata, **kw)
        return operation

    def certain_infoupdate(self):
        self.data['filedata'].update({
            'size': self.downloaded_file.content_length,
            'mimetype': self.downloaded_file.mimetype,
        })
        self.data['filedata']['meta'].update({
            'md5': self.downloaded_file.md5,
            'sha256': self.downloaded_file.sha256,
            'file_id': self.data['file_id'],
        })
        self.data['path'] = self.data['target']

        if 'original_name' in self.data:
            self.data['filedata']['original_name'] = self.data['original_name']


class CopyMailAttachmentToDisk(CopyExternalToDisk):
    subtype = 'move_attach'

    @staticmethod
    def _get_source_service():
        return 'mail2'


class CopyPublicToDisk(CopyExternalToDisk):
    subtype = 'move_public'

    @staticmethod
    def _get_source_service():
        return 'mulca'


class CopyYabrowserFileToDisk(CopyExternalToDisk):
    subtype = 'move_browser'

    @staticmethod
    def _get_source_service():
        return 'browser'


class CopyOnlyOfficeFileToDisk(BaseExternalCopyWebDisk):
    subtype = 'only_office'
    keep_symlinks = True
    keep_lock = True

    def set_completed(self):
        session_id = self.data.get("session_id")
        if session_id and experiment_manager.is_feature_active('only_office_orchestrator'):
            orchestrator_service.delete_session(session_id)
        return super(CopyOnlyOfficeFileToDisk, self).set_completed()
