# -*- coding: utf-8 -*-
import base64
import os
import urllib2

from Crypto import Random

from mpfs.common.errors import DecryptionError, AddressError
from mpfs.common.util.crypt import CryptAgent
from mpfs.config import settings
from mpfs.core.address import sep, SharingURLAddress
from mpfs.core.office.static import EditorConst
from mpfs.core.signals import register_signal, add_timer
from mpfs.core.user.anonymous import AnonymousUID, AnonymousUser
from mpfs.core.user.constants import PUBLIC_UID
from mpfs.engine.process import Signal, get_error_log
from mpfs.core.services.discovery_service import DiscoveryService
from mpfs.core.services.passport_service import passport

from mpfs.core.office.errors import OfficeError, OfficeFileTooLargeError
from mpfs.core.office.models import OfficeAllowedPDDDomain


FEATURE_TOGGLES_HANCOM_EDITOR_ENABLED = settings.feature_toggles['hancom_editor_enabled']
FEATURE_TOGGLES_ONLYOFFICE_EDITOR_FOR_USERS_WITHOUT_EDITOR_ENABLED = settings.feature_toggles['onlyoffice_editor_for_users_without_editor_enabled']
OFFICE_DISCOVERY_CHECK_INTERVAL = settings.office['discovery_check_interval']
OFFICE_OPEN_DOCUMENT_BUTTON_URL = settings.office['open_document_button']['url']
OFFICE_RESOURCE_ID_SECRET_KEY = settings.office['resource_id_secret_key']
OFFICE_RESOURCE_ID_SECRET_KEY_OLD = settings.office['resource_id_secret_key_old']
RESOURCE_ID_SRC_SEPARATOR = ':'
OFFICE_SIZE_LIMITS = settings.office['size_limits']
OFFICE_CONVERT_SIZE_LIMITS = settings.office['convert_size_limits']
ONLY_OFFICE_FEATURE = 'only_office_enabled'
ONLINE_EDITOR_FEATURE = 'online_editor_enabled'


def check_office_online_size_limits(app, action, size, editor_type=None):
    """Проверить допустимый ли размер `size` для `action` и `ext` по ограничениям Office Online
    Нужно для проверки на редактирование.

    :type app: str
    :type action: str
    :type size: int
    :type editor_type: str
    """
    size_limits = OFFICE_SIZE_LIMITS
    if editor_type == EditorConst.ONLY_OFFICE:
        size_limits = settings.office['only_office']['size_limits']
    size = int(size)
    size_limit = size_limits.get(app, {}).get(action, None)
    if size_limit and size > size_limit:
        raise OfficeFileTooLargeError()


def check_convert_size_limits(ext, size):
    """Проверить допустимый ли размер `size` для `action` и `ext` по ограничениям DV на конвертацию.
    Нужно для проверки, отдавать ли ссылку на редактирование или нет, если мы имеем дело со старым бинарным форматом.

    :type ext: str
    :type size: int
    """
    size = int(size)
    size_limit = int(OFFICE_CONVERT_SIZE_LIMITS.get(ext, None))
    if size_limit and size > size_limit:
        raise OfficeFileTooLargeError()


def parse_resource_id(resource_id, secret_key=None, secret_key_old=None):
    """
    Расшифровать resource_id и вернуть owner_uid, file_id.

    owner_uid, file_id позволяют найти ресурс в базе даже если он расшарен.

    :type resource_id: str
    :rtype: tuple[str, str]
    """
    secret_key = secret_key or OFFICE_RESOURCE_ID_SECRET_KEY
    secret_key_old = secret_key_old or OFFICE_RESOURCE_ID_SECRET_KEY_OLD
    result = None

    if isinstance(resource_id, unicode):
        resource_id = str(resource_id)

    for key in filter(None, [secret_key, secret_key_old]):
        try:
            crypt_agent = CryptAgent(key=key)
            src = crypt_agent.decrypt(resource_id, urlsafe=True)
            owner_uid, file_id = src.split(RESOURCE_ID_SRC_SEPARATOR)
            result = (owner_uid, file_id)
        except (DecryptionError, ValueError):
            pass
    return result


def make_resource_id(owner_uid, file_id, secret_key=None):
    """
    Создать зашифрованный resource_id вида uid:owner_uid:file_id.

    :type owner_uid: str
    :type file_id: str
    :type secret_key: str
    :rtype: str
    """
    secret_key = secret_key or OFFICE_RESOURCE_ID_SECRET_KEY

    owner_uid, file_id = map(str, (owner_uid, file_id))
    return CryptAgent(key=secret_key).encrypt(RESOURCE_ID_SRC_SEPARATOR.join((owner_uid, file_id)), urlsafe=True)


class OfficeDocShortID(object):
    """Проверяет file_id на формат office_doc_short_id.

    В офисных методах передаются как обычные file_id, так и идентификаторы c office_doc_short_id:
    doc_short_id/<office_doc_short_id>
    """
    OFFICE_DOC_SHORT_ID_PREFIX = 'doc_short_id/'

    def __init__(self, value):
        self.office_doc_short_id = value

    @staticmethod
    def is_office_doc_short_id(value):
        return str(value).startswith(OfficeDocShortID.OFFICE_DOC_SHORT_ID_PREFIX)

    @classmethod
    def parse(cls, value):
        if cls.is_office_doc_short_id(value):
            return cls(value[len(cls.OFFICE_DOC_SHORT_ID_PREFIX):])

    def serialize(self):
        return self.OFFICE_DOC_SHORT_ID_PREFIX + self.office_doc_short_id


def make_session_context(post_message_origin='', locale=''):
    """
    :type post_message_origin: str
    :type locale: str
    :rtype: str
    """
    if not any((post_message_origin, locale)):
        return ''
    return ';'.join((post_message_origin, locale))


def parse_session_context(sc):
    """
    :type sc: str
    :rtype: tuple[str, str]
    """
    post_message_origin = ''
    locale = ''
    try:
        post_message_origin, locale = sc.split(';')
    except ValueError:
        pass
    return post_message_origin, locale


def was_editor_used(domain_or_domains):
    if not isinstance(domain_or_domains, (list, tuple)):
        domain_or_domains = [domain_or_domains]

    if not domain_or_domains:
        return False

    domain = OfficeAllowedPDDDomain.controller.get(domain={'$in': domain_or_domains})
    return domain is not None


def make_new_path(old_full_path, new_file_name):
    path_parts = old_full_path.split(sep)
    full_filename = path_parts[-1]

    file_ext = os.path.splitext(full_filename)[1]
    path_parts[-1] = ''.join([new_file_name, file_ext])
    new_path = sep.join(path_parts)

    return new_path


def build_office_online_url(client_id, document_id, tld):
    """Создать URL для доступа к файлу через Редактор.

    :type client_id: str | unicode
    :type document_id: str | unicode
    :type tld: str | unicode
    :rtype: unicode
    """
    if isinstance(document_id, unicode):
        document_id = document_id.encode('utf-8')
    document_id = urllib2.quote(document_id, safe='')

    url = OFFICE_OPEN_DOCUMENT_BUTTON_URL % {'tld': tld,
                                             'client_id': client_id,
                                             'document_id': document_id}
    return url


def setup_discovery():
    """Установить кеш и запустить его периодическую проверку.
    """

    register_signal(
        Signal.DISCOVERY,
        lambda signum: DiscoveryService().ensure_cache(),
    )
    add_timer(Signal.DISCOVERY, OFFICE_DISCOVERY_CHECK_INTERVAL)

    try:
        DiscoveryService(timeout=3.0, retry=False).ensure_cache()
    except OfficeError as e:
        get_error_log().error(e, exc_info=True)


def get_editor(user_or_uid, request=None, skip_force_oo=False):
    from mpfs.core.office.logic.microsoft import MicrosoftEditor
    from mpfs.core.office.logic.hancom import HancomEditor
    from mpfs.core.office.logic.only_office import OnlyOfficeEditor
    from mpfs.core import degradations
    from mpfs.core.user.common import CommonUser
    if OnlyOfficeEditor.is_user_allowed(user_or_uid, request, skip_force_oo):
        return OnlyOfficeEditor
    if FEATURE_TOGGLES_HANCOM_EDITOR_ENABLED and HancomEditor.is_user_allowed(user_or_uid, request):
        return HancomEditor
    if MicrosoftEditor.is_user_allowed(user_or_uid, request):
        return MicrosoftEditor

    if isinstance(user_or_uid, (CommonUser, AnonymousUser)):
        uid = user_or_uid.uid
    else:
        uid = user_or_uid

    if FEATURE_TOGGLES_ONLYOFFICE_EDITOR_FOR_USERS_WITHOUT_EDITOR_ENABLED and check_user_country_match(uid, 'RU') and \
       degradations.is_allow_only_office(user_or_uid):

        return OnlyOfficeEditor


def check_user_country_match(uid, country):
    if AnonymousUID.is_anonymous_uid(uid):
        # Анонимы без страны
        return True
    user_country = passport.userinfo(uid).get('country') or country
    return user_country.upper() == country


def generate_office_doc_short_id():
    # Длина случайной строки (безопасной для встраивания в ссылки) по умолчанию
    # 7, чтобы максимальная длина строки на выходе не превышала 10 символов
    # ceil(4 * RANDOM_STRING_LENGTH / 3) == 10
    RANDOM_STRING_LENGTH = 7
    return base64.urlsafe_b64encode(Random.new().read(RANDOM_STRING_LENGTH)).rstrip('=')


class SharingURLAddressHelper(object):
    """Вспомогательный класс для работы с внутренними данными office_online_sharing_url"""
    @staticmethod
    def build_sharing_url_addr(uid, office_doc_short_id):
        encrypted_uid = CryptAgent(key=OFFICE_RESOURCE_ID_SECRET_KEY).encrypt(uid,
                                                                              urlsafe=True,
                                                                              b64encoded=False)
        return SharingURLAddress(encrypted_uid=encrypted_uid, office_doc_short_id=office_doc_short_id)

    @staticmethod
    def decrypt_uid(encrypted_uid):
        try:
            uid = CryptAgent(key=OFFICE_RESOURCE_ID_SECRET_KEY).decrypt(str(encrypted_uid),
                                                                        urlsafe=True,
                                                                        b64encoded=False)
        except (DecryptionError, ValueError) as exc:
            raise AddressError('Failed to decrypt uid: %s' % exc.message)
        return uid
