# -*- coding: utf-8 -*-
"""

MPFS
CORE

Операция

"""
import hashlib
import time
import os
import sys
import traceback
import uuid

import pymongo
from datetime import datetime
from copy import deepcopy

from pymongo import ReadPreference

import mpfs.engine.process
import mpfs.common.errors as errors
import mpfs.common.static.codes as codes
import mpfs.common.static.messages as messages
from mpfs.common.util.filetypes import getGroupByName

from mpfs.config import settings
from mpfs.common.static.tags import *
from mpfs.common.util import to_json, is_jpg, logger, safe_to_int
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.core import factory
from mpfs.core.address import Address, ResourceId
from mpfs.core.bus import Bus
from mpfs.core.event_history.logger import CatchHistoryLogging
from mpfs.core.factory import get_resource, get_resource_by_file_id
from mpfs.core.filesystem import hardlinks
from mpfs.core.filesystem.cleaner.models import DeletedStid, DeletedStidSources
from mpfs.core.filesystem.dao.resource import ResourceDAO, PhotounlimDAO, get_dao_by_address
from mpfs.core.filesystem.hardlinks.common import construct_hid, FileChecksums
from mpfs.core.filesystem.live_photo import LivePhotoFilesManager
from mpfs.core.metastorage.control import operations, disk_info, additional_data
from mpfs.core.office.logic.sharing_url import sync_office_fields_from_link_data
from mpfs.core.operations.dao.operation import OperationDAO
from mpfs.core.operations.stage import get_stage, ExifInfo, Antivirus
from mpfs.core.operations.util import check_store_for_zero_file_size
from mpfs.core.photoslice.albums.data_structures import PhotosliceAlbumTypeDataStructure
from mpfs.core.photoslice.albums.logic import resolve_photoslice_album_type, update_file_data_with_photoslice_album_type
from mpfs.core.queue import mpfs_queue
from mpfs.core.services import kladun_service
from mpfs.core.social.publicator import Publicator
from mpfs.core.user.base import User
from mpfs.engine.http import client as http_client
from mpfs.metastorage.mongo.binary import Binary
from mpfs.core.user.constants import PUBLIC_UID, ADDITIONAL_AREA_PATH, HIDDEN_AREA, \
    PHOTOSTREAM_AREA, PHOTOUNLIM_AREA
from mpfs.core.filesystem.events import FilesystemHardlinkCopyEvent, FilesystemStoreEvent
from mpfs.core.filesystem.resources.disk import append_meta_to_office_files

SERVICES_MULCA_EMPTY_FILE_MID = settings.services['mulca']['empty_file_mid']
RESAVE_RESOURCE_WITHOUT_FILE_ID = settings.feature_toggles['resave_resource_without_file_id']
FEATURE_TOGGLES_SKIP_UPLOAD_ERRORS_FOR_REMOVED_RESOURCES = \
    settings.feature_toggles['skip_upload_errors_for_removed_resources']

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

paths = {}
for state in codes.operation_states:
    paths[state] = messages.old_operation_titles[state] + os.path.sep


def set_source_resource_id(source_resource, odata):
    """Установить `resource_id` исходного ресурса в данные операции.

    :type source_resource: :class:`mpfs.core.filesystem.resource.base.Resource`
    :type odata: dict
    :return:
    """

    resource_id = source_resource.resource_id
    if resource_id is None:
        if source_resource.address.is_system:
            file_id = source_resource.generate_file_id(source_resource.uid, source_resource.path)
            raw_resource_id = ResourceId(source_resource.storage_address.uid, file_id).serialize()
        else:
            if RESAVE_RESOURCE_WITHOUT_FILE_ID:
                file_id = source_resource.generate_file_id(source_resource.uid, source_resource.path)
                source_resource.meta['file_id'] = file_id
                source_resource.save()
                raw_resource_id = ResourceId(source_resource.storage_address.uid, file_id).serialize()

                odata['file_id'] = file_id
            else:
                raw_resource_id = ''

            log.info('[set_source_resource_id] Resource has no file_id: %s:%s',
                     source_resource.uid, source_resource.path)
    else:
        raw_resource_id = resource_id.serialize()

    odata['source_resource_id'] = raw_resource_id


class Operation(object):
    # название асинхронного таска, который будет обрабатывать операцию
    task_name = 'operation'
    type = None
    subtype = None

    def __init__(self, **data):
        """
            version - version in storage, in save will replace operation
            with this version
        """
        self.stages = {}
        self.xml_data = ''
        self.pended = self.is_pended()
        self.version = None
        self.event_log_catcher = None

        for key, value in self.dir().iteritems():
            if not hasattr(self, key):
                setattr(self, key, value)

        for key, value in data.iteritems():
            setattr(self, key, value)

        config_info = settings.operations['types'].get(self.type)
        if config_info:
            for key, value in config_info.iteritems():
                if key == 'pended':
                    pass
                else:
                    setattr(self, key, value)

        self.at_version = self.data.get(AT_VERSION, 0)
        self.uniq_id = self.get_uniq_id()

    @classmethod
    def is_pended(cls):
        try:
            _is_pended = cls.subtype in settings.operations['types'][cls.type]['pended']
        except (AttributeError, KeyError):
            _is_pended = False
        return _is_pended

    def dir(self):
        return {
            'uid':     None, # id владельца
            'id':      None, # id операции
            'state':   None, # статус операции
            'ctime':   None, # дата создания
            'mtime':   None, # дата изменения
            'type':    None, # тип производимого операцией действия
            'subtype':    None, # подтип действия
            'ycrid':   None, # ycrid операции
            'uniq_id': None, # id операции по типу и объектам
            'data':    {}, # dict данных операции
        }

    def dict(self):
        '''
        Получение данных операции в виде dict
        '''
        result = {}
        for attr in self.dir().keys():
            result[attr] = deepcopy(getattr(self, attr))
        return result

    def params(self):
        '''
        Получение параметров операции
        '''
        params_list = ('source', 'target', 'path')
        return dict((k, v) for (k, v) in self.data.iteritems() if k in params_list)

    def change_state(self, new_state, save=True):
        '''
        Изменение состояния операции
        '''
        if self.state == new_state:
            return

        title_from = messages.true_operation_titles[self.state]
        title_to = messages.true_operation_titles[new_state]

        log.info('CHANGING STATE FROM %s (%s) TO %s (%s)' % (self.state, title_from, new_state, title_to))
        if new_state not in codes.operation_states:
            raise errors.OperationWrongState(new_state)

        self.state = new_state

        if save:
            self.save()

    def save(self):
        '''
        Сохранение операции
        '''
        self.mtime = int(time.time())
        dct = deepcopy(self.dict())
        dct['data']['stages'] = {}
        dct['data']['filedata'] = {}
        del dct['id']
        #backward compatibility
        if '/' in self.key:
            self.key = self.key.rsplit('/', 1)[-1]
        ver = operations.put(self.uid, self.key, dct, version=self.version).version
        try:
            self.version = int(ver)
        except ValueError:
            self.version = ver

    @classmethod
    def Create(classname, uid, odata, **kw):
        """
        Создание операции
        """
        ctime = int(time.time())
        oid = odata.pop('id')
        state = odata.pop('state', codes.WAITING)
        ycrid = mpfs.engine.process.get_cloud_req_id()

        raw_op = {
            'uid': uid,
            'id': oid,
            'ctime': ctime,
            'mtime': ctime,
            'state': state,
            'type': classname.type,
            'subtype': classname.subtype,
            'ycrid': ycrid,
            'data': odata
        }

        operation = classname(key=oid, **raw_op)
        uniq_id = operation.uniq_id
        if uniq_id:
            additional_statuses = kw.pop('additional_statuses', [])
            # Try to get already existent operation
            query_args = {
                'uniq_id': uniq_id,
                'state': [codes.WAITING, codes.EXECUTING] + additional_statuses,
            }
            existent_ops = operations.find(uid, None, query_args, None, None, None)
            if existent_ops:
                raw_op = existent_ops[0]
                existent_op = classname(key=raw_op['id'], **raw_op)
                # Check for frozen operations according to https://jira.yandex-team.ru/browse/CHEMODAN-13839
                if existent_op.state == codes.EXECUTING and existent_op.mtime < int(time.time()) - settings.operations['ttl']:
                    # Make frozen operation failed and continue to create new operation.
                    existent_op.set_failed({'message': 'Operation frozen in EXECUTING state more then for %d sec' % settings.operations['ttl']})
                else:
                    # Return existent operation instead of creating new one.
                    existent_op.pended = False
                    return existent_op
            raw_op['uniq_id'] = uniq_id
        # Put new operation to collection.
        operation.check_arguments()
        operations.put(uid, oid, raw_op)
        return operation

    def remove(self):
        '''
        Удаление операции
        '''
        operations.remove(self.uid, self.key)

    def process(self, *args, **kwargs):
        try:
            # проверка юзера на возможность записи
            mpfs.engine.process.usrctl().assert_writable(self.uid)
            with CatchHistoryLogging(catch_messages=True) as catcher:
                self.set_event_log_catcher(catcher=catcher)
                self._process(*args, **kwargs)
        except Exception, e:
            # https://jira.yandex-team.ru/browse/CHEMODAN-14763
            # Ловим все ошибки стораджа (исключения монги перехватываются и преобразуются в исключения стораджа)
            # и если операция синхронная, то пробрасываем их наверх и не фэйлим операцию
            # в надежде, что монга очухается и при повторной попытке обработки операции, всё станет ОК.
            error = {'message': type(e).__name__}
            if isinstance(e, errors.MPFSRootException):
                if isinstance(e, (errors.StorageOperationError, errors.StorageConnectionError)) and not self.pended:
                    raise
                error['code'] = e.code
                error['response'] = e.response
                if e.title:
                    error['title'] = e.title
            else:
                error['code'] = codes.MPFS_ERROR
                error['response'] = codes.INTERNAL_ERROR

            # Всегда логируем stacktrace ошибки
            error_log.exception("Operation failed with: ")

            self.set_failed(error)
            self.data['error']['exception'] = sys.exc_info()

    def error(self):
        error = self.data.get('error', None)
        return error

    def get_status(self):
        result = {
            STATUS: self.state,
            TYPE: self.type,
            PARAMS: self.params(),
            AT_VERSION: self.data.get(AT_VERSION, 0),
        }

        if self.is_failed():
            result.update({ERROR: self.data.get('error', {})})
            result[ERROR][PARAMS] = self.params()
        else:
            if LENTA_BLOCK_ID in self.data:
                result[LENTA_BLOCK_ID] = self.data[LENTA_BLOCK_ID]
        return result

    def _process(self, *args, **kwargs):
        raise errors.MPFSNotImplemented()

    def __getitem__(self, key):
        return self.data.get(key, None)

    def __setitem__(self, key, val):
        self.data[key] = val

    def set_waiting(self):
        self.change_state(codes.WAITING)

    def set_executing(self):
        if not self.check_done():
            self.change_state(codes.EXECUTING)

    def set_rejected(self):
        self.change_state(codes.REJECTED)

    def set_failed(self, error):
        self.data['error'] = error
        try:
            self.change_state(codes.FAILED)
        except (errors.StorageOperationError, errors.StorageConnectionError):
            mpfs_queue.put(
                {
                    'uid': self.uid,
                    'oid': self.id,
                    'error': error
                },
                'operation_failing',
                delay=40
            )
            raise
        self.call_callback()

    def set_failed_queue(self, error):
        self.data['error'] = error
        self.change_state(codes.FAILED)
        self.call_callback()

    def set_done(self):
        from mpfs.core.operations.events import OperationBeforeDoneEvent  # circular import
        OperationBeforeDoneEvent(operation=self).send()
        self.change_state(codes.DONE)

    def set_completed(self):
        from mpfs.core.operations.events import OperationBeforeCompletedEvent  # circular import
        OperationBeforeCompletedEvent(operation=self).send()
        self.change_state(codes.COMPLETED)
        self.call_callback()

    def is_waiting(self):
        return self.state == codes.WAITING

    def is_executing(self):
        return self.state == codes.EXECUTING

    def is_completed(self):
        return self.state == codes.COMPLETED

    def is_done(self):
        return self.state == codes.DONE

    def is_failed(self):
        return self.state == codes.FAILED

    def is_rejected(self):
        return self.state == codes.REJECTED

    def check_done(self):
        if self.stages:
            for stage in self.stages.itervalues():
                if stage.main and stage.status != SUCCESS:
                    return False
            return True
        else:
            return False

    def check_main_stages_only(self):
        if self.stages:
            for stage in self.stages.itervalues():
                if stage.main:
                    if stage.status != SUCCESS:
                        return False
                # Проверка, что все неосновные стадии в initial, кроме стадии callback,
                elif stage.name != 'callback_stage':
                    if stage.status not in (INITIAL, TEMPORARY_FAILURE):
                        return False
            return True
        else:
            return False

    def check_completed(self):
        if self.stages:
            for stage in self.stages.itervalues():
                # Проверка с ExifInfo: https://st.yandex-team.ru/CHEMODAN-21199
                if (stage.status not in (SUCCESS, DISABLED, SKIPPED_SUCCESS) and
                        not (isinstance(stage, (ExifInfo, Antivirus)) and stage.status == SKIPPED_FAILURE)):
                    return False
            return True
        else:
            return False

    def check_failed(self):
        if self.data.get(OPERATION_STATUS) == XML_FAILED:
            return True

        if self.stages:
            return dict(filter(lambda (k, v): v.main and v.status == PERMANENT_FAILURE, self.stages.iteritems()))
        else:
            return False

    def check_in_progress(self):
        return self.data.get(OPERATION_STATUS) == XML_GLOBAL_PROCESSING and not self.check_done()

    def check_processing_finished(self):
        '''
            Основные стадии операции должны быть выполнены, остальные могут быть в состоянии постоянной ошибки.
            Используется для выставления прав файла для модификации
        '''
        return not filter(
            lambda stage: stage.main and stage.status != SUCCESS or \
                not stage.main and stage.status not in (SUCCESS, DISABLED, PERMANENT_FAILURE, SKIPPED_SUCCESS),
                self.stages.itervalues()
        )

    def calculate_space(self):
        return
        uid = self.uid
        if disk_info.has_domain(uid):
            try:
                last_check = int(disk_info.value(uid, 'check_time').value) or 0
            except Exception:
                last_check = 0
        else:
            disk_info.create_domain(uid)
        if time.time() - last_check > 24 * 60 * 60:
            try:
                new_size = get_resource(uid, Address('/disk/')).get_size()
                disk_info.put(uid, 'total_size', new_size)
                disk_info.put(uid, 'check_time', int(time.time()))
            except Exception:
                pass

    def call_callback(self):
        callback_url = self.data.get('callback')
        if callback_url:
            try:
                data = self.get_status()
                status_code = data[STATUS]
                data[STATUS] = messages.old_operation_titles[status_code]
                data[STATE] = messages.true_operation_titles[status_code]
                data = to_json(data)
                try:
                    http_client.open_url(
                        callback_url,
                        pure_data=data,
                        rais=True,
                        headers={'Content-Type':'application/json'}
                    )
                except Exception:
                    log.error('callback failed, it will be processed in queue')
                    mpfs_queue.put({'url' : callback_url, 'data' : data}, 'call_url')
            except Exception:
                log.error('wrong callback data')
                error_log.error(traceback.format_exc())

    def get_uniq_id(self):
        return None

    @classmethod
    def prepare_arguments(cls, uid, odata, **kw):
        '''
            Проверка и дополнение аргументов до создания объекта Operation.
        '''
        pass

    def check_arguments(self):
        '''
            Проверка аргументов перед созданием операции в базе.
        '''
        pass

    def reenque(self, delay):
        """
        Ставим job operation с текущей операцией в очередь
        :param delay: int
        """
        from mpfs.core.operations import manager
        next_start = int(time.time() + delay)
        self.update_dtime()
        manager.enqueue_or_fail_operation(self, stime=next_start)
        log.info("reenque operation %s:%s for task %s" % (self.uid, self.id, self.task_name))

    def process_interruption(self):
        """
        Функция вызывается при внезапной остановке воркера, выполняющего операцию (процесс получил SIGTERM и должен
        освободить ресурсы/снять локи).
        """
        pass

    def set_event_log_catcher(self, catcher):
        """Установить кетчера, записывающего все сообщения, которые попадают в event лог."""
        self.event_log_catcher = catcher

    def update_dtime(self):
        """Обновить dtime операции на текущее время"""
        self.dtime = datetime.utcnow()
        OperationDAO().set_dtime(self.uid, self.id, self.dtime)


class UploadOperation(Operation):

    kladun = kladun_service.Kladun()

    def update_status(self, xml_data=None):
        #=======================================================================
        # status_url - адрес, по которому запрашивается статус загрузки
        # если его нет, то получить инфо о загрузке не получится
        if not 'status_url' in self.data:
            return
        #=======================================================================
        self.status_from_kladun(xml_data)
        self.parse_status()
        if self.check_completed():
            self.set_completed()
        elif self.check_done():
            self.set_done()
        if self.check_failed():
            error = {
                'message': XML_STORE_STATUS,
                'code': 0,
            }
            self.set_failed(error)

    def _process(self, xml_data, *args, **kwargs):
        self.update_status(xml_data)

    def get_status(self):
        if not experiment_manager.is_feature_active('remove_stages'):
            try:
                self.status_from_kladun()
                self.parse_status()
            except errors.OperationNotFoundError:
                pass
        return self.get_status_dict()

    def get_status_dict(self):
        stages = []
        for v in self.stages.itervalues():
            stages.append(v.dict())
        result = {
            STATUS : self.state,
            TYPE: self.type,
            STAGES: stages,
            PARAMS: self.params(),
            AT_VERSION: self.data.get(AT_VERSION, 0)
        }
        if self.is_failed():
            result[ERROR] = self.data[ERROR]
            result[ERROR][PARAMS] = self.params()

        if LENTA_BLOCK_ID in self.data:
            result[LENTA_BLOCK_ID] = self.data[LENTA_BLOCK_ID]

        return result

    def check_data_uploaded(self):
        return self.check_done()

    def parse_status(self):
        for stage_tag, stage_value in self.data.get(STAGES, {}).iteritems():
            stage_object = get_stage(stage_tag, stage_value ,self.id)
            self.stages[stage_object.name] = stage_object
            setattr(self, stage_object.name, stage_object)

    def status_from_kladun(self, xml_data=''):
        element = self.kladun.status(self['status_url'], xml_data)
        try:
            kladun_stages = element.find(STAGES)
            prev = None
            self.data[STAGES] = self.data.get(STAGES, {})
            for stage in kladun_stages.getchildren():
                self.data[STAGES][stage.tag] = self.data[STAGES].get(stage.tag, {})
                stage_status = stage.get(STATUS)
                not_checked = self.data[STAGES][stage.tag].get(STATUS) != 'success' and stage_status == 'success'
                just_processed = 'processing_time' not in self.data[STAGES][stage.tag] and 'processing_time_diff' not in self.data[STAGES][stage.tag]
                if not_checked and just_processed:
                    if prev and 'processing_time' in self.data[STAGES].get(prev, {}):
                        self.data[STAGES][stage.tag]['processing_time'] = time.time()
                        self.data[STAGES][stage.tag]['processing_time_diff'] = time.time() - self.data[STAGES].get(prev, {}).get('processing_time')
                    else:
                        self.data[STAGES][stage.tag]['processing_time'] = time.time()
                elif stage_status == 'success' and  not self.data[STAGES][stage.tag]:
                    self.data[STAGES][stage.tag]['processing_time'] = time.time()
                stage_progress = stage.find(PROGRESS)
                if stage_progress is not None:
                    self.data[STAGES][stage.tag][PROGRESS] = dict(stage_progress.items())
                else:
                    self.data[STAGES][stage.tag][PROGRESS] = {}
                stage_result = stage.find(RESULT)
                if stage_result is not None:
                    self.data[STAGES][stage.tag][RESULT] = stage_result
                else:
                    self.data[STAGES][stage.tag][RESULT] = {}
                self.data[STAGES][stage.tag][DETAILS] = stage.find(DETAILS)
                self.data[STAGES][stage.tag][STATUS] = stage_status
                prev = stage.tag
            self.data[OPERATION_STATUS] = element.get(STATUS)
        except (KeyError, AttributeError):
            error_log.error(traceback.format_exc())
            raise errors.KladunStatusFailure()

    def set_first_user_file_upload(self):
        usr = User(self.uid)
        usr.set_state(
                key = 'first_file',
                value=1
            )
        if self.data.get('client_type') == 'desktop':
            usr.set_state(
                key = 'file_uploaded',
                value=1
            )

    @classmethod
    def post_request_to_kladun(cls, d, **kw):
        try:
            use_https = kw['use_https']
        except KeyError:
            pass
        else:
            if use_https is not None:
                if use_https:
                    d['use-https'] = 'true'
                else:
                    d['use-https'] = 'false'
        return cls.kladun.post_request(d)

    def raise_kladun_conflict(self):
        pass


class SaveOnDiskOperation(UploadOperation):

    type = None
    fail_on_save_or_update_file = False
    kladun = kladun_service.UploadToDisk()
    keep_symlinks = False
    keep_lock = False

    hardlink_copy_exceptions = {
        errors.HardlinkFileNotInStorage.__name__: errors.HardlinkFileNotInStorage,
        errors.HardLinkNotFound.__name__: errors.HardLinkNotFound,
        errors.HardLinkNotReady.__name__: errors.HardLinkNotReady,
        errors.StidLockedError.__name__: errors.StidLockedError,
        errors.HardlinkBroken.__name__: errors.HardlinkBroken,
    }

    def __init__(self, *args, **kwargs):
        self.internal_data = {'save_state_to_db': True}
        super(SaveOnDiskOperation, self).__init__(*args, **kwargs)

    def change_state(self, new_state, **kwargs):
        super(SaveOnDiskOperation, self).change_state(new_state, save=self.internal_data['save_state_to_db'])

    def _process(self, xml_data, stage):
        """
        Обработка осуществляется по следующему алгоритму:
        если это первый коллбек, то пытаемся сделать хардлинк
        если хардлинк не найден - работаем как с новым файлом

        если это не первый коллбек, сохраняем файл
        """
        self.internal_data['save_state_to_db'] = False
        super(SaveOnDiskOperation, self)._process(xml_data)
        self.internal_data['save_state_to_db'] = True

        if not self.is_failed() and not self.is_rejected():
            if self.is_live_photo_store(self.data) and 'live_photo_original_path' not in self.data:
                # подменяем пути для загрузки live photo и видео части - после заргузки видео мы переместим их обратно
                self.data['live_photo_original_path'] = self.get_path()
                path = Address.Make(self.uid, '/hidden/' + uuid.uuid4().hex).id
                self.set_path(path)

            if self.is_kladun_first_callback(stage):
                self.handle_callback_1()
            else:
                self.handle_callback_2_and_3()

        self.save()

    @staticmethod
    def is_kladun_first_callback(stage):
        return stage == COMMIT_FILE_INFO

    def _check_available_space(self, uid, path, size):
        """Проверить доступность места на Диске.

        Это базовое поведение. В некоторых операциях необходимо
        сохранять файл при нехватке места.

        :raises: :class:`~mpfs.common.errors.NoFreeSpace`
        """
        if not self.data.get('skip_check_space', False):
            Bus().check_available_space_on_store_by_raw_address(uid, path, size)

    def check_hids_blockings(self):
        fs = Bus()
        md5, sha256, size = self._get_sha_size_md5()
        hid = Binary(hardlinks.common.construct_hid(md5, size, sha256))
        fs.check_hids_blockings([hid, ])

    def _process_found_hid_on_store_step(self):
        """
        Рерайзит исключения, полученные на этапе проверки хардлинка, если клиент корректные правильные хэши
        :return:
        """
        if self.data.get('hardlink_status') is None:
            return

        try:
            client_checksums = FileChecksums(self.data['md5'], self.data['sha256'], long(self.data['size']))
            kladun_md5, kladun_sha256, kladun_size = self._get_sha_size_md5()
            kladun_checksums = FileChecksums(kladun_md5, kladun_sha256, long(kladun_size))
        except ValueError:
            return
        else:
            if client_checksums != kladun_checksums:
                return

        exception_name, exception_message, exception_data = self.data['hardlink_status']
        if exception_name in self.hardlink_copy_exceptions:
            raise self.hardlink_copy_exceptions[exception_name](message=exception_message, data=exception_data)
        else:
            log.info('unknown exception %s. skip' % exception_name)

    def _try_hardlink_copy(self, action='uploading hardlinked file'):
        try:
            self.hardlink_copy()
            log.info(action)
            return True
        except errors.HardlinkBroken:
            self.data_init_features['hardlink_broken'] = 1
        except errors.HardlinkFileNotInStorage as e:
            self.data['update_mids'] = True
            self.data['broken_mid'] = e.data['mid']
            log.info('reloading file and update file_mid')
        except errors.HardLinkNotFound:
            pass
        except errors.EmptyFileUploadedForNonEmpyStoreError:
            raise
        except Exception:
            error_log.exception("hardlink_copy failed: uid=%s oid=%s" % (self.uid, self.id))
        return False

    def handle_callback_1(self):
        self._check_available_space(self.uid, self.get_path(), self.data['filedata']['size'])
        self.check_hids_blockings()
        self.data['update_mids'] = False
        if self._try_hardlink_copy():
            return
        log.info('uploading simple file')

    def handle_callback_2_and_3(self):
        self.save_or_update_file()

    def handle_callback_3(self, resource, file_data, changes=[], notify_search=True):
        filesystem_module = Bus()
        if settings.feature_toggles['image_tags'] and is_jpg(resource.mimetype):
            try:
                etime = file_data['meta'].get('etime')
                emid, _ = filesystem_module.tags_set_photo_file(resource.uid, resource.address.id,
                                                                etime, resource.dict())
                if emid:
                    file_data['meta']['emid'] = emid
            except Exception:
                error_log.error(traceback.format_exc())

        if 'etime_from_client' in self.data and 'etime' in resource.meta:
            # если в ресурсе проставлен etime, переданный с клиента, игнорируем то, что передал кладун
            file_data.get('meta', {}).pop('etime', None)

        photoslice_album_type_data = PhotosliceAlbumTypeDataStructure.cons_of_dict(
            self.data.get('photoslice_album_type_data'))
        update_file_data_with_photoslice_album_type(self.uid, file_data, photoslice_album_type_data)

        filesystem_module.patch_file(
            self.uid,
            resource.address.id,
            file_data,
            changes,
            force=True,
            notify_search=notify_search,
            is_store=True,
        )

    def set_path(self, path):
        self.data['path'] = path

    def get_path(self):
        return self.data['path']

    def parse_status(self):
        super(SaveOnDiskOperation, self).parse_status()
        self.update_fileinfo()

    def certain_infoupdate(self):
        pass

    def _get_sha_size_md5(self):
        file_data = self.data['filedata']
        md5    = file_data['meta']['md5']
        sha256 = file_data['meta']['sha256']
        size   = file_data['size']
        return md5, sha256, size

    def hardlink_copy(self):
        # данные из Uploader'а
        md5, sha256, size = self._get_sha_size_md5()

        check_store_for_zero_file_size(operation_type=self.type,
                                       size_from_uploader=size,
                                       size_from_client=self.data.get('size'))

        self._process_found_hid_on_store_step()


        bus = Bus(connection_id=self.data['connection_id'])

        file_data = {'meta': {}}
        file_data['meta']['file_id'] = self.data['file_id']

        for k, v in self.data.get('changes', {}).iteritems():
            file_data[k] = v
        for k, v in self.additional_filedata().iteritems():
            file_data[k] = v

        if 'original_name' in self.data:
            file_data['original_name'] = self.data['original_name']

        notify_search = True
        if self.is_live_photo_store(self.data):
            notify_search = False

        resource = bus.hardlink_copy(
            self.uid,
            self.get_path(),
            md5, size, sha256,
            force=True,
            keep_symlinks=self.keep_symlinks,
            filedata=file_data,
            replace_md5=self.data.get('replace_md5'),
            keep_lock=self.keep_lock,
            oper_type=self.type,
            oper_subtype=self.subtype,
            suppress_event=True,
            etime_from_client=self.data.get('etime_from_client'),
            notify_search=notify_search,
            source_id=self.data.get('source_id_to_add'),
            is_live_photo=self.is_live_photo_store(file_data),
            photoslice_album_type_data=PhotosliceAlbumTypeDataStructure.cons_of_dict(
                self.data.get('photoslice_album_type_data')),
        )
        self.data['resource_id'] = resource.resource_id.serialize()
        is_set_public = False
        if self.data.get('set_public'):
            try:
                Publicator().set_public(self.uid, self.get_path(), resource=resource,
                                        oper_type=self.type, oper_subtype=self.subtype)
                is_set_public = True
            except Exception:
                error_log.error('failed to publish %s:%s' % (self.uid, self.get_path()))
                error_log.error(traceback.format_exc())

        sync_office_fields_from_link_data(resource.owner_uid, resource.meta['file_id'])

        FilesystemHardlinkCopyEvent(tgt_resource=resource, tgt_address=resource.visible_address,
                                    type=self.type, subtype=self.subtype,
                                    changes=self.data.get('changes', {}), set_public=is_set_public,
                                    uid=resource.visible_address.uid, size=safe_to_int(size),
                                    ).send_self_or_group(resource=resource)

        if self.is_live_photo_store(self.data) and self.data.get('live_photo_type', None) == 'video':
            self.process_live_video_store(self.uid, bus)

        self.data['hardlinked'] = True
        self.post_process_new_file(resource)
        self.post_process_all()
        self.set_completed()

    def additional_filedata(self):
        return {}

    def update_fileinfo(self):
        try:
            self.data['filedata'] = {
                'meta': {
                    'file_mid'    : self.mulca_file_upload.mulca_id,
                    'digest_mid'  : self.mulca_digest_upload.mulca_id,
                    'drweb'       : self.antivirus_check.drweb,
                    'file_id'     : self.data['file_id']
                },
            }

            try:
                etime = self.exif_info.etime
            except AttributeError:
                etime = None

            try:
                vctime = self.media_info.etime
            except AttributeError:
                vctime = None

            if etime or vctime:
                self.data['filedata']['meta']['etime'] = etime or vctime

            try:
                preview_image = self.preview_image
            except AttributeError:
                pass
            else:
                self.data['filedata']['meta'].update({'previews': preview_image.preview_links})

            try:
                preview_sizes = self.preview_sizes
            except AttributeError:
                pass
            else:
                if preview_sizes.preview_sizes:
                    self.data['filedata']['meta'].update({'preview_sizes': preview_sizes.preview_sizes})

            try:
                self.data['filedata']['meta']['pmid'] = self.single_preview_mid.mulca_id
            except AttributeError:
                try:
                    self.data['filedata']['meta']['pmid'] = self.single_preview_video_mid.mulca_id
                except AttributeError:
                    pass

            try:
                video_info = self.video_info.content
            except AttributeError:
                pass
            else:
                if video_info:
                    self.data['filedata']['meta']['video_info'] = video_info

            for k,v in self.additional_filedata().iteritems():
                self.data['filedata']['meta'][k] = v

            try:
                document_preview = self.preview
            except AttributeError:
                pass
            else:
                mid = document_preview.mulca_id
                if mid:
                    self.data['filedata']['meta']['pmid'] = mid

            try:
                width = self.single_preview_info.original_width
                height = self.single_preview_info.original_height
            except AttributeError:
                pass
            else:
                self.data['filedata']['meta']['width'] = width
                self.data['filedata']['meta']['height'] = height

            try:
                self.data['filedata']['meta']['angle'] = self.single_preview_info.rotate_angle
            except AttributeError:
                pass

            self.certain_infoupdate()
        except Exception:
            error_log.info(traceback.format_exc())

    @staticmethod
    def is_live_photo_store(data):
        return all(i in data for i in ('live_photo_md5', 'live_photo_sha256', 'live_photo_size', 'live_photo_type'))

    def save_or_update_file(self):
        try:
            self.data['filedata']['meta']['modify_uid'] = self.uid
        except (KeyError, TypeError):
            pass
        file_data = self.data['filedata']
        uid = self.uid
        path = self.get_path()

        prev_read_pref = mpfs.engine.process.get_read_preference()
        mpfs.engine.process.set_read_preference(ReadPreference.PRIMARY_PREFERRED)
        try:
            resource = get_resource(uid, self.get_path())
        except errors.ResourceNotFound:
            try:
                resource = get_resource_by_file_id(self.uid, self.data['file_id'])
            except errors.ResourceNotFound:
                resource = None
            else:
                if resource.address.storage_name == HIDDEN_AREA:
                    resource = None
                else:
                    path = resource.address.id
        finally:
            mpfs.engine.process.set_read_preference(prev_read_pref)

        if resource:
            is_new_file = resource.meta.get('file_mid') != file_data['meta'].get('file_mid')
        else:
            is_new_file = True

        if self.check_main_stages_only() and (not resource or is_new_file):
            file_data.update(self.data.get('changes', {}))

            bus = Bus(connection_id=self.data['connection_id'])
            self._check_available_space(uid, path, file_data['size'])

            if 'etime_from_client' in self.data:
                # Сохраняем клиентский etime только для картинок и видео
                # https://st.yandex-team.ru/CHEMODAN-41657
                if getGroupByName(path, mtype=file_data['mimetype']) in ('image', 'video',):
                    file_data['etime'] = self.data['etime_from_client']

            photoslice_album_type_data = PhotosliceAlbumTypeDataStructure.cons_of_dict(
                self.data.get('photoslice_album_type_data'))
            storage_name = Address(path).storage_name
            autouploaded = (storage_name == PHOTOSTREAM_AREA or
                            storage_name == PHOTOUNLIM_AREA)
            file_data['meta']['photoslice_album_type'] = resolve_photoslice_album_type(
                uid, file_data['mimetype'], photoslice_album_type_data, autouploaded=autouploaded)

            new_resource = bus.mkfile(
                uid,
                path,
                data=file_data,
                keep_symlinks=self.keep_symlinks,
                replace_md5=self.data.get('replace_md5'),
                notify_search=False,
                keep_lock=self.keep_lock
            )
            if not self.is_live_photo_store(self.data) and not new_resource.is_shared:
                self._add_source_id_by_operation_data()

            sync_office_fields_from_link_data(new_resource.owner_uid, new_resource.meta['file_id'])

            self.data['resource_id'] = new_resource.resource_id.serialize()

            if self.data.get('set_public'):
                try:
                    Publicator().set_public(uid, path,
                                            oper_type=self.type, oper_subtype=self.subtype,
                                            user_ip=self.data.get('user_ip', ''))
                except Exception:
                    error_log.error('failed to publish %s:%s' % (uid, path))
                    error_log.error(traceback.format_exc())

            if resource and self.data['file_id'] != resource.meta.get('file_id'):
                self.data['file_id'] = resource.meta.get('file_id')
                self.save()

            if self.is_live_photo_store(self.data) and self.data.get('live_photo_type', None) == 'video':
                self.process_live_video_store(uid, bus)

            self.post_process_new_file(new_resource)
            self.call_callback()

        elif resource and not is_new_file:
            notify_search = True
            if self.is_live_photo_store(self.data):
                notify_search = False

            self.handle_callback_3(resource, file_data, notify_search=notify_search)

            sync_office_fields_from_link_data(resource.owner_uid, resource.meta['file_id'])

            self.post_process_all()
            if self.data.get('hardlink_broken'):
                mpfs_queue.put({'data': file_data['meta']}, 'update_hardlink')

            # https://st.yandex-team.ru/CHEMODAN-18963
            # Обновляем все возможные mids для этого hid-а
            if self.data.get('update_mids'):
                hid = getattr(resource, 'hid', None)
                if hid:
                    odata = {'hid': str(hid)}
                    try:
                        odata['file_mid'] = self.mulca_file_upload.mulca_id
                    except AttributeError:
                        try:
                            odata['file_mid'] = self.data['filedata']['file_mid']
                        except KeyError:
                            pass
                    try:
                        odata['digest_mid'] = self.mulca_digest_upload.mulca_id
                    except AttributeError:
                        try:
                            odata['digest_mid'] = self.data['filedata']['digest_mid']
                        except KeyError:
                            pass
                    try:
                        odata['pmid'] = self.single_preview_mid.mulca_id
                    except AttributeError:
                        try:
                            odata['pmid'] = self.single_preview_video_mid.mulca_id
                        except AttributeError:
                            pass
        elif resource and is_new_file:
            # 409 только если файл есть и то что нам загружают отличается от того, что есть в Диске.
            # CHEMODAN-16358
            # этот чудо-метод на самом деле швыряет исключение только в DiskUploadOperation
            self.raise_kladun_conflict()
        else:
            # При 3тьем callbacke'е от Kladun'а resource == None - ресурс был удален между 2рым callback'ом и 3тьим
            # если можно,
            if self.fail_on_save_or_update_file:
                if FEATURE_TOGGLES_SKIP_UPLOAD_ERRORS_FOR_REMOVED_RESOURCES:
                    try:
                        stid = file_data['meta']['pmid']
                    except KeyError:
                        stid = None
                    if stid:
                        unused_preview_stid = DeletedStid(stid=stid, stid_source=DeletedStidSources.KLADUN_CLEAN_UP)
                        DeletedStid.controller.bulk_create([unused_preview_stid], get_size_from_storage=True)
                    log.info('Resource was removed after 2nd Kladun callback - no exception will be raised')
                else:
                    # то в любой непонятной ситуации -- 503
                    raise errors.KladunUnexpectedCallbackState()

    def process_live_video_store(self, uid, bus):
        if self.data.get('live_photo_operation_id') is None:
            photo_store_operation_oid = LivePhotoFilesManager.get_paired_live_photo_store_operation_id(
                uid, self.data['live_photo_operation_path'],
                self.data['md5'], self.data['sha256'], self.data['size'],
                self.data['live_photo_md5'], self.data['live_photo_sha256'], self.data['live_photo_size']
            )
        else:
            photo_store_operation_oid = self.data['live_photo_operation_id']

        video_src_rawaddress = self.get_path()
        video_dst_rawaddress = Address.Make(uid, ADDITIONAL_AREA_PATH + '/' + uuid.uuid4().hex).id

        try:
            from mpfs.core.operations import manager
            photo_store_operation = manager.get_operation(uid, photo_store_operation_oid)
        except errors.OperationNotFound:
            photo_store_operation = None

        resource = None
        if photo_store_operation:
            self.check_resource_exists(uid, photo_store_operation.get_path())

            photo_src_rawaddress = photo_store_operation.get_path()
            photo_dst_rawaddress = self.data['live_photo_original_path']

            try:
                bus.move_resource(uid, photo_src_rawaddress, photo_dst_rawaddress,
                                  force=False, lock=True, photounlim_allowed=True, is_live_photo=True)
            except errors.CopyTargetExists:
                photo_dst_rawaddress = Address(photo_dst_rawaddress).add_suffix('_' + str(int(time.time() * 1000))).id
                bus.move_resource(uid, photo_src_rawaddress, photo_dst_rawaddress,
                                  force=False, lock=True, photounlim_allowed=True, is_live_photo=True)

            photo_store_operation.set_path(photo_dst_rawaddress)
            photo_store_operation.save()

            resource = bus.resource(uid, photo_dst_rawaddress)
            event = FilesystemStoreEvent(
                uid=uid, tgt_resource=resource, tgt_address=Address(photo_dst_rawaddress),
                type=self.type, subtype=self.subtype,
                set_public=self.data.get('set_public', False),
                changes=self.data.get('changes', {}),
                is_live_photo=True,
                size=safe_to_int(self.data['size']),
            )
            event.send_self_or_group(resource=resource)
        else:
            # заливаем видео без фото (надо поискать фото в диске пользователя по hid'у)
            photo_md5 = self.data['live_photo_md5']
            photo_sha256 = self.data['live_photo_sha256']
            photo_size = self.data['live_photo_size']

            try:
                live_photo = self.get_live_photo_file(uid, photo_md5, photo_sha256, photo_size)
                photo_dst_rawaddress = Address.Make(uid, live_photo.path).id
            except errors.LivePhotoMultipleFound:
                log.info(
                    'Multiple photos found for live video with path `%s` '
                    '(photo_sha256: %s, photo_md5: %s, photo_size: %s, '
                    'video_sha256: %s, video_md5: %s, video_size: %s)',
                    video_dst_rawaddress, photo_sha256, photo_md5, photo_size,
                    self.data['sha256'], self.data['md5'], self.data['size']
                )
                photo_dst_rawaddress = None
        try:
            bus.move_resource(uid, video_src_rawaddress, video_dst_rawaddress,
                              force=False, lock=True, notify_search=False, is_live_photo=True)
        except (errors.CopyParentNotFound, errors.StorageKeyNotFound):
            try:
                additional_data.create(uid)
                additional_data.make_folder(uid, ADDITIONAL_AREA_PATH, {})
            except (errors.StorageDomainAlreadyExists, errors.StorageFolderAlreadyExist):
                pass
            bus.move_resource(uid, video_src_rawaddress, video_dst_rawaddress,
                              force=False, lock=True, notify_search=False, is_live_photo=True)

        self.set_path(video_dst_rawaddress)
        self.save()

        if photo_dst_rawaddress is not None:
            self.link_live_photo_files(uid, Address(photo_dst_rawaddress).path, Address(video_dst_rawaddress).path)
            resource = resource or bus.resource(uid, photo_dst_rawaddress)
            if not resource.is_shared:
                self._set_live_photo_flag_to_source_ids_and_add_one()

    def check_resource_exists(self, uid, path):
        get_resource(uid, path)  # нужно, чтобы зарайзить исключение, если ресурса нет

    def link_live_photo_files(self, uid, photo_path, video_path):
        dao = get_dao_by_address(Address.Make(uid, photo_path))
        dao.link_live_photo_files(uid, photo_path, video_path)

    @classmethod
    def get_live_photo_file(cls, uid, photo_md5, photo_sha256, photo_size):
        photo_hid = construct_hid(photo_md5, photo_size, photo_sha256)

        disk_files_count = ResourceDAO().count_live_photo_files(uid, photo_hid)
        photounlim_files_count = PhotounlimDAO().count_live_photo_files(uid, photo_hid)
        files_count = disk_files_count + photounlim_files_count

        if files_count == 0:
            raise errors.LivePhotoNotFound()
        elif files_count > 1:
            raise errors.LivePhotoMultipleFound()

        if disk_files_count:
            return ResourceDAO().get_live_photo_file(uid, photo_hid)
        elif photounlim_files_count:
            return PhotounlimDAO().get_live_photo_file(uid, photo_hid)

    def _add_source_id_by_operation_data(self):
        source_id = self.data.get('source_id_to_add')
        if source_id is None:
            return
        task_doc = {
            'uid': self.uid,
            'source_ids': [source_id, ],
        }
        if self.is_live_photo_store(self.data):
            task_doc.update({
                'hid': FileChecksums(
                    self.data['live_photo_md5'], self.data['live_photo_sha256'], int(self.data['live_photo_size'])).hid,
                'is_live_photo': True,
            })
        else:
            task_doc.update({
                'hid': FileChecksums(self.data['md5'], self.data['sha256'], int(self.data['size'])).hid,
                'is_live_photo': False,
            })
        mpfs_queue.put(task_doc, 'add_source_ids')

    def _set_live_photo_flag_to_source_ids_and_add_one(self):
        source_id = self.data.get('source_id_to_add')
        source_ids = []
        if source_id is not None:
            source_ids.append(source_id)
        task_doc = {
            'uid': self.uid,
            'source_ids': source_ids,
            'hid': FileChecksums(self.data['live_photo_md5'], self.data['live_photo_sha256'], int(self.data['live_photo_size'])).hid,
        }
        mpfs_queue.put(task_doc, 'set_live_photo_flag_to_source_ids_and_add_one')

    def post_process_all(self):
        self.set_first_user_file_upload()

        if self.data.get('set_public'):
            uid, path = self.uid, self.get_path()
            resource = factory.get_resource(uid, path)
            if resource.is_infected():
                Publicator().set_private(uid, path)
        self.stat_log()

    def get_status(self):
        result = super(SaveOnDiskOperation, self).get_status()
        if hasattr(self, 'mpfs_request') and (self.is_done() or self.is_completed()):
            try:
                res = factory.get_resource(self.uid, self.get_path())
                append_meta_to_office_files(res, self.mpfs_request)
                self.mpfs_request.form = res.form
            except Exception:
                pass
        return result

    def stat_log(self):
        try:
            resource = Bus().resource(self.uid, self.get_path())

            logger.log_stat(self.uid, self.type, self.subtype,
                            int(self.data.get('hardlinked', False)),
                            resource.get_size(), resource.media_type, Address(self.get_path()).path)
        except Exception:
            error_log.error('failed to write stat_log')
            error_log.error(traceback.format_exc())

    def post_process_new_file(self, new_resource):
        """
        Бизнес-логика, выполняется после полной загрузки нового файла
        """
        pass


class DiskUploadOperation(SaveOnDiskOperation):
    fail_on_save_or_update_file = True

    def raise_kladun_conflict(self):
        raise errors.KladunConflict()


class ServiceUploadOperation(UploadOperation):

    kladun = kladun_service.UploadToService()


class DiskInternalOperation(Operation):

    @classmethod
    def Create(classname, uid, odata, **kw):
        classname.prepare_arguments(uid, odata, **kw)
        return super(DiskInternalOperation, classname).Create(uid, odata, **kw)

    def check_arguments(self):
        fs = Bus()
        path = self.data.get('path') or self.data.get('source')
        if path:
            fs.check_lock(path)
            resource_uid = self.data.get('src_uid', self.uid)  # нужно для перекладывания данных другого пользователя
            fs.info(resource_uid, path)

    def get_status(self):
        result = Operation.get_status(self)
        affected_resource = self.data.get('affected_resource')
        if affected_resource is not None and hasattr(self, 'mpfs_request'):
            try:
                '''
                    Неявно заполняется self.mpfs_request
                '''
                Bus(request=self.mpfs_request).info(self.uid, affected_resource)
            except errors.NotFound:
                pass
        return result

    def release_lock(self, address):
        fs = Bus()
        lock = fs.get_lock(address)
        if lock is None:
            return

        lock_oid = lock.get('data', {}).get('oid')

        if self.id == lock_oid:
            log.info('unset lock on interruption by operation: %s (path: %s)' % (self.id, address))
            fs.unset_lock(address)
        else:
            log.info('do not unset lock on interruption: oid - %s, lock oid - %s, path - %s' % (
                self.id, str(lock_oid), address))


class CopyMove(DiskInternalOperation):

    @classmethod
    def Create(classname, uid, odata, **kw):
        source_id = odata['source']
        resource_uid = odata.get('src_uid', uid)  # нужно для перекладывания данных другого пользователя
        resource = Bus().get_resource(resource_uid, source_id)
        odata['file_id'] = resource.meta.get('file_id', '')
        set_source_resource_id(resource, odata)
        return super(CopyMove, classname).Create(uid, odata, **kw)

    @classmethod
    def prepare_arguments(classname, uid, odata, **kw):
        resource_uid = odata.get('src_uid', uid)  # нужно для перекладывания данных другого пользователя
        odata['resource_type'] = Bus().info(resource_uid, odata['source'])['this']['type']
        if not odata['force']:
            try:
                Bus().info(uid, odata['target'])
            except errors.NotFound:
                pass
            else:
                raise errors.CopyTargetExists()

        try:
            tgt_address = Address(odata['target'])
            parent_addr = tgt_address.get_parent()
            factory.get_resource(uid, parent_addr)
        except errors.ResourceNotFound:
            raise errors.CopyParentNotFound


class UnknownOperation(Operation):
    pass
