# -*- coding: utf-8 -*-

from mpfs.common.static import tags
from mpfs.common.static.tags.push import OPERATIONS
from mpfs.core.event_dispatcher.events import subscribe
from mpfs.core.event_history.logger import get_pattern
from mpfs.core.operations.filesystem.copy import CopyAttachDisk, CopyOnDisk, CopyWebDisk, CopyMailDisk
from mpfs.core.operations.filesystem.store import ExtractFileFromArchive
from mpfs.core.operations.events import (
    OperationBeforeCompletedEvent,
    OperationBeforeDoneEvent,
    OperationCreatedEvent
)
from mpfs.core.lenta.utils import EventLogMessagesTransformer
from mpfs.core.pushnotifier.queue import PushQueue


@subscribe(OperationCreatedEvent)
def push_operation_created_notification(event):
    operation = event.data['operation']
    oid = operation.id
    uid = operation.uid
    connection_id = event.data['connection_id']

    if operation.type == 'wake_up':
        return

    push_queue = PushQueue()
    push_queue.put({
        'class': OPERATIONS,
        'connection_id': connection_id,  # на будущее для Верстки
        'uid': uid,
        'xiva_data': {
            'root': {
                'tag': 'operation',
                'parameters': {
                    'oid': oid,
                    'type': 'created'  # тип пуша, а не операции
                }
            }
        }
    })


@subscribe(event_class=OperationBeforeCompletedEvent)
@subscribe(event_class=OperationBeforeDoneEvent)
def obtain_lenta_block_id_and_store_in_operation(event):
    operation = event.data['operation']
    if tags.LENTA_BLOCK_ID in operation.data:
        return

    catcher = operation.event_log_catcher
    # строгое соответствие, тк наследники могут иметь другую логику
    if type(operation) in (CopyOnDisk, CopyWebDisk, ExtractFileFromArchive):
        messages = catcher.get_messages(
            pattern=get_pattern(operation=operation)
        )
        EventLogMessagesTransformer.to_lenta_block_id_for_op(
            messages=messages,
            operation=operation,
            save=False  # сохранение произойдет в set_done/set_completed
        )
    elif type(operation) in (CopyAttachDisk, CopyMailDisk):
        messages = catcher.get_messages(
            pattern=get_pattern(operation=operation)
        )
        EventLogMessagesTransformer.to_lenta_block_id_for_public_op(
            messages=messages,
            operation=operation,
            save=False
        )
