# -*- coding: utf-8 -*-

import traceback

import mpfs.engine.process
from mpfs.common import errors
from mpfs.config import settings
from mpfs.core.operations.base import Operation
from mpfs.core.bus import Bus
from mpfs.common.static import tags, codes
from mpfs.core import factory
from mpfs.common.util import from_json
from mpfs.common.static import messages

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

FEATURE_TOGGLES_BULK_ACTIONS_LIMIT = settings.feature_toggles['bulk_actions_limit']
FEATURE_TOGGLES_BULK_ACTIONS_CHECK_DRY_RUN = settings.feature_toggles['bulk_actions_check_dry_run']


class BulkActionOperation(Operation):
    """
    Групповая операция

    Принимает JSON-массив команд, которые последовательно выполняет.
    Для операций копирования-перемещния в статусе при успешном завершении
    покажет данные по ресурсу.
    """
    type = 'bulk'
    subtype = 'filesystem'

    actions = {
        'copy': {
            'handle': Bus().copy_resource,
            'args': ['uid', 'src', 'dst', 'force', 'force_djfs_albums_callback'],
            'return_resource': 'dst',
        },
        'move': {
            'handle': Bus().move_resource,
            'args': ['uid', 'src', 'dst', 'force'],
            'return_resource': 'dst',
        },
        'rm': {
            'handle': Bus().rm,
            'args': ['uid', 'path'],
            'kwargs': ['lock']
        },
        'trash_restore': {
            'handle': Bus().trash_restore,
            'args': ['uid', 'path'],
            'kwargs': ['force'],
        },
        'restore_deleted': {
            'handle': Bus().restore_deleted,
            'args': ['uid', 'path'],
            'kwargs': ['force'],
        },
    }

    @classmethod
    def Create(cls, uid, odata, **kw):
        cmd = odata.get('cmd', '[]')
        cls.check_actions_limit(uid, cmd)

        return super(BulkActionOperation, cls).Create(uid, odata, **kw)

    @staticmethod
    def check_actions_limit(uid, raw_bulk_actions):
        try:
            bulk_actions = from_json(raw_bulk_actions)
        except Exception:
            if not FEATURE_TOGGLES_BULK_ACTIONS_CHECK_DRY_RUN:
                raise errors.BadRequestError('Body should be a JSON array')
            log.info('Wrong body format: JSON array required')
        else:
            if len(bulk_actions) > FEATURE_TOGGLES_BULK_ACTIONS_LIMIT:
                if not FEATURE_TOGGLES_BULK_ACTIONS_CHECK_DRY_RUN:
                    raise errors.BulkActionsLimitExceeded()
                log.info('The bulk request contains too many actions: uid=%s actions=%s.' % (uid, len(bulk_actions)))

    def _process(self):
        protocol = self.data[tags.PROTOCOL] = []
        commands = self.data['commands'] = from_json(self.data['cmd'])
        for command in commands:
            protocol.append({
                tags.STATUS  : codes.WAITING,
                tags.TYPE    : command.get('action'),
                tags.RESULT  : None,
            })
        self.save()

        for i in xrange(len(protocol)):
            result = None
            status = None
            try:
                command = commands[i]
                action = command.get('action')
                handle = self.actions.get(action).get('handle')
                arglist = self.actions.get(action).get('args')
                kwargslist = self.actions.get(action).get('kwargs', [])

                params = map(
                    (lambda x: command.get('params').get(x)),
                    arglist
                )

                kwargs = {
                    x: command.get('params').get(x) for x in kwargslist
                }

                result = handle(*params, **kwargs)
                status = codes.COMPLETED
            except Exception, e:
                result = traceback.format_exc()
                error_log.error(result)
                status = codes.FAILED
            finally:
                protocol[i][tags.RESULT] = result
                protocol[i][tags.STATUS] = status
                self.save()

        self.set_completed()

    def get_status(self):
        """
        Если идет веб-запрос и операция завершена:
        - смотрим все выполненные команды
        - если команда успешная и должна выдать данные по ресурсу, то
          поднимаем ресурс и проставляем его форму
        """

        protocols = self.data.get(tags.PROTOCOL, [])
        commands = self.data.get('commands', [])

        for i in xrange(len(protocols)):
            protocol = protocols[i]

            if hasattr(self, 'mpfs_request'):
                command = commands[i]
                action = self.actions.get(command.get('action'))

                if protocol[tags.STATUS] == codes.COMPLETED and 'return_resource' in action:
                    resource_arg = action.get('return_resource')
                    resource_path = command.get('params').get(resource_arg)
                    resource_data = protocol.get(tags.RESULT).get('this')
                    resource = factory.get_resource(self.uid,
                                                    resource_path,
                                                    data=resource_data
                                                    )
                    protocols[i][tags.FORM] = resource.form

            status = protocol[tags.STATUS]
            protocol[tags.STATUS] = messages.old_operation_titles.get(status, status)
            protocol[tags.STATE] = messages.true_operation_titles.get(status, status)

        result = {
            tags.STATUS   : self.state,
            tags.TYPE     : self.type,
            tags.PROTOCOL : protocols,
        }

        if self.is_failed():
            result[tags.ERROR] = self.data[tags.ERROR]

        return result

