# -*- coding: utf-8 -*-
import os
import time
import traceback
import mpfs.engine.process

from mpfs.core.operations.base import Operation, ServiceUploadOperation
from mpfs.core.operations.base import SaveOnDiskOperation, CopyMove
from mpfs.core.services.mail_service import MailStidService
from mpfs.core.services.narod_service import OriginalNarod
from mpfs.core.factory import get_resource
from mpfs.core.address import Address
from mpfs.core.services import kladun_service
from mpfs.common import errors
from mpfs.common.util import hashed, normalize_string
from mpfs.core.bus import Bus
from mpfs.common.static.tags import *
from mpfs.core import factory
from mpfs.core.user import base as user
from mpfs.core.filesystem.helpers.lock import LockUpdateTimer
from mpfs.core.filesystem.resources.base import Resource
from mpfs.core.user.constants import PHOTOUNLIM_AREA

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class CopyToService(ServiceUploadOperation):
    type = 'copy'
    kladun = kladun_service.Publish()

    @classmethod
    def Create(classname, uid, odata, **kw):
        source_id = odata['source']
        file_data = Bus().info(uid, source_id)['this']
        d = {
             'uid'      : uid,
             'file-id'  : file_data['meta'].get('file_id', ''),
             'path'     : file_data['id'],
             'mulca-id' : file_data['meta']['file_mid'],
             'service'  : Address(odata['target']).storage_name,
             'oid'      : odata['id'],
             }
        int_status_url = classname.post_request_to_kladun(d, **kw)
        odata['status_url'] = int_status_url
        return super(CopyToService, classname).Create(uid, odata, **kw)


class CopyDiskFotki(CopyToService):

    subtype = 'disk_fotki'


class CopyToDisk(SaveOnDiskOperation):

    type = 'copy'
    kladun = kladun_service.Capture()

    @classmethod
    def check_already_exist(cls, uid, target_id, odata):
        try:
            Bus().info(uid, target_id)
            if not odata.get('force'):
                raise errors.FileAlreadyExist()
        except errors.NotFound:
            pass

    @classmethod
    def prepare_arguments(cls, uid, odata, **kw):
        source = Address(odata['source'])
        target = Address(odata['target'])
        try:
            source_object = kw['source_object']
        except KeyError:
            source_object = get_resource(uid, source)
        parent_folder = factory.get_resource(uid, target.get_parent())
        parent_folder.check_rw()
        cls.check_already_exist(uid, target.id, odata)

        return source, target, source_object, odata

    @classmethod
    def Create(classname, uid, odata, **kw):
        source, target, source_object, odata = classname.prepare_arguments(uid, odata, **kw)
        if not odata.get('target'):
            odata['target'] = target.id

        if 'predefined_file_id' in odata:
            odata['file_id'] = odata['predefined_file_id']
        else:
            odata['file_id'] = Resource.generate_file_id(uid, odata['target'])
        source_service = 'mail' if source.storage_name == 'mulca' else source.storage_name

        d = {
            'uid': uid,
            'file-id': odata['file_id'],
            'path': target.id,
            'source-service': source_service,
            'oid': odata['id'],
        }

        hardlink = None

        if source.storage_name == 'narod':
            sha256 = source_object.meta.get('sha256')
            if sha256:
                md5 = source_object.meta['md5']
                size = source_object.size
                try:
                    hardlink = Bus().hardlink(md5, size, sha256)
                except errors.HardLinkNotFound, e:
                    pass
                except errors.HardlinkBroken, e:
                    pass

        if hardlink:
            odata['path'] = target.id
            operation = super(CopyToDisk, classname).Create(uid, odata, **kw)
            operation.data['filedata']= {
                'meta' : {
                    'md5' : hardlink.md5,
                    'sha256' : hardlink.sha256,
                    },
                'size' : hardlink.size,
            }
            try:
                operation.hardlink_copy()
            except errors.HardlinkFileNotInStorage as e:
                hardlink = None
                operation.set_failed(e)
        else:
            if 'free_space' in odata:
                d['free_space'] = odata['free_space']

            mail_stid_service = MailStidService()
            if source.storage_name == 'mail':
                d['source-service'] = 'mail2'
                d['service-file-id'] = mail_stid_service.get_service_file_id(uid, source_object.meta['mid'], source_object.meta['hid'])
            elif source.storage_name == 'mulca':
                d['service-file-id'] = '%s:%s/%s' % (uid, source_object.meta['stid'], source_object.meta['part'])
            elif source.storage_name == 'yavideo':
                d['service-file-id'] = '%s:%s' % (uid, source.path)
            else:
                d['service-file-id'] = '%s:%s' % (uid, source.name)

            int_status_url = classname.post_request_to_kladun(d, **kw)
            odata['status_url'] = int_status_url
            if source.storage_name == 'mail':
                odata['source_data'] = {'source': 'mail'}
            else:
                odata['source_data'] = source_object.dict()

            # https://jira.yandex-team.ru/browse/CHEMODAN-5520
            for field in ('ctime', 'mtime'):
                if field in odata['source_data']:
                    del(odata['source_data'][field])
            operation = super(CopyToDisk, classname).Create(uid, odata, **kw)

        return operation

    def certain_infoupdate(self):
        self.data['filedata'].update({
            'size'      : self.downloaded_file.content_length,
            'mimetype'  : self.downloaded_file.mimetype,
            'source'    : self.data['source_data']['source'],
        })
        self.data['filedata']['meta'].update({
            'md5'       : self.downloaded_file.md5,
            'sha256'    : self.downloaded_file.sha256,
            'file_id'   : self.data['file_id'],
        })
        self.data['path'] = self.data['target']

        if 'original_name' in self.data:
            self.data['filedata']['original_name'] = self.data['original_name']


class CopyFotkiDisk(CopyToDisk):
    subtype = 'fotki_disk'


class CopyNarodDisk(CopyToDisk):
    subtype = 'narod_disk'

    @classmethod
    def prepare_arguments(cls, uid, odata, **kw):
        source = Address(odata['source'])
        target = Address(odata['target'])
        try:
            source_object = kw['source_object']
        except KeyError:
            source_object = get_resource(uid, source)

        if not odata.get('allow_passworded', False):
            if source_object.meta['pass'] != '':
                raise errors.PreconditionsFailed(odata['source'])

        parent_folder = factory.get_resource(uid, target.get_parent())
        parent_folder.check_rw()

        try:
            Bus().info(uid, target.id)
            if not odata.get('force'):
                raise errors.FileAlreadyExist()
        except errors.NotFound:
            pass

        return source, target, source_object, odata


class CopySocialDisk(CopyToDisk):

    type = 'social_copy'

    @classmethod
    def prepare_arguments(cls, uid, odata, **kw):
        target = Address(odata['target'])

        parent_folder = factory.get_resource(uid, target.get_parent())
        parent_folder.check_rw()

        if not odata.get('force') and Bus().exists(uid, target.id):
            raise errors.FileAlreadyExist()

        return target

    @classmethod
    def Create(cls, uid, odata, **kw):
        target = cls.prepare_arguments(uid, odata, **kw)
        if 'file_id' not in odata:
            odata['file_id'] = Resource.generate_file_id(uid, odata['target'])
        odata['source_data'] = dict(source=odata['provider'])

        d = {
            'uid': uid,
            'file-id': odata['file_id'],
            'path': target.id,
            'source-service': odata['provider'],
            'oid': odata['id'],
            'service-file-url': odata['service_file_url'],
            # str() prevents filtering inside open_url
            'disable-retries': str(odata.get('disable_retries', False)),
            'disable-redirects': str(odata.get('disable_redirects', False)),
            'exclude-orientation': str(odata.get('exclude_orientation', False)),
        }

        if 'mulca-id' in odata:
            d['mulca-id'] = odata['mulca-id']

        if 'latitude' in odata:
            d['latitude'] = odata['latitude']

        if 'longitude' in odata:
            d['longitude'] = odata['longitude']
        if 'created' in odata:
            d['created'] = odata['created']

        # только для ханкома
        if 'service_file_id' in odata:
            d['service-file-id'] = odata['service_file_id']

        int_status_url = cls.post_request_to_kladun(d, **kw)
        odata['status_url'] = int_status_url

        return super(CopyToDisk, cls).Create(uid, odata, **kw)


class CopyVkDisk(CopySocialDisk):
    subtype = 'vkontakte_disk'


class CopyFacebookDisk(CopySocialDisk):
    subtype = 'facebook_disk'


class CopyOdnoklassnikiDisk(CopySocialDisk):
    subtype = 'odnoklassniki_disk'


class CopyMailruDisk(CopySocialDisk):
    subtype = 'mailru_disk'


class CopyGoogleDisk(CopySocialDisk):
    subtype = 'google_disk'


class CopyInstagramDisk(CopySocialDisk):
    subtype = 'instagram_disk'


class CopyAviaryDisk(CopySocialDisk):
    subtype = 'aviary_disk'
    fs = Bus()

    @classmethod
    def format_target_path(cls, uid, path, filename_suffix):
        source = Address(path, uid=uid)
        # photounlim сохраняем в Фотокамеру
        if source.storage_name == PHOTOUNLIM_AREA:
            photostream = cls.fs.mksysdir(uid, 'photostream')
            source.change_parent(photostream['id'])
        return source.add_suffix(filename_suffix).path

    def set_completed(self):
        super(CopyAviaryDisk, self).set_completed()

        render_operation_id = self.data.get('render_operation_id', None)
        if render_operation_id is not None:
            # placed here due to circular import
            from mpfs.core.operations import manager
            operation = manager.get_operation(self.uid, render_operation_id)
            operation.set_completed()


class BaseExternalCopyWebDisk(CopySocialDisk):
    type = 'external_copy'
    subtype = None
    fs = Bus()

    def set_completed(self):
        super(BaseExternalCopyWebDisk, self).set_completed()
        path = Address(self.data['target']).id
        self.fs.patch_file(self.uid, path, {
            'external_url': self.data['service_file_url'],
        })


class CopyWebDisk(BaseExternalCopyWebDisk):
    subtype = 'web_disk'

    @classmethod
    def prepare_arguments(cls, uid, odata, **kw):
        odata['target'] = Bus().autosuffix_address(Address(odata['target'])).id
        return super(CopyWebDisk, cls).prepare_arguments(uid, odata, **kw)


class CopyNarodLegacy(CopyNarodDisk):
    subtype = None

    @classmethod
    def prepare_arguments(cls, uid, odata, **kw):
        source, target, source_object, odata = super(CopyNarodLegacy, cls).prepare_arguments(uid, odata, **kw)

        if int(source_object.meta.get('blocked', 0)):
            raise errors.ResourceBlocked(source)

        return source, target, source_object, odata

    @classmethod
    def Create(classname, uid, odata, **kw):
        tgt_address = Address(odata['target'])
        src_address = Address(odata['source'])
        original_name = tgt_address.name
        address = tgt_address.rename('%s_%s' % (original_name, time.time()))
        odata['target'] = address.id
        odata['original_name'] = original_name
        return super(CopyNarodLegacy, classname).Create(uid, odata, **kw)

    def set_completed(self):
        tgt_address = Address(self.data['target'])
        src_address = Address(self.data['source'])

        public_url = ''
        try:
            if not self.data.get('password', None):
                try:
                    from mpfs.core.social.publicator import Publicator
                    publication_result = Publicator().set_public(self.uid, tgt_address.id,
                                                                 oper_type=self.type, oper_subtype=self.subtype)
                    public_url = publication_result['short_url']
                except errors.ViralFilesNotAllowedToPublicate, e:
                    public_url = 'http://narod.yandex.ru/disk/passworded'
            else:
                public_url = 'http://narod.yandex.ru/disk/passworded'
            OriginalNarod().set_file_moved(src_address, public_url)
        except Exception:
            error_log.error(traceback.format_exc())
            error_log.info('failed to update narod redirect %s %s' % (src_address.id, public_url))

        super(CopyNarodLegacy, self).set_completed()


class CopyOriginalNarodLegacyNarod(CopyNarodLegacy):
    subtype = 'narod_lnarod'

    @classmethod
    def Create(classname, uid, odata, **kw):
        if user.NeedInit(uid, type='narod'):
            user.Create(uid, type='narod')
        return super(CopyOriginalNarodLegacyNarod, classname).Create(uid, odata, **kw)


class CopyOriginalNarodAttach(CopyNarodLegacy):
    subtype = 'narod_attach'

    @classmethod
    def Create(classname, uid, odata, **kw):
        if user.NeedInit(uid, type='attach'):
            user.Create(uid, type='attach')
        return super(CopyOriginalNarodAttach, classname).Create(uid, odata, **kw)


class CopyMailDisk(CopyToDisk):
    subtype = 'mail_disk'


class ImportMailAttach(CopyToDisk):  # отличается от CopyMailDisk только тем, что принимает на вход mid и hid аттача, а не "uid:/mail/folder:xxx/file:yyy"
    subtype = 'import_mail_attach'

    @classmethod
    def prepare_arguments(cls, uid, odata, **kw):
        source_mid = odata['source_mid']
        source_hid = odata['source_hid']
        target = Address(odata['target'])

        parent_folder = factory.get_resource(uid, target.get_parent())
        parent_folder.check_rw()
        if odata['autosuffix']:
            target = Bus().autosuffix_address(target)
        else:
            cls.check_already_exist(uid, target.id, odata)

        return source_mid, source_hid, target, odata

    @classmethod
    def Create(cls, uid, odata, **kw):
        source_mid, source_hid, target, odata = cls.prepare_arguments(uid, odata, **kw)
        odata['target'] = target.id

        odata['file_id'] = Resource.generate_file_id(uid, odata['target'])

        mail_stid_service = MailStidService()
        d = {
            'uid': uid,
            'file-id': odata['file_id'],
            'path': target.id,
            'source-service': 'mail2',
            'oid': odata['id'],
            'service-file-id': mail_stid_service.get_service_file_id(uid=uid, mail_mid=source_mid, mail_hid=source_hid)
        }

        if 'free_space' in odata:
            d['free_space'] = odata['free_space']

        odata['status_url'] = cls.post_request_to_kladun(d, **kw)
        operation = super(CopyToDisk, cls).Create(uid, odata, **kw)

        return operation

    def certain_infoupdate(self):
        self.data['path'] = self.data['target']
        self.data['filedata'].update({
            'size': self.downloaded_file.content_length,
            'mimetype': self.downloaded_file.mimetype,
        })
        self.data['filedata']['meta'].update({
            'md5': self.downloaded_file.md5,
            'sha256': self.downloaded_file.sha256,
            'file_id': self.data['file_id'],
        })


class ImportFileFromService(CopyToDisk):
    subtype = 'import_file_from_service'
    fs = Bus()

    @classmethod
    def Create(cls, uid, odata, **kw):
        cls.fs.mksysdir(uid, type='downloads')
        downloads_folder_address = cls.fs.get_downloads_address(uid)
        downloads_folder = factory.get_resource(uid, downloads_folder_address)
        downloads_folder.check_rw()

        dst_address = downloads_folder_address.get_child_file(odata['file_name'])
        if odata['autosuffix']:
            dst_address = cls.fs.autosuffix_address(dst_address)
        elif not odata['force']:
            cls.check_already_exist(uid, dst_address.id, odata)

        odata['target'] = dst_address.id
        odata['file_id'] = Resource.generate_file_id(uid, odata['target'])

        d = {
            'uid': uid,
            'file-id': odata['file_id'],
            'path': dst_address.id,
            'source-service': odata['service_id'],
            'oid': odata['id'],
            'service-file-id': odata['service_file_id']
        }

        if 'free_space' in odata:
            d['free_space'] = odata['free_space']

        odata['status_url'] = cls.post_request_to_kladun(d, **kw)
        operation = super(CopyToDisk, cls).Create(uid, odata, **kw)

        return operation

    def certain_infoupdate(self):
        self.data['path'] = self.data['target']
        self.data['filedata'].update({
            'size': self.downloaded_file.content_length,
            'mimetype': self.downloaded_file.mimetype,
        })
        self.data['filedata']['meta'].update({
            'md5': self.downloaded_file.md5,
            'sha256': self.downloaded_file.sha256,
            'file_id': self.data['file_id'],
        })


class CopyMulcaDisk(CopyToDisk):
    subtype = 'mulca_disk'


class CopyYavideoDisk(CopyToDisk):
    subtype = 'yavideo_disk'

    @classmethod
    def check_already_exist(cls, uid, target_id, odata):
        pass

    def _process(self, *args, **kwargs):
        # To avoid file name collisions
        tgt_address = Address(self.data['target'])
        tgt_address = Bus().autosuffix_address(tgt_address)
        self.data['new_path'] = tgt_address.id
        super(CopyToDisk, self)._process(*args, **kwargs)

    def get_path(self):
        return self.data['new_path']


class CopyOnDisk(CopyMove):
    type = 'copy'
    subtype = 'disk_disk'

    def _process(self, *args, **kwargs):
        uid = self.uid
        source = self.data['source']
        target = self.data['target']
        force = self.data['force']
        make_visible = bool(self.data.get('make_visible', False))
        lock_source = self.data.get('lock_source', True)
        src_uid = self.data.get('src_uid')
        force_djfs_albums_callback = self.data.get('force_djfs_albums_callback', False)

        bus = Bus(connection_id=self.data['connection_id'])
        timer = LockUpdateTimer(addresses=(source, target))

        try:
            timer.start()
            predefined_file_id = self.data.get('predefined_file_id')
            result = bus.copy_resource(uid, source, target, force, lock_source=lock_source, make_visible=make_visible,
                                       predefined_file_id=predefined_file_id,
                                       src_uid=src_uid, lock_data={'oid': self.id},
                                       force_djfs_albums_callback=force_djfs_albums_callback)
            self.data['affected_resource'] = result['this']['id']

            if self.data.get('set_public', False):
                from mpfs.core.social.publicator import Publicator
                Publicator().set_public(uid, Address(target).id, oper_type=self.type, oper_subtype=self.subtype)

            self.set_completed()
        finally:
            timer.stop()

    def process_interruption(self):
        lock_source = self.data.get('lock_source', True)
        if lock_source:
            src_address = self.data['source']
            self.release_lock(src_address)

    def get_uniq_id(self):
        return hashed(
                str(self.uid) +
                str(self.type) +
                normalize_string(self.data['source']) +
                normalize_string(self.data['target'])
        )


class CopyAttachDisk(CopyOnDisk):
    subtype = 'attach_disk'


class CopyLegacyNarodDisk(CopyOnDisk):
    subtype = 'lnarod_disk'


class CopyDiskLegacyNarod(CopyOnDisk):
    subtype = 'disk_lnarod'

    def _process(self, *args, **kwargs):
        uid = self.uid

        if user.NeedInit(uid, type='narod'):
            user.Create(uid, type='narod')

        tgt_address = Address(self.data['target'])
        original_name = tgt_address.name
        address = tgt_address.rename('%s_%s' % (original_name, time.time()))

        source = self.data['source']
        resource_data = {'original_name' : original_name}
        bus = Bus(connection_id=self.data['connection_id'])

        result = bus.copy_resource(uid, source, address.id, self.data['force'], resource_data=resource_data)
        self.data['affected_resource'] = result['this']['id']

        from mpfs.core.social.publicator import Publicator
        Publicator().set_public(self.uid, address.id, oper_type=self.type, oper_subtype=self.subtype)

        self.set_completed()


class CopyDiskAttach(CopyOnDisk):
    subtype = 'disk_attach'

    @classmethod
    def prepare_arguments(cls, uid, odata, **kw):
        odata['resource_type'] = Bus().info(uid, odata['source'])['this']['type']

    def _process(self, *args, **kwargs):
        uid = self.uid

        if user.NeedInit(uid, type='attach'):
            user.Create(uid, type='attach')

        tgt_address = Address(self.data['target'])
        original_name = tgt_address.name
        address = tgt_address.rename('%s_%s' % (original_name, time.time()))

        source = self.data['source']
        resource_data = {'original_name' : original_name}
        bus = Bus(connection_id=self.data['connection_id'])

        result = bus.copy_resource(uid, source, address.id, self.data['force'], resource_data=resource_data)
        self.data['affected_resource'] = result['this']['id']

        from mpfs.core.social.publicator import Publicator
        Publicator().set_public(self.uid, address.id, oper_type=self.type, oper_subtype=self.subtype)
        self.set_completed()


class CopyMailAttach(CopyToDisk):
    subtype = 'mail_attach'

    @classmethod
    def prepare_arguments(cls, uid, odata, **kw):
        source, target, source_object, odata = super(CopyMailAttach, cls).prepare_arguments(uid, odata, **kw)
        odata['resource_type'] = Bus().info(uid, odata['source'])['this']['type']
        return source, target, source_object, odata

    @classmethod
    def Create(cls, uid, odata, **kw):
        tgt_address = Address(odata['target'])
        original_name = tgt_address.name
        tgt_address = tgt_address.rename('%s_%s' % (original_name, time.time()))
        odata['target'] = tgt_address.id
        odata['original_name'] = original_name

        if user.NeedInit(uid, type='attach'):
            user.Create(uid, type='attach')

        operation = super(CopyMailAttach, cls).Create(uid, odata, **kw)
        return operation

    def additional_filedata(self):
        return {'original_name': self.data['original_name']}

    def set_completed(self):
        tgt_address = Address(self.data['target'])
        self.data['affected_resource'] = tgt_address.id

        from mpfs.core.social.publicator import Publicator
        Publicator().set_public(self.uid, tgt_address.id, oper_type=self.type, oper_subtype=self.subtype)

        super(CopyMailAttach, self).set_completed()


class CopyPhotounlimDisk(CopyOnDisk):
    subtype = '_'.join([PHOTOUNLIM_AREA, 'disk'])


# handled in djfs-worker
class FilesystemCopy(Operation):
    # такой тип, чтобы вёрстка отображала бегущую операцию
    type = 'copy'
    subtype = 'copy'
    task_name = 'filesystem_copy_operation'
