# -*- coding: utf-8 -*-
import time

from pymongo import ReadPreference

import mpfs.engine.process

from mpfs.common import errors
from mpfs.common.static.codes import EXECUTING
from mpfs.common.static.tags import XML_STORE_STATUS

from mpfs.core import factory
from mpfs.core.address import Address
from mpfs.core.bus import Bus
from mpfs.core.filesystem import hardlinks
from mpfs.core.filesystem.resources.base import Resource
from mpfs.core.services import kladun_service
from mpfs.core.operations.base import DiskUploadOperation
from mpfs.core.user.constants import HIDDEN_AREA

log = mpfs.engine.process.get_default_log()


class StoreDiskDelta(DiskUploadOperation):

    type = 'dstore'
    subtype = 'disk'
    kladun = kladun_service.Patch()

    @classmethod
    def Create(classname, uid, odata, **kw):
        if 'new_path' in odata:
            rawpath = odata['new_path']
        else:
            rawpath = odata['path']
        path = Address(rawpath)

        odata['file_id'] = odata['file']['meta'].get('file_id') or Resource.generate_file_id(uid, path.id)
        d = {
            'uid'          : uid,
            'path'         : path.id,
            'file-id'      : odata['file_id'],
            'mulca-id'     : odata['file']['meta']['file_mid'],
            'original-md5' : odata['file']['meta']['md5'],
            'content-type' : odata['file']['mimetype'],
            'oid'          : odata['id'],
            'service'      : 'disk',
        }

        if 'free_space' in odata:
            d['free_space'] = odata['free_space']

        ext_patch_url, int_status_url = classname.post_request_to_kladun(d, **kw)

        odata['patch_url'] = ext_patch_url
        odata['status_url'] = int_status_url
        odata['state'] = EXECUTING
        odata['original_md5'] = odata['file']['meta']['md5']
        odata['file'] = {}

        return super(StoreDiskDelta, classname).Create(uid, odata, **kw)

    def _process(self, xml_data, stage):
        self.update_status(xml_data)

        if self.is_kladun_first_callback(stage):
            self.handle_callback_1()
        elif self.check_data_uploaded() and not self.data.get('hardlinked'):
            self.handle_callback_2_and_3()

    def handle_callback_1(self):
        def common_upload():
            log.info('patching by simple file')
            Bus().check_available_space(
                self.uid,
                self.get_path(),
                required_space=self.kladun_patched_parse.content_length,
            )

        if not (self.data.get('own_hardlink') or
                self.data.get('hardlinked')):
            if self._try_hardlink_copy(action='patching from hardlink'):
                return

        common_upload()

    def handle_callback_2_and_3(self):
        self.save_or_update_file()

    def _get_sha_size_md5(self):
        md5    = self.kladun_patched_parse.md5
        sha256 = self.kladun_patched_parse.sha256
        size   = self.kladun_patched_parse.content_length
        return md5, sha256, size

    def hardlink_copy(self):
        self._process_found_hid_on_store_step()

        md5, sha256, size = self._get_sha_size_md5()

        fs = Bus(connection_id=self.data.get('connection_id', ''))
        existing_file = factory.get_resource(self.uid, self.get_path())

        if existing_file.md5() != self.data['original_md5']:
            raise errors.KladunFileModifiedBetweenCallbacks()

        existing_file_data = existing_file.dstore_data()
        existing_file_data.update(self.data.get('changes', {}))
        fs.check_available_space(
            self.uid,
            self.get_path(),
            required_space=size,
        )
        link = hardlinks.HardLink(md5, size, sha256)
        existing_file_data['hid'] = link._id
        existing_file_data.update(self.data.get('changes'))

        fs.hardlink_copy(
            self.uid,
            self.get_path(),
            md5, size, sha256,
            force = True,
            keep_symlinks = True,
            notify = False,
            filedata = existing_file_data,
            oper_type=self.type,
            oper_subtype=self.subtype
        )

        self.data['hardlinked'] = True
        self.set_completed()

    def save_or_update_file(self):
        file_data = {
            'file_mid'    : self.mulca_file_upload.mulca_id,
            'digest_mid'  : self.mulca_digest_upload.mulca_id,
            'md5'         : self.kladun_patched_parse.md5,
            'drweb'       : self.antivirus_check.drweb,
            'sha256'      : self.kladun_patched_parse.sha256,
            'file_id'     : self.data['file_id'],
        }

        try:
            etime = self.exif_info.etime
        except AttributeError:
            etime = None

        try:
            vctime = self.media_info.etime
        except AttributeError:
            vctime = None

        if etime or vctime:
            file_data['etime'] = etime or vctime

        if hasattr(self, 'preview_image'):
            file_data['previews'] = self.preview_image.preview_links
        else:
            file_data['previews'] = {}

        if hasattr(self, 'preview_sizes'):
            file_data.update({'preview_sizes': self.preview_sizes.preview_sizes})
        else:
            file_data.update({'preview_sizes': {}})

        self.set_preview_stid(file_data)

        try:
            video_info = self.video_info.content
        except AttributeError:
            pass
        else:
            if video_info:
                file_data['video_info'] = video_info

        fs = Bus(connection_id=self.data.get('connection_id', ''))

        prev_read_pref = mpfs.engine.process.get_read_preference()
        mpfs.engine.process.set_read_preference(ReadPreference.PRIMARY_PREFERRED)
        try:
            existing_resource = factory.get_resource(self.uid, self.get_path())
        except errors.ResourceNotFound:
            try:
                existing_resource = factory.get_resource_by_file_id(self.uid, self.data['file_id'])
            except errors.ResourceNotFound:
                existing_resource = None
            else:
                if existing_resource.address.storage_name == HIDDEN_AREA:
                    existing_resource = None
                else:
                    self.data['path'] = existing_resource.address.id
        finally:
            mpfs.engine.process.set_read_preference(prev_read_pref)

        existing_resource_md5 = existing_resource.md5()
        existent_file_data = existing_resource.dstore_data()
        is_new_file = existing_resource.meta['file_mid'] != file_data['file_mid']

        existent_file_data['size'] = self.kladun_patched_parse.content_length
        existent_file_data['meta'].update(file_data)
        client_data = self.data.get('changes', {})
        client_data['mtime'] = int(client_data.get('mtime', time.time()))
        existent_file_data.update(client_data)


        if is_new_file:
            if not self.check_main_stages_only() and self.check_data_uploaded():
                # CHEMODAN-16358
                raise errors.KladunFileModifiedBetweenCallbacks()
            else:

                if existing_resource_md5 != self.data['original_md5']:
                    raise errors.KladunFileModifiedBetweenCallbacks()

                fs.check_available_space(
                    self.uid,
                    self.get_path(),
                    required_space=existent_file_data['size'],
                )

                fs.mkfile(
                    self.uid,
                    self.get_path(),
                    data=existent_file_data,
                    keep_symlinks=True,
                    notify=True,
                    is_store=True,
                )
                self.data['own_hardlink'] = True
                self.set_done()
        else:
            self.handle_callback_3(existing_resource, existent_file_data)
            self.set_completed()

    def set_preview_stid(self, file_data):
        try:
            file_data['pmid'] = self.preview.mulca_id
        except AttributeError:
            pass
        else:
            return

        try:
            file_data['pmid'] = self.single_preview_mid.mulca_id
        except AttributeError:
            pass
        else:
            return

        try:
            file_data['pmid'] = self.single_preview_video_mid.mulca_id
        except AttributeError:
            pass
        else:
            return


class StoreDiskDeltaPhotostream(StoreDiskDelta):

    type = 'dstore'
    subtype = 'photostream'

    @classmethod
    def Create(classname, uid, odata, **kw):
        fs = Bus()
        path = fs.preprocess_path(uid, odata['path'])
        odata['new_path'] = path
        return super(StoreDiskDeltaPhotostream, classname).Create(uid, odata, **kw)

    def set_path(self, path):
        self.data['new_path'] = path

    def get_path(self):
        return self.data['new_path']
