# -*- coding: utf-8 -*-
from mpfs.core.operations.base import CopyMove
from mpfs.core.bus import Bus
from mpfs.core.address import Address
from mpfs.common.util import hashed, normalize_string
from mpfs.core import factory
from mpfs.common import errors
from mpfs.core.filesystem.helpers.lock import LockUpdateTimer
from mpfs.core.user.constants import PHOTOUNLIM_AREA
from mpfs.core.yateam.logic import is_yateam_root, user_has_nda_rights, recreate_yateam_dir


class MoveOnDisk(CopyMove):

    type = 'move'
    subtype = 'disk_disk'

    @classmethod
    def prepare_arguments(classname, uid, odata, **kw):
        src_resource = factory.get_resource(uid, Address(odata['source']))
        tgt_address = Address(odata['target'])
        try:
            parent_addr = tgt_address.get_parent()
            target_parent = factory.get_resource(uid, parent_addr)
        except errors.ResourceNotFound:
            raise errors.CopyParentNotFound
        else:
            source_is_shared = src_resource.is_group_root or \
                                src_resource.is_shared_root or src_resource.with_shared
            target_parent_is_shared = target_parent.is_shared or target_parent.is_group


            try:
                tgt_resource = factory.get_resource(uid, Address(odata['target']))
            except errors.NotFound:
                tgt_resource = None
            else:
                if not odata['force']:
                    raise errors.CopyTargetExists()
            #===================================================================
            # https://jira.yandex-team.ru/browse/CHEMODAN-7425
            target_is_shared = tgt_resource and (tgt_resource.is_shared or tgt_resource.is_group or tgt_resource.with_shared)
            source_shared_and_target_shared = source_is_shared and (target_parent_is_shared or target_is_shared)
            source_not_shared_and_target_shared_root = not source_is_shared and tgt_resource and (tgt_resource.is_shared_root or tgt_resource.is_group_root or tgt_resource.with_shared)
            #===================================================================
            if source_shared_and_target_shared or source_not_shared_and_target_shared_root:
                raise errors.MoveWrongDestination()

            if target_parent.is_fully_public() or target_parent.is_public_internal():
                src_resource.check_rw(operation='set_public')

    def _process(self):
        uid = self.uid
        source = self.data['source']
        target = self.data['target']
        force = self.data['force']
        fs = Bus(connection_id=self.data.get('connection_id', ''))
        timer = LockUpdateTimer(addresses=(source, target))
        try:
            timer.start()
            result = fs.move_resource(uid, source, target, force, lock_data={'oid': self.id})
            if is_yateam_root(Address(source).path) and user_has_nda_rights(uid):
                recreate_yateam_dir(uid)
            self.data['affected_resource'] = result['this']['id']
            self.set_completed()
        finally:
            timer.stop()

    def process_interruption(self):
        src_address = self.data['source']
        self.release_lock(src_address)

    def get_status(self):
        result = super(MoveOnDisk, self).get_status()
        if self.data.get('affected_resource') is not None and hasattr(self, 'mpfs_request') and self.mpfs_request.form is not None:
            self.mpfs_request.form.id_old = Address(self.data['source']).path
        return result

    def get_uniq_id(self):
        return hashed(
            str(self.uid) +
            str(self.type) +
            normalize_string(self.data['source']) +
            normalize_string(self.data['target'])
        )


class MoveYavideoDisk(MoveOnDisk):
    subtype = 'yavideo_disk'


class MoveAttachDisk(MoveOnDisk):
    subtype = 'attach_disk'

    @classmethod
    def Create(cls, uid, odata, **kw):
        if not Address(odata['source']).path.startswith('/attach/YaFotki/'):
            raise errors.OperationClassNotFound('attach_disk')
        return super(MoveAttachDisk, cls).Create(uid, odata, **kw)


class MovePhotounlimDisk(MoveOnDisk):
    subtype = '_'.join([PHOTOUNLIM_AREA, 'disk'])


class MoveOnPhotounlim(MoveOnDisk):
    subtype = '_'.join([PHOTOUNLIM_AREA, PHOTOUNLIM_AREA])

    @classmethod
    def prepare_arguments(classname, uid, odata, **kw):
        # убеждаемся, что такой файл есть
        factory.get_resource(uid, Address(odata['source']))

        tgt_address = Address(odata['target'])
        try:
            factory.get_resource(uid, tgt_address.get_parent())
        except errors.ResourceNotFound:
            raise errors.CopyParentNotFound

        try:
            tgt_resource = factory.get_resource(uid, tgt_address)
        except errors.NotFound:
            tgt_resource = None

        if tgt_resource and not odata['force']:
            raise errors.CopyTargetExists()
