# -*- coding: utf-8 -*-
from mpfs.common import errors
from mpfs.common.util import hashed
from mpfs.core.address import Address
from mpfs.core.operations.base import DiskInternalOperation, Operation, set_source_resource_id
from mpfs.core.bus import Bus
from mpfs.core.services.yavideo_service import YaVideo
from mpfs.core.filesystem.helpers.lock import LockUpdateTimer
from mpfs.core.user.constants import PHOTOUNLIM_AREA
from mpfs.core.yateam.logic import is_yateam_root, user_has_nda_rights, recreate_yateam_dir


class RemoveDisk(DiskInternalOperation):
    type = 'remove'
    subtype = 'disk'

    @classmethod
    def Create(cls, uid, odata, **kw):
        path = odata['path']
        resource = Bus().get_resource(uid, path)
        set_source_resource_id(resource, odata)
        return super(RemoveDisk, cls).Create(uid, odata, **kw)

    def _process(self):
        uid = self.uid
        path = self.data['path']
        timer = LockUpdateTimer(addresses=(self.data['path'],))

        try:
            timer.start()
            Bus().rm(uid, path, lock_data={'oid': self.id})
            if is_yateam_root(Address(path).path) and user_has_nda_rights(uid):
                recreate_yateam_dir(uid)
            self.set_completed()
        finally:
            timer.stop()

    def process_interruption(self):
        address = self.data['path']
        self.release_lock(address)


class RemoveBulk(DiskInternalOperation):
    type = 'remove'
    subtype = 'bulk'

    def get_uniq_id(self):
        return hashed(str(self.uid) + str(self.type) + ';'.join(self.data['paths']))

    def _process(self, *args, **kwargs):
        for path in self.data['paths']:
            timer = LockUpdateTimer(addresses=(path,))
            try:
                timer.start()
                try:
                    Bus().rm(self.uid, path, lock_data={'oid': self.id})
                except errors.ResourceNotFound:
                    continue
                if is_yateam_root(Address(path).path) and user_has_nda_rights(self.uid):
                    recreate_yateam_dir(self.uid)
            finally:
                timer.stop()
        self.set_completed()

    def process_interruption(self):
        for path in self.data['paths']:
            self.release_lock(path)


class RemoveYavideo(Operation):
    type = 'remove'
    subtype = 'yavideo'

    def _process(self):
        uid = self.uid
        path = self.data['path']

        YaVideo().remove_film_or_album(uid, path)
        self.set_completed()


class RemovePhotounlim(RemoveDisk):
    subtype = PHOTOUNLIM_AREA
