# -*- coding: utf-8 -*-

import re
import datetime
import time
import traceback

import mpfs.engine.process
from mpfs.common import errors

from mpfs.common.static import codes
from mpfs.core import factory
from mpfs.core.address import Address
from mpfs.core.billing.client import Client
from mpfs.core.billing.product import Product
from mpfs.core.billing.processing.common import simple_create_service
from mpfs.core.filesystem.hardlinks.common import FileChecksums
from mpfs.core.filesystem.live_photo import LivePhotoFilesManager
from mpfs.core.filesystem.quota import Quota
from mpfs.core.filesystem.resources.base import Resource
from mpfs.core.services import kladun_service
from mpfs.core.services.mail_service import MailStidService
from mpfs.core.operations.base import DiskUploadOperation, ServiceUploadOperation, SaveOnDiskOperation
from mpfs.core.bus import Bus
from mpfs.core.user import base as user
from mpfs.core.user.base import User
from mpfs.core.user.constants import PHOTOUNLIM_AREA, CLIENT_AREA, CLIENT_AREA_PATH, PHOTOSTREAM_AREA
from mpfs.core.metastorage.control import client_data
from mpfs.core.services import passport_service, zaberun_service
from mpfs.core.social.publicator import Publicator
from mpfs.core.queue import mpfs_queue
from mpfs.config import settings


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

passport = passport_service.Passport()
zaberun = zaberun_service.Zaberun()

PROMO_AUTO_UPLOAD_32_GB = settings.promo['autoupload32gb']
FEEDBACK_FOS_EMAILS = settings.feedback['fos_email']
FEEDBACK_FOS_TEMPLATE = settings.feedback['fos_template']
FOS_SUBJECT_TEMPLATE = "%(os_version)s - %(app_version)s - %(fos_subject)s"
FOS_BODY_TEMPLATE = """
%(support_text)s

-----
Login: %(login)s
UID: %(uid)s
App Version: %(app_version)s
OS Version: %(os_version)s
Paid: %(is_paid)s
%(download_url)s
""".strip()

unchangeble_setprop_fields = (
    'file_mid',
    'file_url',
    'digest_url',
    'sha256',
    'md5',
    'size',
    'preview',
    'file_id',
    'previews',
    'preview_sizes',
    'digest_mid',
    'drweb',
    ''
)


class StoreDiskMixin(object):
    @staticmethod
    def check_parent_rw(raw_addr):
        parent_addr = Address(raw_addr).get_parent()
        parent_folder = factory.get_resource(parent_addr.uid, parent_addr)
        parent_folder.check_rw()

    def certain_infoupdate(self):
        self.data['filedata'].update({
            'size': self.kladun_parse.content_length,
            'mimetype': self.kladun_parse.mimetype,
        })

        self.data['filedata']['meta'].update({
            'md5': self.kladun_parse.md5,
            'sha256': self.kladun_parse.sha256,
        })


class StoreDisk(StoreDiskMixin, DiskUploadOperation):
    type = 'store'
    subtype = 'disk'

    @classmethod
    def Create(cls, uid, odata, **kw):
        if cls.is_live_photo_store(odata) and odata.get('live_photo_type', None) == 'video':
            photo_md5 = odata['live_photo_md5']
            photo_sha256 = odata['live_photo_sha256']
            photo_size = odata['live_photo_size']

            if odata.get('live_photo_operation_id') is None:
                photo_store_operation_oid = LivePhotoFilesManager.get_paired_live_photo_store_operation_id(
                    uid, odata['live_photo_operation_path'],
                    odata['md5'], odata['sha256'], odata['size'],
                    photo_md5, photo_sha256, photo_size
                )
            else:
                photo_store_operation_oid = odata['live_photo_operation_id']

            try:
                from mpfs.core.operations import manager
                manager.get_operation(uid, photo_store_operation_oid)
            except errors.OperationNotFound:
                # Eсли не нашли парную операцию (загрузку фото), то ищем такую фотку в /disk пользователя.
                # Этот метод райзит исключения, если файл не найден или найдено несколько таких, делаем это здесь,
                # чтобы сразу отловить эту ситуацию в ручке store, а не на этапе обработки колбеков от кладуна.
                try:
                    cls.get_live_photo_file(uid, photo_md5, photo_sha256, photo_size)
                except errors.LivePhotoMultipleFound:
                    pass

                # Проверяем также, если видео уже было загружено, то отдаем 409, чтобы не грузить его повторно
                size, md5, sha256 = odata['size'], odata['md5'], odata['sha256']
                Bus().check_photostream_live_photo_video_part(uid, size, md5, sha256)

        cls.prepare_arguments(uid, odata, **kw)

        try:
            path = odata['new_path']
        except KeyError:
            path = odata['path']

        odata['file_id'] = Resource.generate_file_id(uid, path)

        d = {
            'uid': uid,
            'path': path,
            'oid': odata['id'],
            'file-id': odata['file_id'],
            'service': 'disk',
        }
        if 'tld' in odata:
            d['tld'] = odata['tld']

        if 'free_space' in odata:
            d['free_space'] = odata['free_space']

        if 'upload-max-speed-bps' in odata and odata['upload-max-speed-bps']:
            d['upload-max-speed-bps'] = odata['upload-max-speed-bps']

        d.update(kw.get('d', {}))
        odata['kladun_data'] = d.copy()

        ext_upload_url, int_status_url = cls.post_request_to_kladun(d, **kw)
        odata['upload_url'] = ext_upload_url
        odata['status_url'] = int_status_url
        odata['state'] = codes.EXECUTING
        return super(StoreDisk, cls).Create(uid, odata, **kw)

    def kladun_data(self):
        """
        Возвращает данные, отправленные кладуну.

        "kladun_service" эти данные прохожят еще одну обработку см. "_preprocess_body_arguments"
        """
        return self.data['kladun_data'].copy()

    @classmethod
    def prepare_arguments(cls, uid, odata, **kw):
        cls.check_parent_rw(odata['path'])


class ExtractFileFromArchive(StoreDiskMixin, SaveOnDiskOperation):
    type = 'store'
    subtype = 'extract-file-from-archive'
    kladun = kladun_service.ExtractFileFromArchive()

    @classmethod
    def Create(classname, uid, odata, **kw):
        odata_required_keys = {'id', 'archive_addr', 'file_in_archive_path', 'dst_addr', 'connection_id'}
        missed_required_keys = odata_required_keys - odata.viewkeys()
        if missed_required_keys:
            raise KeyError("Missed required params: %s" % list(missed_required_keys))

        fs = Bus()
        archive_raw_addr = odata.pop('archive_addr').id
        dst_raw_addr = odata.pop('dst_addr').id
        fs.check_source(archive_raw_addr)
        fs.check_target(dst_raw_addr)
        fs.check_available_space(uid, dst_raw_addr)
        free_space = Quota().free_with_shared_support(address=Address(dst_raw_addr).get_parent())

        archive_resource = factory.get_resource(uid, archive_raw_addr)
        file_id = Resource.generate_file_id(uid, dst_raw_addr)

        mail_stid_service = MailStidService()
        if archive_resource.address.storage_name == 'mail':
            service_file_id = mail_stid_service.get_service_file_id(
                uid, archive_resource.meta['mid'], archive_resource.meta['hid']
            )
            source_service = 'mail2'
        elif archive_resource.address.storage_name == 'mulca':
            service_file_id = '%s:%s/%s' % (uid, archive_resource.meta['stid'], archive_resource.meta['part'])
            source_service = 'mail'
            # FIXME: raise NotImplementedError()?
        else:
            service_file_id = '%s:%s' % (archive_resource.uid, archive_resource.meta['file_mid'])
            source_service = 'mulca'
        kladun_data = {
            'uid': uid,
            'oid': odata['id'],
            'path': dst_raw_addr,
            'file-id': file_id,
            'source-service': source_service,
            'service-file-id': service_file_id,
            'file-to-extract': odata['file_in_archive_path'],
            'max-file-size': free_space
        }
        ext_upload_url, int_status_url = classname.post_request_to_kladun(kladun_data)
        odata['upload_url'] = ext_upload_url
        odata['status_url'] = int_status_url
        odata['state'] = codes.EXECUTING
        odata['path'] = dst_raw_addr
        odata['file_id'] = file_id
        operation = super(ExtractFileFromArchive, classname).Create(uid, odata, **kw)
        return operation

    @classmethod
    def prepare_arguments(classname, uid, odata, **kw):
        classname.check_parent_rw(odata['path'])


class OverwriteDisk(StoreDisk):
    type = 'overwrite'
    subtype = 'disk'
    keep_symlinks = True


class OverwriteShare(StoreDisk):
    type = 'overwrite'
    subtype = 'share'
    keep_symlinks = True


class StoreAttach(StoreDisk):
    type = 'store'
    subtype = 'attach'
    user_type = 'attach'

    @classmethod
    def Create(cls, uid, odata, **kw):
        address = Address(odata['path'])
        original_name = address.name

        address = address.rename('%s_%s' % (original_name, time.time()))
        odata['path'] = address.id
        odata['original_name'] = original_name

        # https://st.yandex-team.ru/CHEMODAN-31103
        # Если пользователь новый, то создаем стандартного.
        # Если пользователь стандартный, то ничего не делаем.
        # Если пользователь аттачевый, то апгрейдим его до стандартного.
        if user.NeedInit(uid):
            user.Create(uid, source='mail-attach')

        # AttachFile.Create умеет создавать аттачевый неймспейс, если он не существует (он зовётся в дебрях mkfile)

        operation = super(StoreAttach, cls).Create(uid, odata, **kw)
        return operation

    def additional_filedata(self):
        return {'original_name': self.data['original_name']}

    def post_process_all(self):
        set_public = self.data.get('set_public')
        if set_public is not False:
            Publicator().set_public(self.uid, self.data['path'],
                                    oper_type=self.type, oper_subtype=self.subtype,
                                    user_ip=self.data.get('user_ip', ''))
        self.set_completed()
        self.stat_log()

    def check_done(self):
        return False

    def check_completed(self):
        return False


class StoreClientFile(StoreDisk):
    type = 'store'
    subtype = CLIENT_AREA

    @classmethod
    def Create(cls, uid, odata, **kw):
        if not client_data.check(uid):
            client_data.create(uid)
            client_data.make_folder(uid, CLIENT_AREA_PATH, {})
        operation = super(StoreClientFile, cls).Create(uid, odata, **kw)
        return operation

    def post_process_new_file(self, new_resource):
        user_info = passport.userinfo(self.uid)
        login = user_info['login']
        sender_name = user_info.get('username', '')
        sender_email = self.data['fos_reply_email']
        # TODO: как обращаться к пользователю, какой-нибудь display_name?
        recipient_email_type = self.data['fos_recipient_type']

        os_version = self.data['fos_os_version']
        app_version = self.data['fos_app_version']
        fos_subject = self.data['fos_subject']
        support_text = self.data['fos_support_text']

        expire_seconds = self.data['fos_expire_seconds']
        download_url = zaberun.generate_public_url(
            mid=new_resource.file_mid(), filename=new_resource.name,
            expire_seconds=expire_seconds, owner_uid=new_resource.owner_uid
        )
        user_sender = User(self.uid)
        is_paid = user_sender.is_paid()
        subject = FOS_SUBJECT_TEMPLATE % {
            'os_version': os_version,
            'app_version': app_version,
            'fos_subject': fos_subject
        }
        body = FOS_BODY_TEMPLATE % {
            'support_text': support_text,
            'login': login,
            'uid': self.uid,
            'app_version': app_version,
            'os_version': os_version,
            'is_paid': is_paid,
            'download_url': download_url
        }
        data = {
            'email_to': FEEDBACK_FOS_EMAILS[recipient_email_type],
            'template_name': FEEDBACK_FOS_TEMPLATE,
            'sender_name': sender_name,
            'sender_email': sender_email,
            'template_args': {
                'subject': subject,
                'body': body,
            },
            'headers': {'x-otrs-paid': 'true' if is_paid else 'false'},
        }
        mpfs_queue.put(data, 'send_email')


class StoreYaShellSettings(StoreDisk):
    type = 'store'
    subtype = 'settings'
    user_type = 'settings'

    @classmethod
    def Create(classname, uid, odata, **kw):
        if user.NeedInit(uid, type=classname.user_type):
            user.Create(uid, type=classname.user_type)

        operation = super(StoreYaShellSettings, classname).Create(uid, odata, **kw)
        return operation


class OverwriteYaShellSettings(OverwriteDisk):
    type = 'overwrite'
    subtype = 'settings'
    keep_symlinks = True


class StorePhotostream(StoreDisk):
    type = 'store'
    subtype = 'photostream'
    test_login_re = re.compile(PROMO_AUTO_UPLOAD_32_GB['test_login_pattern'])
    ycrid_re = re.compile(PROMO_AUTO_UPLOAD_32_GB['ycrid_pattern'])

    @classmethod
    def prepare_arguments(classname, uid, odata, **kw):
        fs = Bus()
        path = fs.preprocess_path(uid, odata['path'])
        size, md5, sha256 = odata['size'], odata['md5'], odata['sha256']
        try:
            odata['new_path'] = fs.check_photostream_file_existing(
                uid, path, size, md5, sha256, is_live_photo=classname.is_live_photo_store(odata))
        except errors.LivePhotoUploadAttemptRegularPhotoExists:
            raise
        except errors.FileAlreadyExist:
            source_id = odata.get('source_id_to_add')
            if source_id:
                mpfs_queue.put({
                    'uid': uid,
                    'hid': FileChecksums(md5, sha256, int(size)).hid,
                    'source_ids': [source_id, ],
                    'is_live_photo': classname.is_live_photo_store(odata)},
                    'add_source_ids')
            raise

        if 'changes' in odata:
            odata['changes'].pop('mtime', None)

    @classmethod
    def _is_test_user(cls, user):
        """Для таких пользователей не проверяем временные границы действия акции"""
        user_info = user.get_user_info()
        login = user_info.get('login')
        if login and cls.test_login_re.search(login):
            return True
        else:
            return False

    @classmethod
    def _notify_about_promo_auto_upload_32_gb(cls, usr):
        ycrid = mpfs.engine.process.get_cloud_req_id()
        if ycrid is None or not cls.ycrid_re.search(ycrid):
            return
        now = time.time()

        if not(PROMO_AUTO_UPLOAD_32_GB['start_timestamp'] <= now <= PROMO_AUTO_UPLOAD_32_GB['end_timestamp'] or cls._is_test_user(usr)):
            log.info(
                'Promo has not started yet: %s <= %s <= %s.' % (
                    PROMO_AUTO_UPLOAD_32_GB['start_timestamp'], now, PROMO_AUTO_UPLOAD_32_GB['end_timestamp']
                )
            )
            return

        product = Product('32_gb_autoupload')
        assert product.singleton
        client = Client(usr.uid)
        service = simple_create_service(client=client, product=product)  # FIXME: Надо ли отправлять email?

        if not service:
            return
        else:
            log.info(
                'Service "32_gb_autoupload" created for %s.' % usr.uid
            )

        payload = {
            'uid': usr.uid,
            '_type': 'congratulations_2017_32gb',
            'service': 'disk',
            'actor': 'ya_disk',
            'meta': {
                'action': {
                    'type': 'action',
                    'action': 'GO_TO_TUNE_PAGE'
                },
                'mobile-link': {
                    'type': 'link',
                    'link': PROMO_AUTO_UPLOAD_32_GB['landing_pages']['default'],
                    'ru_link': PROMO_AUTO_UPLOAD_32_GB['landing_pages']['ru'],
                    'en_link': PROMO_AUTO_UPLOAD_32_GB['landing_pages']['en'],
                    'uk_link': PROMO_AUTO_UPLOAD_32_GB['landing_pages']['ua'],
                    'tr_link': PROMO_AUTO_UPLOAD_32_GB['landing_pages']['tr']
                }
            }
        }
        if PROMO_AUTO_UPLOAD_32_GB.get('preview'):
            payload['meta']['entity'] = {
                'type': 'preview',
                'preview': PROMO_AUTO_UPLOAD_32_GB['preview']
            }
        mpfs_queue.put(payload, 'notifier_add_notification')

    def _process(self, xml_data, stage):
        usr = user.User(self.uid)
        locale = usr.get_supported_locale()
        Bus().mkdir_photostream_root(self.uid, locale)
        super(StorePhotostream, self)._process(xml_data, stage)

        # https://st.yandex-team.ru/CHEMODAN-34568
        if self.is_kladun_first_callback(stage):
            try:
                self._notify_about_promo_auto_upload_32_gb(usr)
            except Exception:
                error_log.error(traceback.format_exc())

    def set_path(self, path):
        self.data['new_path'] = path

    def get_path(self):
        return self.data['new_path']


class StoreOperationService(ServiceUploadOperation):
    type = 'store'
    kladun = kladun_service.UploadToService()

    @classmethod
    def Create(classname, uid, odata, **kw):
        d = {
            'uid': uid,
            'oid': odata['id'],
            'service': Address(odata['path']).storage_name,
        }
        ext_upload_url, int_status_url = classname.post_request_to_kladun(d, **kw)
        odata['upload_url'] = ext_upload_url
        odata['status_url'] = int_status_url
        return super(StoreOperationService, classname).Create(uid, odata, **kw)


class StoreShare(StoreDisk):
    subtype = 'share'


class StoreNarod(StoreOperationService):
    subtype = 'narod'


class StoreNotes(StoreDisk):
    subtype = 'notes'


class StoreFotki(StoreOperationService):
    subtype = 'fotki'

    def _process(self, xml_data):
        super(StoreFotki, self)._process(xml_data)
        if self.check_data_uploaded():
            requested_path = Address(self.data['path'])
            parent = requested_path.get_parent()
            foto_path = parent.get_child_file(self.stages['publishing'].file_id)
            foto_resource = factory.get_resource(self.uid, foto_path)
            setprop_changes = {'title': requested_path.name}
            if not parent.is_storage:
                setprop_changes['newAlbumId'] = parent.name
            foto_resource.patch_file(setprop_changes, [])

    def get_status(self):
        self.update_status()
        if self.stages and 'publishing' in self.stages:
            parent_path = Address(self.data['path']).get_parent()
            if self.stages['publishing'].file_id:
                mpfs_file_path = parent_path.get_child_file(self.stages['publishing'].file_id)
                self.stages['publishing'].mpfs_file_id = mpfs_file_path.id
        return self.get_status_dict()


class StorePhotounlim(StoreDisk):
    subtype = PHOTOUNLIM_AREA

    @classmethod
    def prepare_arguments(classname, uid, odata, **kw):
        fs = Bus()
        path = fs.preprocess_path(uid, odata['path'])
        size, md5, sha256 = odata['size'], odata['md5'], odata['sha256']
        try:
            fs.check_photostream_file_existing(
                uid, path, size, md5, sha256, is_live_photo=classname.is_live_photo_store(odata))
        except errors.LivePhotoUploadAttemptRegularPhotoExists:
            raise
        except errors.FileAlreadyExist:
            source_id = odata.get('source_id_to_add')
            if source_id:
                mpfs_queue.put({
                    'uid': uid,
                    'hid': FileChecksums(md5, sha256, int(size)).hid,
                    'source_ids': [source_id, ],
                    'is_live_photo': classname.is_live_photo_store(odata)},
                    'add_source_ids')
            raise

        if 'changes' in odata:
            odata['changes'].pop('mtime', None)


class OverwritePhotounlim(StorePhotounlim):
    type = 'overwrite'
    keep_symlinks = True

    @classmethod
    def prepare_arguments(classname, uid, odata, **kw):
        fs = Bus()
        path = fs.preprocess_path(uid, odata['path'])
        size, md5, sha256 = odata['size'], odata['md5'], odata['sha256']

        if 'new_path' not in odata:
            try:
                fs.check_photostream_file_existing(
                    uid, path, size, md5, sha256, is_live_photo=classname.is_live_photo_store(odata),
                    replace_hid=odata.get('replace_hid')
                )
            except errors.LivePhotoUploadAttemptRegularPhotoExists:
                raise
            except errors.FileAlreadyExist:
                source_id = odata.get('source_id_to_add')
                if source_id:
                    mpfs_queue.put({
                        'uid': uid,
                        'hid': FileChecksums(md5, sha256, int(size)).hid,
                        'source_ids': [source_id, ],
                        'is_live_photo': classname.is_live_photo_store(odata)},
                        'add_source_ids')
                raise

        if 'changes' in odata:
            odata['changes'].pop('mtime', None)
        if 'replace_mtime' in odata:
            if 'changes' not in odata:
                odata['changes'] = {}
            odata['changes']['mtime'] = odata['replace_mtime']

    def set_path(self, path):
        self.data['new_path'] = path

    def get_path(self):
        return self.data['new_path']


class OverwritePhotostream(StorePhotostream):
    type = 'overwrite'
    subtype = PHOTOSTREAM_AREA
    keep_symlinks = True

    @classmethod
    def prepare_arguments(classname, uid, odata, **kw):
        fs = Bus()
        path = fs.preprocess_path(uid, odata['path'])
        size, md5, sha256 = odata['size'], odata['md5'], odata['sha256']

        if 'new_path' not in odata:
            try:
                odata['new_path'] = fs.check_photostream_file_existing(
                    uid, path, size, md5, sha256, is_live_photo=classname.is_live_photo_store(odata),
                    replace_hid=odata.get('replace_hid')
                )
            except errors.LivePhotoUploadAttemptRegularPhotoExists:
                raise
            except errors.FileAlreadyExist:
                source_id = odata.get('source_id_to_add')
                if source_id:
                    mpfs_queue.put({
                        'uid': uid,
                        'hid': FileChecksums(md5, sha256, int(size)).hid,
                        'source_ids': [source_id, ],
                        'is_live_photo': classname.is_live_photo_store(odata)},
                        'add_source_ids')
                raise

        if 'changes' in odata:
            odata['changes'].pop('mtime', None)
        if 'replace_mtime' in odata:
            if 'changes' not in odata:
                odata['changes'] = {}
            odata['changes']['mtime'] = odata['replace_mtime']

    def set_path(self, path):
        self.data['new_path'] = path

    def get_path(self):
        return self.data['new_path']
