# -*- coding: utf-8 -*-

import mpfs.engine.process

from mpfs.common import errors
from mpfs.common.util import hashed, normalize_string
from mpfs.core.address import Address
from mpfs.core.bus import Bus
from mpfs.core.filesystem.resources.disk import MPFSFolder
from mpfs.core.operations.base import DiskInternalOperation, set_source_resource_id
from mpfs.core.filesystem.helpers.lock import LockUpdateTimer
from mpfs.config import settings
from mpfs.core.yateam.logic import is_yateam_root, user_has_nda_rights, recreate_yateam_dir
error_log = mpfs.engine.process.get_error_log()


SYSTEM_SYSTEM_FILESYSTEM_LOCK_AUTOUPDATE_PERIOD = settings.system['system']['filesystem_lock_autoupdate_period']
FEATURE_TOGGLES_USE_FILESYSTEM_LOCK_AUTOUPDATE_PERIOD = \
    settings.feature_toggles['use_filesystem_lock_autoupdate_period']
FEATURE_TOGGLES_TRASH_DROP_ALL_LOCK_CHECK = settings.feature_toggles['trash_drop_all_lock_check']


class TrashOperation(object):

    def get_uniq_id(self):
        return hashed(str(self.uid) + str(self.type) + normalize_string(self.data['path']))

    @classmethod
    def Create(cls, uid, odata, **kw):
        source_id = odata.get('path', '/trash')
        resource = Bus().get_resource(uid, source_id)
        odata['file_id'] = resource.meta.get('file_id', '')
        set_source_resource_id(resource, odata)
        return super(TrashOperation, cls).Create(uid, odata, **kw)


class TrashDrop(TrashOperation, DiskInternalOperation):
    type = 'trash'
    subtype = 'drop'

    def _process(self, *args, **kwargs):
        fs = Bus(connection_id=self.data['connection_id'], async=True)
        timer = LockUpdateTimer(addresses=(Address.Make(self.uid, '/trash').id,))

        try:
            timer.start()
            try:
                fs.trash_drop_all(self.uid, self.uid, self.key, lock_data={'oid': self.id})
            except errors.ResourceCheckLockFailed:
                if FEATURE_TOGGLES_TRASH_DROP_ALL_LOCK_CHECK:
                    # не фейлим операцию и не прокидываем исключение дальше
                    self.reenque(SYSTEM_SYSTEM_FILESYSTEM_LOCK_AUTOUPDATE_PERIOD)
            else:
                self.set_completed()
        finally:
            timer.stop()

    def process_interruption(self):
        address = Address.Make(self.uid, '/trash').id
        self.release_lock(address)

    def get_uniq_id(self):
        return hashed(self.uid + self.type + self.subtype)


class TrashAppend(TrashOperation, DiskInternalOperation):
    type = 'trash'
    subtype = 'append'

    def _process(self, *args, **kwargs):
        bus = Bus(connection_id=self.data['connection_id'], async=True)
        tgt_address = bus.get_trash_address(self.data['path'])
        timer = LockUpdateTimer(addresses=(self.data['path'], tgt_address.id))

        try:
            timer.start()
            affected_resource = bus.trash_append(self.uid, self.data['path'],
                                                 lock_data={'oid': self.id}, tgt_address=tgt_address)
            if is_yateam_root(Address(self.data['path']).path) and user_has_nda_rights(self.uid):
                recreate_yateam_dir(self.uid)
            self.data['affected_resource'] = affected_resource.get('this', {}).get('id')
            self.set_completed()
        finally:
            timer.stop()

    def process_interruption(self):
        address = self.data['path']
        self.release_lock(address)


class TrashAppendBulk(TrashAppend):
    subtype = 'append_bulk'

    def get_uniq_id(self):
        return hashed(str(self.uid) + str(self.type) + ';'.join(self.data['paths']))

    def _process(self, *args, **kwargs):
        bus = Bus(connection_id=self.data['connection_id'], async=True)

        for path in self.data['paths']:
            timer = LockUpdateTimer(addresses=(path,))
            try:
                timer.start()
                try:
                    bus.trash_append(self.uid, path, lock_data={'oid': self.id})
                except errors.ResourceNotFound:
                    continue
                if is_yateam_root(Address(path).path) and user_has_nda_rights(self.uid):
                    recreate_yateam_dir(self.uid)
            finally:
                timer.stop()
        self.set_completed()

    def process_interruption(self):
        for path in self.data['paths']:
            self.release_lock(path)


class TrashDropPath(TrashOperation, DiskInternalOperation):
    type = 'trash'
    subtype = 'droppath'

    def _process(self, *args, **kwargs):
        fs = Bus(connection_id=self.data['connection_id'], async=True)
        timer = LockUpdateTimer(addresses=(self.data['path'],))

        try:
            timer.start()
            fs.trash_drop_element(self.uid, self.data['path'], lock_data={'oid': self.id})
            self.set_completed()
        finally:
            timer.stop()

    def process_interruption(self):
        address = self.data['path']
        self.release_lock(address)


class TrashRestore(TrashOperation, DiskInternalOperation):
    type = 'trash'
    subtype = 'restore'

    def _is_restore_folder_after_interruption(self, bus):
        if self.data.get('was_interrupted', False):
            # https://st.yandex-team.ru/CHEMODAN-28742
            # если операция была прервана и мы пытаемся ее продолжить - проверяем, если восстанавливали папку,
            # то проставляем флажок force, чтобы восстановление продолжилось в ту же папку, а не создавалась
            # новая с суффиксом
            resource = bus.get_resource(self.uid, self.data['path'])

            if isinstance(resource, MPFSFolder):
                return True
        return False

    def _process(self, *args, **kwargs):
        bus = Bus(connection_id=self.data['connection_id'], async=True)
        force = self.data['force'] or self._is_restore_folder_after_interruption(bus)
        tgt_address = bus.get_trash_restore_tgt_address(
            self.uid,
            self.data['path'],
            new_name=self.data['name'],
            force=force,
        )
        timer = LockUpdateTimer(addresses=(self.data['path'], tgt_address.id))

        try:
            timer.start()
            affected_resource = bus.trash_restore(
                self.uid,
                self.data['path'],
                new_name=self.data['name'],
                force=force,
                tgt_address=tgt_address,
                lock_data={'oid': self.id}
            )
            self.data['affected_resource'] = affected_resource
            self.set_completed()
        finally:
            timer.stop()

    def process_interruption(self):
        address = self.data['path']
        self.release_lock(address)

        self.data['was_interrupted'] = True
        self.save()

    def check_arguments(self):
        """Проверить, что исходный ресурс не залочен и существует
        """
        fs = Bus(connection_id=self.data['connection_id'], async=True)
        try:
            fs.check_lock(self.data['path'])
            fs.info(self.uid, self.data['path'])
        except (errors.ResourceNotFound, errors.ResourceLocked):
            raise
