# -*- coding: utf-8 -*-
"""

MPFS
CORE

Операции для работы с инвайтами

"""
import traceback

from copy import deepcopy

import mpfs.engine.process
import mpfs.invite.events as invite_events
from mpfs.core.db_rps_limiter import InDBRPSLimiter

from mpfs.config import settings
from mpfs.core.operations.social import SendMessage, ListContacts, Operation
from mpfs.core.metastorage.control import invite_mpfs_sent as sent
from mpfs.core.services.passport_service import Passport
from mpfs.core.services.directory_service import DirectoryService, DirectoryContact
from mpfs.core.social.share import ShareProcessor
from mpfs.core.social.share.invite import get_invite_hash
from mpfs.core.social.share.link import LinkToGroup
from mpfs.common.errors.share import ShareError, ShareNotFound
from mpfs.common.errors import StorageInitUser
from mpfs.common.util import mailer, email_valid


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
passport = Passport()
directory_service = DirectoryService()


DEFAULT_LOCALE = settings.user['default_locale']


class SendInvite(SendMessage):
    '''
    Отправка инвайта
    '''

    type = 'invites'
    subtype = 'send'

    def set_completed(self):
        super(SendInvite, self).set_completed()
        sent.put({
            'uid': self.uid,
            'p'  : self.data.get('provider'),
            'u'  : self.data.get('userid')
        })
        invite_events.InviteSentEvent(uid=self.uid, address=self.data.get('userid'),
                                           project='mpfs', provider=self.data.get('provider')).send()


class SendSharedInvite(SendMessage):

    type = 'shared_invites'
    subtype = 'send'
    ratelimit = InDBRPSLimiter('shared_invites')

    def _send_email(self, uid, provider, userid, template, params, headers=None):
        userinfo = passport.userinfo(login=userid.split('@')[0])
        # В тестинге не пытаемся послать письма
        if (settings.feature_toggles['disable_sharing_invite_emails'] or
                settings.feature_toggles['mobile_sharing_invite_email'] and userinfo['has_mobile_disk']):
            self.set_completed()
        elif self.ratelimit.check_limit_and_increment_counter(uid):
            is_spam = self.data.get('so_is_spam', False)
            if not is_spam:
                so_receipt = self.data.get('so_receipt')
                if so_receipt:
                    headers = {"X-Yandex-CF-Receipt": so_receipt}
                super(SendSharedInvite, self)._send_email(uid, provider, userid, template, params, headers)
            else:
                self.set_completed()

    def set_completed(self):
        super(SendSharedInvite, self).set_completed()
        from mpfs.core.social.share import ShareProcessor

        share_processor = ShareProcessor(connection_id=self.data['connection_id'])
        result = share_processor.create_group_invite(self.data)

        if self.data.get('auto_accept'):
            # Заглушка для AttributeError
            share_processor.request = None

            uid = self.data['user_uid']
            hash_ = result['hash']
            share_processor.activate_group_invite(uid, hash_)


class SendSharedGroupInvite(Operation):

    type = 'invites'
    subtype = 'invite_group_contacts'

    def _send_email(self, email, uid, invite_hash, gid, inviter_userinfo):
        if not SendSharedInvite.ratelimit.check_limit_and_increment_counter(self.uid):
            log.info('UID %s send to many emails with invites.' % self.uid)
            return

        template = 'sharedFolder/access'

        user_info = passport.userinfo(uid)
        params = {
            'rights': str(self.data['rights']),
            'folderName': self.data['folder_name'],
            'hash': invite_hash,
            'friendName': inviter_userinfo['public_name'],
            'friendEmail': inviter_userinfo['decoded_email'] or inviter_userinfo['email'],
            'friendLogin': inviter_userinfo['login'],
            'locale': user_info['language'],
            'gid': gid,
        }

        try:
            mailer.send(email, template, params,
                        sender_email=inviter_userinfo['email'], sender_name=inviter_userinfo['public_name'])
        except Exception:
            error_log.exception('failed to send invite email to user with email %s' % email)

    @staticmethod
    def _check_already_accepted_invite(gid, uid):
        link_args = {
            'gid': gid,
            'uid': uid,
        }

        try:
            LinkToGroup.load(**link_args)
        except ShareNotFound:
            return False

        return True

    def _process(self, *args, **kwargs):
        entity = self.data['entity']

        contacts = []
        if email_valid(entity):
            user_info = passport.userinfo(login=entity)
            contacts.append({'uid': user_info['uid'], 'email': entity})
        else:
            org_id, entity_type, entity_id = directory_service.decode_group_entity(entity)
            if entity_type == 'department':
                contacts = directory_service.get_department_contacts(org_id, entity_id, self.data['user_ip'])
            elif entity_type == 'group':
                contacts = directory_service.get_group_contacts(org_id, entity_id, self.data['user_ip'])

        share_processor = ShareProcessor(connection_id=self.data['connection_id'])
        gid = self.data['gid']
        inviter_userinfo = passport.userinfo(self.uid)

        for contact in contacts:
            if isinstance(contact, DirectoryContact):
                contact_uid = contact.uid
                email = contact.email
            else:
                contact_uid = str(contact['uid'])
                email = contact['email']
            if contact_uid == self.uid:
                continue  # для себя приглашение не отправляем

            if self._check_already_accepted_invite(gid, contact_uid):
                continue  # если пользователь уже принял приглашение - второй раз не отправляем

            # такой же, как и для обычных инвайтов, они должны пересекаться, иначе будем приглашать пользователей
            # по несколько раз
            invite_hash = get_invite_hash(gid, email, 'ya_directory')

            username = ''
            avatar = ''
            try:
                user_info = passport.userinfo(contact_uid)
                username = user_info['public_name']
                avatar = user_info['avatar']
            except Exception:
                error_log.exception('failed to fetch info for user %s' % contact_uid)

            data = deepcopy(self.data)
            data['provider'] = 'ya_directory'
            data['userid'] = email
            data['user_avatar'] = avatar
            data['user_name'] = username
            data['hash'] = invite_hash

            # https://st.yandex-team.ru/CHEMODAN-75427 Возвращаем None, чтобы писался не строкой в базу, а NULL
            if contact_uid == "None":
                contact_uid = None
            data['user_uid'] = contact_uid

            try:
                share_processor.create_group_invite(data)
                if self.data.get('auto_accept'):
                    share_processor.request = None  # Заглушка для AttributeError
                    share_processor.activate_group_invite(contact_uid, invite_hash)
            except (ShareError, StorageInitUser):
                error_log.exception('failed to share with gid %s for contact %s' % (gid, contact_uid))

            if not self.data.get('auto_accept'):
                self._send_email(email, contact_uid, invite_hash, gid, inviter_userinfo)

        self.set_completed()


class ListInviteContacts(ListContacts):
    '''
    Получение списка друзей со статусами отправленности инвайта
    '''
    type = 'invites'
    subtype = 'list_contacts'

    def _get_sent_invites(self, provider):
        try:
            return self._sent_invites_dict.get(provider, [])
        except Exception:
            result = {}
            for item in sent.get_all(uid=self.uid):
                prov, userid = item['p'], str(item['u'])

                if userid.isdigit():
                    userid = int(userid)

                if prov in result:
                    result[prov].append(userid)
                else:
                    result[prov] = [userid]
            setattr(self, '_sent_invites_dict', result)
            return result.get(provider, [])

    def format_contacts(self, contacts, provider=None):
        '''
        Форматируем контакты так, чтобы в них был признак высланности инвайта
        '''
        if not provider:
            return super(ListInviteContacts, self).format_contacts(contacts)

        sent_invites = self._get_sent_invites(provider)
        for item in contacts:
            if item.get('userid') in sent_invites:
                item['sent'] = 1

        return list({'user': item} for item in contacts)
