# -*- coding: utf-8 -*-
import time
import traceback

import mpfs.engine.process

from mpfs.core import factory
from mpfs.common import errors
from mpfs.common.static import codes
from mpfs.common.static import tags
from mpfs.config import settings
from mpfs.core.address import Address
from mpfs.core.bus import Bus
from mpfs.core.operations.base import Operation

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

MAIL_ATTACHES_REENQUEUE_DELAY = settings.operations['mail_attaches']['reenqueue_delay']
MAIL_ATTACHES_TTL = settings.operations['mail_attaches']['ttl']


class ImportMailAttaches(Operation):
    type = 'mail_attaches'
    subtype = 'import_mail_attaches'

    def get_status(self):
        result = super(ImportMailAttaches, self).get_status()
        result[tags.PROTOCOL] = self.data[tags.PROTOCOL]
        return result

    @classmethod
    def Create(cls, uid, odata, **kw):
        parent_folder = factory.get_resource(uid, odata['path'])
        parent_folder.check_rw()
        odata[tags.IS_FIRST_EXECUTION] = True
        odata[tags.PROTOCOL] = []
        return super(ImportMailAttaches, cls).Create(uid, odata, **kw)

    def update_status(self):
        if not self.is_executing():
            return

        from mpfs.core.operations import manager
        has_failed_suboperations = False
        has_incomplete_suboperations = False
        has_changed_data = False
        for operation_meta in self.data[tags.PROTOCOL]:
            if operation_meta[tags.STATUS] in (codes.WAITING, codes.EXECUTING):
                new_status = manager.get_operation(self.uid, operation_meta[tags.OID]).state
                if new_status != operation_meta[tags.STATUS]:
                    has_changed_data = True
                operation_meta[tags.STATUS] = new_status
            if operation_meta[tags.STATUS] == codes.FAILED:
                has_failed_suboperations = True
            elif operation_meta[tags.STATUS] in (codes.WAITING, codes.EXECUTING):
                has_incomplete_suboperations = True
        if has_failed_suboperations:
            self.set_failed({'message': 'suboperation failed'})
            error_log.warn('Some suboperations failed oid=%s' % self.id)
        elif not has_incomplete_suboperations:
            self.set_completed()
        if has_changed_data:
            self.save()

    def _get_dest_addresses(self, items):
        dest_folder_address = Address(self.data['path'])
        dest_addresses = [dest_folder_address.get_child(item['file_name']) for item in items]
        if self.data['autosuffix']:
            dest_addresses = Bus().autosuffix_addresses(dest_addresses)
        elif not self.data['force']:
            for addr in dest_addresses:
                if Bus().exists(self.uid, addr.id):
                    raise errors.FileAlreadyExist()
        return dest_addresses

    def _initiate_copy(self):
        from mpfs.core.operations import manager

        items = self.data['items']
        dest_addresses = self._get_dest_addresses(items)
        free_space = self.data['free_space']
        operations = []
        for item, addr in zip(items, dest_addresses):
            odata = dict(
                target=addr.id,
                source_mid=item['mid'],
                source_hid=item['hid'],
                free_space=free_space,
                force=self.data['force'],
                callback='',
                connection_id='',
                autosuffix=False,
            )
            try:
                operation = manager.create_operation(self.uid, 'copy', 'import_mail_attach', odata)
            except Exception:
                operation_meta = {tags.OID: None, tags.STATUS: codes.FAILED, tags.RESULT: None}
                error_log.warn('Error while creating subop for import_mail_attaches (oid=%s): %s' %
                               (self.id, traceback.format_exc()))
                self.set_failed(({'message': 'failed to create suboperation'}))
                break
            else:
                operation_meta = {tags.OID: operation.id, tags.STATUS: operation.state, tags.RESULT: None}
            finally:
                operations.append(operation_meta)

        self.data[tags.PROTOCOL] = operations
        self.save()

    def reenque(self, delay=MAIL_ATTACHES_REENQUEUE_DELAY):
        super(ImportMailAttaches, self).reenque(delay)

    def _process(self):
        if self.data[tags.IS_FIRST_EXECUTION]:
            self._initiate_copy()
            self.data[tags.IS_FIRST_EXECUTION] = False
            self.save()
            self.reenque()
            log.info('Started import_mail_attaches for %s attaches' % len(self.data['items']))
        else:
            if time.time() - self.ctime > MAIL_ATTACHES_TTL:
                self.set_failed({'message': 'Operation in EXECUTING state more than TTL: %d sec' % MAIL_ATTACHES_TTL})
                return
            self.update_status()
            if self.state == codes.EXECUTING:
                self.reenque()
