# -*- coding: utf-8 -*-

"""
MPFS
CORE

Менеджер Операций

"""
import os
import time
import inspect
import collections
import traceback

import mpfs.engine.process
import mpfs.engine.queue2.queue_log
import mpfs.common.errors as errors
import mpfs.common.static.codes as codes
import mpfs.common.static.messages as messages
import mpfs.common.util.generator as generator
import mpfs.core.factory as factory
import mpfs.core.office.operations
import mpfs.core.wake_up.operations
import mpfs.core.social.share.operations
import mpfs.core.file_recovery.operations
import mpfs.core.support.operations

from mpfs.common.static import tags
from mpfs.config import settings
from mpfs.core.address import Address
from mpfs.core.filesystem.resources.photounlim import is_converting_address_to_photounlim_for_uid_needed
from mpfs.core.operations.dao.operation import OperationDAO
from mpfs.core.operations.filesystem import copy, bulk, move, store, dstore, remove, trash
from mpfs.core.operations import social, invites, legacy, aviary, system, download, user, mail_attaches
from mpfs.core.metastorage.control import groups, group_links
from mpfs.core.metastorage.control import operations
from mpfs.core.operations.util import OperationPinger
from mpfs.core.queue import mpfs_queue
from mpfs.core.operations.events import OperationCreatedEvent
from mpfs.core.user.constants import PHOTOUNLIM_AREA


OPERATIONS_ENQUEUE_ROBUST = settings.operations['enqueue_robust']
OPERATIONS_TYPES = settings.operations['types']
OPERATIONS_FAIL_NOT_SCHEDULED_WITH_QUELLER_OPERATIONS = settings.operations['fail_not_scheduled_with_queller_operations']

QUEUE2_CREATING_OPERATIONS_LIMITATION_ENABLED = settings.queue2['creating_operations_limitation']['enabled']
QUEUE2_CREATING_OPERATIONS_LIMITATION_DRY_RUN = settings.queue2['creating_operations_limitation']['dry_run']
QUEUE2_CREATING_OPERATIONS_LIMITATION_LIMITED_UIDS = settings.queue2['creating_operations_limitation']['limited_uids']
QUEUE2_CREATING_OPERATIONS_LIMITATION_LIMITED_TYPES = settings.queue2['creating_operations_limitation']['limited_types']
QUEUE2_CREATING_OPERATIONS_LIMITATION_MAX_ACTIVE_OPERATIONS = settings.queue2['creating_operations_limitation']['max_active_operations']


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


classes = collections.defaultdict(dict)
for op_module in (copy, bulk, move, store, dstore, remove, trash, social, invites, legacy, aviary, system, download,
                  mpfs.core.office.operations, user, mpfs.core.social.share.operations,
                  mpfs.core.file_recovery.operations, mpfs.core.wake_up.operations, mail_attaches,
                  mpfs.core.support.operations):
    for cls_name, cls in filter(
            lambda (c_name, c): hasattr(c, 'subtype') and c.subtype,
            inspect.getmembers(op_module, inspect.isclass)
    ):
        classes[cls.type][cls.subtype] = cls

paths = {}
for state in codes.operation_states:
    paths[state] = messages.old_operation_titles[state] + os.path.sep


def load(uid, oid):
    """
    Метод загрузки данных произвольной операции
    """
    key = oid
    resp = operations.show(uid, key)

    if not resp.value:
        raise errors.OperationNotFound('%s %s' % (str(uid), oid))

    loaded = resp.value

    return loaded


def search(uid, query_args):
    """
    Метод поиска операции
    """
    result = operations.find(uid, None, query_args, None, None, None)

    return result


def _get_classname(type, subtype):
    """
    Возвратит нужный classname для указанного типа
    Если не найдет - бросит OperationNotFound
    """
    try:
        classname = classes[type][subtype]
        return classname
    except KeyError:
        error_log.error("OperationClassNotFound: %s:%s" % (type, subtype))
        raise errors.OperationClassNotFound('%s:%s' % (type, subtype))


def get_operation(uid, oid, data={}, allow_unknown_type=False):
    """
    Получение операции
    """
    if not data:
        found_operation = load(uid, oid)
        data = found_operation.data
        version = found_operation.version
        key = found_operation.key
        data['id'] = key
    else:
        version = None
        key = data.get('id')

    if 'subtype' not in data:
        raise errors.OperationNotFound()

    try:
        classname = _get_classname(data['type'], data['subtype'])
    except errors.OperationClassNotFound:
        if allow_unknown_type:
            from mpfs.core.operations.base import UnknownOperation
            classname = UnknownOperation
        else:
            raise

    operation = classname(**data)
    operation.version = version
    operation.key = key
    return operation


def generate_operation_id(uid):
    return generator.sha256(uid, time.time())


def enqueue_or_fail_operation(operation, robust=False, stime=None):
    state_before_put = operation.state
    uid = operation.uid
    oid = operation.id
    jtype = operation.task_name
    op_type = operation.type
    try:
        mpfs_queue.put({'oid': oid, 'uid': uid}, jtype, stime=stime, robust=robust, op_type=op_type)
    except Exception as e:
        # операция могла поставиться и её статус мог измениться, несмотря на исключение, тогда её фейлить не надо
        operation = get_operation(uid, oid)
        if (operation.state in (codes.WAITING, codes.EXECUTING) and operation.state == state_before_put
                and OPERATIONS_FAIL_NOT_SCHEDULED_WITH_QUELLER_OPERATIONS):
            operation.set_failed({'message': 'Failed to put operation to queue: %s' % e})
        log.info('Failed to put %s to queue (uid=%s, oid=%s): %s' % (jtype, uid, oid, e))
        raise


def check_active_operations_limit(uid):
    if not QUEUE2_CREATING_OPERATIONS_LIMITATION_ENABLED:
        return

    active_operations_count = count_operations_by_status_and_type(
        uid,
        types=QUEUE2_CREATING_OPERATIONS_LIMITATION_LIMITED_TYPES,
        states=[codes.EXECUTING, codes.WAITING],
        limit=QUEUE2_CREATING_OPERATIONS_LIMITATION_MAX_ACTIVE_OPERATIONS + 1,
    )
    if active_operations_count >= QUEUE2_CREATING_OPERATIONS_LIMITATION_MAX_ACTIVE_OPERATIONS:
        if (QUEUE2_CREATING_OPERATIONS_LIMITATION_DRY_RUN and
                uid in QUEUE2_CREATING_OPERATIONS_LIMITATION_LIMITED_UIDS or
                not QUEUE2_CREATING_OPERATIONS_LIMITATION_DRY_RUN):
            log.info(
                'Executing operations count limit exceeded for uid `%s` (%d >= %d)',
                uid, active_operations_count, QUEUE2_CREATING_OPERATIONS_LIMITATION_MAX_ACTIVE_OPERATIONS
            )
            raise errors.TooManyExecutingOperationsError(uid)

        # dry-run для всех кто не в списке лимитированных
        log.info(
            'Active operations limitation dry-run: uid %s should be limited with currently %s operations in EXECUTING or WAITING state' %
            (uid, active_operations_count)
        )


def create_operation(uid, type, subtype, odata, **kw):
    """
    Создание нового объекта операции
    """
    classname = _get_classname(type, subtype)
    if 'id' not in odata:
        odata['id'] = generate_operation_id(uid)

    if classname.is_pended():
        check_active_operations_limit(uid)

    try:
        odata[tags.AT_VERSION] = mpfs.engine.process.usrctl().version(uid)
    except errors.StorageInitUser:
        odata[tags.AT_VERSION] = 0

    op = classname.Create(uid, odata, **kw)

    if op.pended:
        enqueue_or_fail_operation(op, robust=OPERATIONS_ENQUEUE_ROBUST)
        log.info("created pended operation %s:%s for task %s" % (uid, op.id, op.task_name))
    else:
        log.info("created operation %s:%s" % (uid, op.id))

    OperationCreatedEvent(operation=op, connection_id=odata.get('connection_id', '')).send()

    return op


def delete_operation(uid, oid):
    """
    Удаление операции
    """
    operation = get_operation(uid, oid)
    operation.remove()
    del operation
    log.info("deleted operation %s:%s" % (uid, oid))
    return


def process_operation(uid, oid, *args, **kwargs):
    """
    Обработка операции
    """
    operation = get_operation(uid, oid)
    mpfs.engine.queue2.queue_log.OperationJobBinding.bind_operation(operation)

    if operation.is_waiting():
        operation.set_executing()

    if operation.is_failed() or operation.is_completed():
        log.info('operation %s:%s is already completed or failed' % (uid, oid))
        return operation

    with OperationPinger(operation):
        log.info('processing operation %s:%s' % (uid, oid))
        operation.process(*args, **kwargs)

        if operation.is_completed():
            log.info('operation %s:%s completed' % (uid, oid))
        elif operation.is_done():
            log.info('operation %s:%s main stages done' % (uid, oid))

        if operation.is_executing():
            log.info('operation %s:%s not done' % (uid, oid))

        if operation.is_rejected():
            log.info('operation %s:%s rejected' % (uid, oid))

        if operation.is_failed():
            error = operation.error()
            log.info("operation %s:%s failed" % (uid, oid))
            error_log.error("operation %s:%s failed: %s" % (uid, oid, str(error)))
            if error.get('code') == codes.CODE_ERROR:
                # send email
                pass
    return operation


def search_operation(uid, query_args={}):
    return search(uid, query_args)


def get_opened_store_operation(uid, path, md5, user_agent, **data):
    """
    Частный случай поиска открытой операции загрузки
    """
    query_args = {
        'state': [codes.WAITING, codes.EXECUTING],
        'md5': md5,
    }
    address = Address(path)
    paths = [path]
    if address.storage_name == 'photostream':
        address.change_storage(PHOTOUNLIM_AREA)
        paths.append(address.id)

    for odata in search_operation(uid, query_args):
        operation = get_operation(uid, odata['id'])
        # ищем в двух местах в случае с автозагрузкой,
        # так как точно определить куда лил пользователь файл (в лимит или безлимит)
        # здесь не представляется возможным из-за нескольких настроек безлимита
        if operation.data['path'] in paths:
            operation.data.update(data)
            operation.save()
            return operation
    raise errors.OperationNotFound()


def get_executing_operations(uid):
    """
    Ищет все активные операции пользователя
    """
    for data in search(uid, {'state': [codes.EXECUTING]}):
        yield data


def _get_active_operations(uid):
    """
    Ищет все активные операции пользователя
    """
    for data in search(uid, {'state': [codes.EXECUTING, codes.WAITING]}):
        yield data


def count_operations_by_status_and_type(uid, types, states, limit=None):
    """Подсчитывает количество операций.
    Можно задать лимит.
    """
    return OperationDAO().get_count_by_types_and_states(uid, types=types, states=states, limit=limit)


def get_active_operations(uid, show_hidden=False):
    """
    Ищет все активные операции пользователя
    """
    query_args = {
        'state': [codes.WAITING, codes.EXECUTING],
    }

    result = []

    for data in _get_active_operations(uid):
        type = data.get('type')
        subtype = data.get('subtype')

        if subtype == 'restore':
            try:
                resource = factory.get_resource(data.get('uid'), data.get('data', {}).get('path'))
                data['data']['target'] = resource.original_id
            except Exception:
                error_log.error(traceback.format_exc())

        if not show_hidden and subtype in OPERATIONS_TYPES.get(type, {}).get('hidden', []):
            continue

        result.append(data)

    return result


def get_unfinished_operations_by_uid(uid):
    """
    Найти все активные и незавершенные операции пользователя плюс операции пользователей, которые участвуют в общих
    папках этого пользователя.
    Возвращает map вида <uid: list[operation]>
    """
    active_operations_by_uid = {}
    uids = {uid}

    query = {'state': [codes.WAITING, codes.EXECUTING, codes.DONE]}

    self_groups = groups.get_all(owner=uid)
    for group in self_groups:
        links = group_links.get_all(gid=group['_id'])
        uids = uids.union({x['uid'] for x in links})

    for current_uid in uids:
        active_operations = list(search(current_uid, query))
        if active_operations:
            active_operations_by_uid[current_uid] = active_operations

    return active_operations_by_uid


def fail_operations(uid):
    """
    Зафейлить активные операции пользователя
    """

    for data in get_active_operations(uid):
        op = get_operation(uid, data['id'], data=data)
        op.set_failed({'message': 'Operation failed manually'})


def emergency_set_failed(uid, oid):
    """
    Зафэйлить операцию в аварийном режиме не раззиповывая её контент и не указывая причину прерывания операции.
    """
    operations.update({'uid': uid, '_id': oid}, {'$set': {'state': codes.FAILED}})
