# -*- coding: utf-8 -*-
"""

MPFS
CORE

Операции
для работы с социальными сервисами

"""
import time
import traceback

import mpfs.engine.process

from mpfs.config import settings
from mpfs.common.static import codes
from mpfs.common.static.tags import SUCCESS, FAILURE, STATE, RESULT, REASON, TYPE, STATUS, ERROR, STAGES, PROTOCOL, OID, LOCALE, IS_FIRST_EXECUTION, PROVIDER_DIR, PROVIDER, ID, CONNECTION_ID, IMPORT_TYPE
from mpfs.core.address import Address
from mpfs.core import factory
from mpfs.core.bus import Bus
from mpfs.core.user.base import User
from mpfs.core.user.constants import DEFAULT_FOLDERS, SOCIAL_PROVIDERS_DIRS, PHOTOS_OF_ME_LOCALIZED, ODNOKLASSNIKI_PERSONAL_PHOTOS, VK_WALL_PHOTOS, VK_PROFILE_PHOTOS, PHOTOS_OF_ME_MAILRU, MAILRU_PROFILE_PHOTOS, VK_SAVED_PHOTOS
from mpfs.core.operations.base import Operation, UploadOperation
from mpfs.core.queue import mpfs_queue
from mpfs.core.services.abook_service import AbookService
from mpfs.core.services import passport_service
from mpfs.core.services.socialproxy_service import SocialProxy
from mpfs.core.services import kladun_service
from mpfs.common.util.templater import TemplateProcessor
from mpfs.common.util import mailer, to_json, logger
from mpfs.common import errors
from mpfs.core.albums.models import Album

OPERATIONS_SOCIAL_SUB_OPER_LIFE_TIME = settings.operations['social']['sub_oper_life_time']

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
stat_social_log = logger.get('stat-social')

socialproxy = SocialProxy()
abook = AbookService()
passport = passport_service.Passport()


class SendMessage(Operation):

    '''
    Отправка сообщения в произвольную социальную сеть, включая email
    '''

    type = 'social'
    subtype = 'send_message'

    def __init__(self, **data):
        super(SendMessage, self).__init__(**data)
        self.stages = []

    def _send_email(self, uid, provider, userid, template, params, headers=None):
        try:
            abook.add_email(uid, userid)
        except Exception:
            error_log.info("Error adding contact to abook")
            error_log.info(traceback.format_exc())
        userinfo = passport.userinfo(uid)
        mailer.send(userid, template, params, sender_email=userinfo['email'], sender_name=userinfo['public_name'], headers=headers)
        self.set_completed()

    def _send_social(self, uid, provider, userid, template, params):
        message = TemplateProcessor(template, params).process('social')
        self.data['social_task'] = socialproxy.send_message(uid, provider, userid, message)
        self.save()

    @classmethod
    def Create(classname, uid, odata, **kw):
        provider = odata.get('provider')
        template = odata.get('template')
        userid   = odata.get('userid')
        params   = odata.pop('params')

        operation = super(SendMessage, classname).Create(uid, odata, **kw)

        providers = {
            'email'    : operation._send_email,
            'facebook' : operation._send_social,
        }

        '''
        https://jira.yandex-team.ru/browse/CHEMODAN-8045
        providers = {
            'email'    : operation._send_email,
            'facebook' : operation._send_social,
            'vkontakte': operation._send_social,
        }
        '''

        try:
            call_sending_method = providers[provider]
        except Exception:
            raise errors.SocialWrongProviderError(provider)

        call_sending_method(uid, provider, userid, template, params)
        return operation

    def get_status(self):
        if not self.is_completed() and not self.is_failed():
            self.update_status()
        return super(SendMessage, self).get_status()

    def update_status(self):
        task = self.data.get('social_task')
        if not task:
            return

        result = socialproxy.check_message(task)
        if result.get('state') == SUCCESS:
            self.set_completed()
            log.info('delivery success: from %s to %s %s, template %s' % (
                self.uid,
                self.data.get('provider'),
                self.data.get('userid'),
                self.data.get('template')
            ))
        elif result.get('state') == FAILURE:
            self.set_failed({'message': result.get('reason')})
            log.info('delivery failed: from %s to %s %s, template %s' % (
                self.uid,
                self.data.get('provider'),
                self.data.get('userid'),
                self.data.get('template')
            ))

class ListContacts(Operation):

    type = 'social'
    subtype = 'list_contacts'

    def __init__(self, **data):
        super(ListContacts, self).__init__(**data)
        self.stages = []

    @classmethod
    def Create(classname, uid, odata, **kw):
        operation = super(ListContacts, classname).Create(uid, odata, **kw)
        try:
            social_tasks = socialproxy.ask_friends(uid)
            # https://jira.yandex-team.ru/browse/CHEMODAN-18586
            # https://jira.yandex-team.ru/browse/CHEMODAN-18540
            # оставляем только FB ибо тлен
            if 'facebook' in social_tasks:
                social_tasks = {'facebook': social_tasks['facebook']}
            else:
                social_tasks = {}
            # https://jira.yandex-team.ru/browse/CHEMODAN-18540
            # https://jira.yandex-team.ru/browse/CHEMODAN-18586
        except Exception:
            social_tasks = {}
            error_log.error(traceback.format_exc())
        finally:
            operation.data['social_tasks'] = social_tasks
            operation.save()

        return operation

    def format_contacts(self, contacts, provider=None):
        result = []
        for item in contacts:
            if 'userid' in item:
                result.append({'user': item})
            elif 'groupid' in item:
                result.append({'group': item})
        return result

    def get_abook_contacts(self):
        try:
            show_groups = self.data.get('show_groups', False)

            self.stages.append({
                'service': 'email',
                'status' : SUCCESS,
                'details': self.format_contacts(
                    abook.get_contacts(self.uid, show_groups),
                    provider='email'
                ),
            })
        except Exception, e:
            error_log.info(traceback.format_exc())
            self.stages.append({
                'service': 'email',
                'status' : FAILURE,
                'error': e,
            })
        finally:
            self.data['abook_processed'] = True
            self.save()

    def abook_processed(self):
        return self.data.get('abook_processed', False)

    def social_processed(self):
        return not bool(len(self.data['social_tasks']))

    def get_social_proxy_contacts(self):
        processed_tasks = {}

        for provider, profile in self.data['social_tasks'].iteritems():
            task_result = socialproxy.get_friends(profile)
            task_state  = task_result.get(STATE)

            if task_state == SUCCESS:
                self.stages.append({
                    'service': provider,
                    'status' : SUCCESS,
                    'details': self.format_contacts(
                        task_result.get(RESULT),
                        provider=provider
                    )
                })
            elif task_state == FAILURE:
                self.stages.append({
                    'service': provider,
                    'status' : FAILURE,
                    'error': task_result.get(REASON)
                })

            if task_state in (SUCCESS, FAILURE):
                processed_tasks[provider] = True


        for provider in processed_tasks.iterkeys():
            del(self.data['social_tasks'][provider])

        if len(processed_tasks):
            self.save()

    def update_status(self):
        if not self.abook_processed():
            self.get_abook_contacts()

        if not self.social_processed():
            self.get_social_proxy_contacts()

        if not self.is_completed() and self.abook_processed() and self.social_processed():
            self.set_completed()

    def get_status(self):
        self.update_status()
        result = {STATUS : self.state, TYPE: self.type}
        if self.is_failed():
            result[ERROR] = self.data[ERROR]
        else:
            result[STAGES] = self.stages
        return result


class ImportSocialPhotos(Operation):

    DEFAULT_LOCALE = 'ru'
    SUB_OPER_FINAL_STATES = (codes.COMPLETED, codes.DONE, codes.FAILED)
    SUB_OPER_LIFE_TIME = OPERATIONS_SOCIAL_SUB_OPER_LIFE_TIME

    type = 'social'
    subtype = 'import_photos'
    social_proxy = SocialProxy()
    fs = Bus()

    @classmethod
    def Create(cls, uid, odata, **kw):
        log.info('Start social photos import for %s' % uid)

        locale = User(uid).locale
        social_dir = cls._create_social_networks_dir(uid, locale)

        provider = odata[PROVIDER]
        provider_dir = cls._get_provider_dir(uid, provider, social_dir, locale)

        odata[IMPORT_TYPE] = 'following' if cls.fs.exists(uid, provider_dir) else 'first'
        cls._create_provider_dir_if_not_exists(uid, provider, provider_dir)

        odata[LOCALE] = locale
        odata[IS_FIRST_EXECUTION] = True
        odata[PROVIDER_DIR] = provider_dir
        odata[PROTOCOL] = []

        return super(ImportSocialPhotos, cls).Create(uid, odata, **kw)

    def _initiate_import(self):
        sub_ops = []

        uid = self.uid
        provider = self.data[PROVIDER]
        locale = self.data[LOCALE]
        provider_dir = self.data[PROVIDER_DIR]

        import_albums, import_tagged = self._split_import_type(self.data[TYPE])
        log.info('import_albums: %s, import_tagged: %s' % (import_albums, import_tagged))

        if provider == socialproxy.GOOGLE_PROVIDER: # Google+ API doesn't support tagged photos
            import_tagged = False
        if provider == socialproxy.INSTAGRAM_PROVIDER: # there's no albums in Instagram
            self._download_social_album_photos(uid, provider, None, provider_dir, sub_ops)
        else:
            if import_albums:
                social_proxy_result = self._with_retries(lambda: self.social_proxy.get_albums(uid, provider, locale), uid)
                albums = self._get_albums(social_proxy_result)
                for album in albums:
                    album_dir = provider_dir + '/' + self._remove_special_symbols(album['title'])
                    self._download_social_album_photos(uid, provider, album['aid'], album_dir, sub_ops)

                self.import_additional_albums(uid, provider, provider_dir, locale, sub_ops)

            if import_tagged:
                tagged_photos = self._with_retries(lambda: self.social_proxy.user_photos(uid, provider), uid)
                if len(tagged_photos) > 0:
                    album_dir = self._create_tagged_dir(uid, provider, provider_dir, locale)
                    self._download_photos(uid, provider, album_dir, tagged_photos, sub_ops)

        self.data[PROTOCOL] = sub_ops
        self.data[IS_FIRST_EXECUTION] = False
        self.save()

    @classmethod
    def _get_provider_dir(cls, uid, provider, social_dir, locale):
        content = cls.fs.content(uid, social_dir)['list']
        for dir in content:
            if dir['meta'].get('folder_type') == provider:
                return Address.Make(uid, dir['id']).id
        return social_dir + cls._get_dirname_localized(SOCIAL_PROVIDERS_DIRS[provider], locale)

    @staticmethod
    def _remove_special_symbols(album_title):
        return album_title.replace('/', '.')

    @classmethod
    def _create_provider_dir_if_not_exists(cls, uid, provider, provider_dir):
        cls._create_dir_if_not_exists(uid, provider_dir)
        cls.fs.patch_file(uid, provider_dir, {
            'last_import_time': int(time.time()),
            'folder_type': provider,
        })

    def import_additional_albums(self, uid, provider, provider_dir, locale, sub_ops):
        if provider == socialproxy.ODNOKLASSNIKI_PROVIDER:
            personal_photos_dir = provider_dir + '/' + self._get_dirname_localized(ODNOKLASSNIKI_PERSONAL_PHOTOS, locale)
            self._download_social_album_photos(uid, provider, 'personal', personal_photos_dir, sub_ops)

        if provider == socialproxy.VK_PROVIDER:
            wall_photos_dir = provider_dir + '/' + self._get_dirname_localized(VK_WALL_PHOTOS, locale)
            self._download_social_album_photos(uid, provider, 'wall', wall_photos_dir, sub_ops)

            profile_photos_dir = provider_dir + '/' + self._get_dirname_localized(VK_PROFILE_PHOTOS, locale)
            self._download_social_album_photos(uid, provider, 'personal', profile_photos_dir, sub_ops)

            saved_photos_dir = provider_dir + '/' + self._get_dirname_localized(VK_SAVED_PHOTOS, locale)
            self._download_social_album_photos(uid, provider, 'saved', saved_photos_dir, sub_ops)

        if provider == socialproxy.MAILRU_PROVIDER:
            personal_photos_dir = provider_dir + '/' + MAILRU_PROFILE_PHOTOS
            self._download_social_album_photos(uid, provider, 'personal', personal_photos_dir, sub_ops)


    def _download_social_album_photos(self, uid, provider, aid, photos_dir, sub_ops):
        photos = self._with_retries(lambda: self.social_proxy.album_photos(uid, provider, aid), uid)
        if len(photos) > 0:
            self._create_dir_if_not_exists(uid, photos_dir)
            self._download_photos(uid, provider, photos_dir, photos, sub_ops)

    @staticmethod
    def _split_import_type(type):
        return {
            'all': (True, True),
            'albums_only': (True, False),
            'tagged_only': (False, True),
        }[type]

    def _download_photos(self, uid, provider, album_dir, photos, sub_ops):
        from mpfs.core.operations import manager

        photos_to_import = self._get_photos_to_import(uid, photos, album_dir)
        for photo in photos_to_import:
            factory = lambda: manager.create_operation(uid, 'social_copy', provider + '_disk',
                                                       self._get_odata_for_photo(album_dir, photo, provider, self.data[CONNECTION_ID]))
            self._create_sub_op(factory, sub_ops, self.id)

    @staticmethod
    def _create_sub_op(factory, sub_ops, root_op_id):
        try:
            sub_op = factory()
            sub_ops.append({
                OID: sub_op.id,
                STATUS: codes.WAITING,
                RESULT: None
            })
        except Exception:
            error_log.warn('Error while creating subop for import social photos (%s): %s' %
                           (traceback.format_exc(), root_op_id))
            sub_ops.append({
                OID: None,
                STATUS: codes.FAILED,
                RESULT: None
            })

    @classmethod
    def _create_social_networks_dir(cls, uid, locale):
        cls.fs.mksysdir(uid, 'social', locale)
        return Address.Make(uid, cls._get_dirname_localized(DEFAULT_FOLDERS['social'], locale)).id

    @classmethod
    def _create_tagged_dir(cls, uid, provider, provider_dir, locale):
        if provider == socialproxy.MAILRU_PROVIDER:
            dir_name = PHOTOS_OF_ME_MAILRU
        else:
            dir_name = cls._get_dirname_localized(PHOTOS_OF_ME_LOCALIZED, locale)
        dir_path = provider_dir + '/' + dir_name
        cls._create_dir_if_not_exists(uid, dir_path)
        return dir_path

    @classmethod
    def _create_dir_if_not_exists(cls, uid, addr):
        if not cls.fs.exists(uid, addr):
            cls.fs.mkdir(uid, addr)

    @classmethod
    def _get_photos_to_import(cls, uid, photos, album_dir):
        dir_content = map(lambda file: file['name'], cls.fs.content(uid, album_dir).get('list'))
        if dir_content:
            return filter(lambda photo: cls.format_file_name(photo) not in dir_content, photos)
        else:
            return photos

    @staticmethod
    def _with_retries(fn, uid):
        retries_count = settings.operations['social']['retries_count']
        for i in range(1, retries_count + 1):
            try:
                return fn()
            except errors.SocialProxyRateLimitExceeded:
                retries_interval_seconds = settings.operations['social']['retries_interval_seconds']
                error_log.warn('SocialProxy %d attempt failed for uid=%s, wait %d seconds before next attempt' %
                               (i, uid, retries_interval_seconds))
                time.sleep(retries_interval_seconds)
            except Exception:
                raise
        raise errors.SocialProxyRateLimitExceeded()

    @classmethod
    def _get_odata_for_photo(cls, album_dir, photo, provider, connection_id):
        result = {
            'target': album_dir + '/' + cls.format_file_name(photo),
            'service_file_url': photo['url'],
            'connection_id': connection_id,
            'provider': provider,
        }
        if 'created' in photo:
            result['created'] = photo['created']
        if 'location' in photo:
            location = photo['location']
            result['latitude'] = location['latitude']
            result['longitude'] = location['longitude']
        return result

    @staticmethod
    def _get_albums(social_proxy_result):
        if social_proxy_result['state'] == 'success':
            return social_proxy_result['result']
        else:
            error_log.warn('Bad response from social proxy: %s' % str(social_proxy_result))
            raise errors.SocialProxyBadResult()

    @staticmethod
    def format_file_name(photo):
        name = photo['pid']
        if 'order_no' in photo:
            name = "%s %s" % (name, photo['order_no'])

        if photo.get('is_video'):
            return '%s.mp4' % name
        else:
            return '%s.jpg' % name

    @classmethod
    def _get_dirname_localized(cls, dirnames, locale):
        return dirnames[locale] if locale in dirnames else dirnames[cls.DEFAULT_LOCALE]

    def get_status(self):
        result = super(ImportSocialPhotos, self).get_status()
        result[PROTOCOL] = self.data[PROTOCOL]
        return result

    def _process(self, *args, **kwargs):
        try:
            if self.data[IS_FIRST_EXECUTION]:
                self._initiate_import()
                self._reenque()
            else:
                self._check_sub_ops()
        except:
            self._log_import_stat('error')
            raise

    def _check_sub_ops(self):
        from mpfs.core.operations import manager

        failed_sub_ops_count = 0
        sub_ops_processed = True
        for sub_op_meta in self.data[PROTOCOL]:
            # поднимаем незавершенные подоперации, проверям их статус
            if sub_op_meta[OID] and sub_op_meta[STATUS] not in self.SUB_OPER_FINAL_STATES:
                sub_op = manager.get_operation(self.uid, sub_op_meta[OID])
                if time.time() - int(sub_op.ctime) > self.SUB_OPER_LIFE_TIME:
                    msg = 'Suboperation "%s" expired. Exists more then %s sec. Main oid: "%s"' % (sub_op.id, self.SUB_OPER_LIFE_TIME, self.id)
                    error_log.warn(msg)
                    sub_op.set_failed({'message': msg})

                sub_op_meta[STATUS] = sub_op.state
                # если операция в конечном состоянии, то не сбрасываем флаг завершения
                if sub_op.state not in self.SUB_OPER_FINAL_STATES:
                    sub_ops_processed = False

            # считаем кол-во зафейленных операций
            if sub_op_meta[STATUS] == codes.FAILED:
                failed_sub_ops_count += 1

        self.save()
        if sub_ops_processed:
            if failed_sub_ops_count > 0:
                error_log.warn('Failed sub ops count=%d for oid=%s' % (failed_sub_ops_count, self.id))
                self._log_import_stat('partially')
            else:
                self._log_import_stat('ok')
            self.set_completed()
        else:
            self._reenque()

    def _reenque(self):
        self.reenque(settings.operations['social']['reenqueue_delay'])

    def _log_import_stat(self, result):
        stat_social_log.info('action=import\tuid=%s\tservice=%s\toid=%s\ttype=%s\tresult=%s' %
                             (self.uid, self.data[PROVIDER], self.id, self.data.get(IMPORT_TYPE), result))


class ExportPhotos(UploadOperation):
    '''
    Экспорт фото в социальные сети
    '''

    type = 'social'
    subtype = 'export_photos'
    kladun = kladun_service.ExportPhotos()

    def __init__(self, **data):
        super(ExportPhotos, self).__init__(**data)

    @classmethod
    def Create(cls, uid, odata, **kw):
        # TODO: handle errors?
        cls.blocked_hid_check(uid, odata['photos'])
        pmids = cls.get_previews_by_filenames(uid, odata['photos'])

        profile_id = socialproxy.get_profile_for_provider(uid, odata['provider'])

        d = {
             'uid' : uid,
             'oid' : odata['id'],
             'album-id' : odata['albumid'],
             'provider' : odata['provider'],
             'profile-id' : profile_id,
             'pmids' : to_json(pmids),
        }

        int_status_url = cls.kladun.post_request(d)
        odata['status_url'] = int_status_url

        return super(ExportPhotos, cls).Create(uid, odata, **kw)

    @classmethod
    def blocked_hid_check(cls, uid, photos):
        fs = Bus()
        photos_hids = []
        for photo in photos:
            resource = factory.get_resource(uid, photo)
            photos_hids.append(resource.hid)
        fs.check_hids_blockings(photos_hids, method='export_photos')

    @classmethod
    def get_previews_by_filenames(cls, uid, photos):
        fs = Bus()
        preview_stids = []

        for photo in photos:
            address = Address(photo, uid=uid)
            if address.uid != uid:
                continue
            data = fs.info(uid, photo)
            pmid = data['this']['meta'].get('pmid')
            if pmid is not None:
                preview_stids.append(pmid)

        return preview_stids

    def _process(self, *args, **kwargs):
        super(ExportPhotos, self)._process(*args, **kwargs)
        if self.is_completed() or self.is_done():
            self._log_export_stat('ok')
        elif self.is_failed() or self.is_rejected():
            self._log_export_stat('error')

    def _log_export_stat(self, result):
        stat_social_log.info('action=export\tuid=%s\tservice=%s\toid=%s\tresult=%s' %
                             (self.uid, self.data['provider'], self.id, result))


class PostAlbumToSocials(Operation):
    """
    Публикация альбома в социальные сети
    """
    type = 'social'
    subtype = 'post_album'

    def _process(self, *args, **kwargs):
        resp = self._post_album()

        if resp['state'] == 'success':
            self.set_completed()
        else:
            error = ''
            if 'reason' in resp:
                error = resp['reason']
            self.set_failed({'message': error, 'code': codes.ALBUMS_POST_TO_SOCIAL_FAILED})

    def _post_album(self):
        provider = self.data['provider']

        if 'album_id' in self.data:
            album = Album.controller.get(uid=self.uid, id=self.data['album_id'])
            return album.send_social_wall_post(provider)
        else:
            # для обратной совместимости со старыми операциями параметризованными короткими урлами вместо album_id
            album_short_url = self.data['album_short_url']
            return socialproxy.wall_post(self.uid, provider, album_short_url)
