# -*- coding: utf-8 -*-

import re
import calendar
from datetime import datetime


import mpfs.engine.process
log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

from mpfs.config import settings
from mpfs.common.util import from_json
from mpfs.common import errors
from mpfs.common.static.tags import *
from mpfs.core.services.kladun_service import Kladun


def get_stage(tag, value, op_id):
    stages = {
        'incoming-http$' : FileUploadKladun,
        'incoming-file$' : FileParseKladun,
        'mulca-file$' : FileUploadMulca,
        'mulca-digest$' : DigestUploadMulca,
        'search-export$' : SearchExport,
        'antivirus$' : Antivirus,
        'preview$' : Preview,
        'preview-image': PreviewImage,
        'original-file$' : OriginalParseKladun,
        'patched-file$' : PatchedParseKladun,
        'downloaded-file$' : ServiceDownload,
        'conversion$' : Conversion,
        'publish$' : Publish,
        'text$' : Text,
        'callback$' : Callback,
        'expected-patched-md5$' : ExpectedPatchedParseKladun,
        'search-export2$': SearchExport,
        'commit-file-info': CommitFileInfo,
        'commit-file-upload': CommitFileUpload,
        'generate-image-preview': PreviewSizes,
        'exif-info': ExifInfo,
        'video-info': VideoInfo,
        'generate-image-one-preview' : SinglePreviewImageINFO,
        'preview-image-mulca-upload' : SinglePreviewImageMID,
        'media-info' : MediaInfo,
        'preview-video-mulca-upload' : SinglePreviewVideoMID,
        'export-photo-\d' : ExportPhoto,
        'export-photo-url' : ExportPhotoUrl,
        'export-photo-upload' : ExportPhotoUpload,
    }
    stage_class = None
    name = ''
    try:
        stage_class = stages[tag]
    except KeyError:
        for k, v in stages.iteritems():
            if re.match(k, tag):
                stage_class = v
                break
        if stage_class is None:
            if settings.feature_toggles['raise_on_operations_bad_stage']:
                raise errors.KladunUnknownStage('%s %s' % (tag, value))
            else:
                name = tag.replace('-', '_')
                stage_class = Stage
    return stage_class(value, tag=tag, op_id=op_id, name=name)


class Stage(object):

    name = ''
    main = False
    op_id = None

    def __init__(self, data, name='', tag='', op_id=''):
        if not self.name and name:
            self.name = name
        self.status = data[STATUS]
        self.processing_time = '%f' % data['processing_time_diff'] if 'processing_time_diff' in data else ''
        self.op_id = op_id

    def update(self, *args, **kwargs):
        pass

    def progress(self):
        return ''

    def details(self):
        return {}

    def dict(self):
        return {
                'status' : self.status,
                'name' : self.name,
                'progress' : self.progress(),
                'processing_time' : self.processing_time,
                'details' : self.details(),
                }


class SingletonStage(Stage):

    def __new__(cls, *args, **kwargs):
        if not hasattr(cls, '_instance') or kwargs.get('op_id') != cls._instance.op_id:
            cls._instance = super(Stage, cls).__new__(cls)
        return cls._instance


class Publish(Stage):

    name = 'publishing'
    main = True

    def __init__(self, *args, **kwargs):
        super(Publish, self).__init__(*args, **kwargs)
        data = args[0]
        self.service_file_id = data[RESULT].get('service-file-id', '')
        self.service = data[RESULT].get('service', '')
        self.uid = data[RESULT].get('uid', '')
        self.file_id = data[RESULT].get('file-id', '')
        self.mpfs_file_id = '/' + self.service + '/' + self.file_id

    def details(self):
        result = super(Publish, self).details()
        result.update({
                        'service_file_id' : self.service_file_id,
                        'service' : self.service,
                        'uid' : self.uid,
                        'file_id' : self.file_id,
                        'mpfs_file_id' : self.mpfs_file_id,
                        })
        return result


class Text(Stage):

    name = 'text'


class Conversion(Stage):

    name = 'conversion'


class SearchExport(Stage):

    name = 'search_export'


class Callback(Stage):

    name = 'callback_stage'

    def __init__(self, *args, **kwargs):
        data = args[0]
        super(Callback, self).__init__(data)
        # Кладун не передает success,
        # иначе операция никогда не будет COMPLETED.
        self.status = SUCCESS

class ServiceDownload(Stage):

    name = 'downloaded_file'
    main = True

    def __init__(self, *args, **kwargs):
        super(ServiceDownload, self).__init__(*args, **kwargs)
        data = args[0]
        self.content_length = data[RESULT].get(CONTENT_LENGTH_TAG, 0)
        self.md5 = data[RESULT].get(MD5, '')
        self.sha256 = data[RESULT].get(SHA256, '')
        self.mimetype = data[RESULT].get(CONTENT_TYPE, '').replace('"', '')

    def details(self):
        result = super(ServiceDownload, self).details()
        result.update({
                        'size' : self.content_length,
                        'md5' : self.md5,
                        'sha256' : self.sha256,
                        'mimetype' : self.mimetype,
                    })
        return result


class FileUploadKladun(Stage):

    name = 'kladun_upload'
    main = True

    def __init__(self, *args, **kwargs):
        super(FileUploadKladun, self).__init__(*args, **kwargs)
        data = args[0]
        self.uploaded = int(data[PROGRESS].get(CURRENT, 0))
        self.total = int(data[PROGRESS].get(TOTAL, 0))
        self.mimetype = data[RESULT].get(CONTENT_TYPE, '').replace('"', '')

    def progress(self):
        return self.uploaded * 100 / self.total if self.total else 0

    def details(self):
        result = super(FileUploadKladun, self).details()
        result.update({
                        'uploaded' : self.uploaded,
                        'total' : self.total,
                        'mimetype' : self.mimetype,
                    })
        return result


class FileParseKladun(Stage):

    name = 'kladun_parse'
    main = True

    def __init__(self, *args, **kwargs):
        super(FileParseKladun, self).__init__(*args, **kwargs)
        data = args[0]
        self.content_length = data[RESULT].get(CONTENT_LENGTH_TAG, 0)
        self.md5 = data[RESULT].get(MD5, '')
        self.sha256 = data[RESULT].get(SHA256, '')
        self.mimetype = data[RESULT].get(CONTENT_TYPE, '').replace('"', '')

    def details(self):
        result = super(FileParseKladun, self).details()
        result.update({
                        'content_size' : self.content_length,
                        'md5' : self.md5,
                        'sha256' : self.sha256,
                        'mimetype' : self.mimetype,
                    })
        return result


class OriginalParseKladun(Stage):

    name = 'kladun_original_parse'
    main = True


class PatchedParseKladun(Stage):

    name = 'kladun_patched_parse'
    main = True

    def __init__(self, *args, **kwargs):
        data = args[0]
        super(PatchedParseKladun, self).__init__(data)
        self.content_length = data[RESULT].get(CONTENT_LENGTH_TAG, 0)
        self.md5 = data[RESULT].get(MD5, '')
        self.sha256 = data[RESULT].get(SHA256, '')

    def details(self):
        result = super(PatchedParseKladun, self).details()
        result.update({
            'content_size' : self.content_length,
            'md5' : self.md5,
            'sha256' : self.sha256,
        })
        return result


class ExpectedPatchedParseKladun(Stage):

    main = True
    name = 'expected_kladun_patched_parse'


class MulcaStage(Stage):

    def __init__(self, *args, **kwargs):
        super(MulcaStage, self).__init__(*args, **kwargs)
        data = args[0]
        self.mulca_id = data[RESULT].get(MULCA_ID, '')

    def details(self):
        result = super(MulcaStage, self).details()
        result.update({
            'mulca_id' : self.mulca_id,
        })
        return result


class FileUploadMulca(MulcaStage):

    name = 'mulca_file_upload'
    main = True


class DigestUploadMulca(MulcaStage):

    name = 'mulca_digest_upload'
    main = True


class Preview(MulcaStage):

    name = 'preview'

    def __init__(self, *args, **kwargs):
        super(MulcaStage, self).__init__(*args, **kwargs)
        data = args[0]
        self.mulca_id = data[RESULT].get(MULCA_ID, '')

    def details(self):
        result = super(MulcaStage, self).details()
        result.update({
                       'mulca_id' : self.mulca_id,
                       })
        return result


class PreviewImage(SingletonStage):

    name = 'preview_image'
    sizes = ['s', 'l', 'm', 'xs', 'xxs', 'xxxs', 'xl', 'xxl', 'xxxl']
    main = False

    def __init__(self, *args, **kwargs):
        if self.op_id is None or self.op_id != kwargs.get('op_id'):
            Stage.__init__(self, *args, **kwargs)
            self.preview_links = {}
            self.preview_statuses = {}
        self.update(*args, **kwargs)

    def details(self):
        result = super(PreviewImage, self).details()
        result.update({
            'preview_sizes' : self.preview_links
        })
        return result

    def update(self, *args, **kwargs):
        data = args[0]
        tag = kwargs.get('tag', '')

        if tag.replace('-', '_') == self.name:
            for size in self.sizes:
                if data[RESULT].get(size) is not None:
                    self.preview_links[size.upper()] = data[RESULT].get(size)
                    self.preview_statuses[size.upper()] = self.status
        else:
            size_name = filter(None, tag.split('-'))[-1]
            self.preview_links[size_name.upper()] = data[RESULT].get(MULCA_ID, '')
            self.preview_statuses[size_name.upper()] = data[STATUS]

        if not filter(
            lambda x: x != SUCCESS,
            self.preview_statuses.itervalues()
        ): self.status = SUCCESS



class PreviewSizes(Stage):

    name = 'preview_sizes'

    def __init__(self, *args, **kwargs):
        data = args[0]
        super(PreviewSizes, self).__init__(*args, **kwargs)
        self.preview_format = data[RESULT].get('format')
        self.preview_sizes = {}
        if self.status in (SUCCESS, ):
            for each_size in data[RESULT].iterchildren():
                size_name = each_size.tag.split('preview-image-')[-1].upper()
                if size_name:
                    self.preview_sizes[size_name] = dict(each_size.items())

    def details(self):
        result = super(PreviewSizes, self).details()
        result.update({
                      'preview_format' : self.preview_format,
                      'preview_sizes' : self.preview_sizes,
                      })


class Antivirus(Stage):

    name = 'antivirus_check'

    def __init__(self, *args, **kwargs):
        data = args[0]
        super(Antivirus, self).__init__(*args, **kwargs)
        self.drweb = 0
        if self.status == PERMANENT_FAILURE:
            self.drweb = 4
        elif self.status in (SUCCESS,):
            result = data[RESULT].get('result-ext') or data[RESULT].get('result', 'unknown')
            stats = {
                'true'     : 1,
                'false'    : 2,
                'healthy'  : 1,
                'infected' : 2,
                'unknown'  : 4
            }
            self.drweb = stats.get(result, 0)

    def details(self):
        result = super(Antivirus, self).details()
        result.update({
            'drweb' : self.drweb,
         })
        return result


class ExportPhotoBase(Stage):

    def __init__(self, *args, **kwargs):
        super(ExportPhotoBase, self).__init__(*args, **kwargs)
        data = args[0]
        tag = kwargs.get('tag')
        self.index = tag.rsplit('-', 1)[1]
        self.name = self.name + "_" + self.index
        self.invalid_token = data[DETAILS].get('invalid_token', 'false')

    def details(self):
        result = { 'ind' : self.index }
        if self.invalid_token == 'true':
            result.update({ 'invalid_token' : '1'})
        return result


class ExportPhotoUrl(ExportPhotoBase):

    name = 'export_photo_url'


class ExportPhotoUpload(ExportPhotoBase):

    name = 'export_photo_upload'


class ExportPhoto(ExportPhotoBase):

    name = 'export_photo'


class SinglePreviewImageINFO(Stage):

    name = 'single_preview_info'

    def __init__(self, *args, **kwargs):
        super(SinglePreviewImageINFO, self).__init__(*args, **kwargs)
        data = args[0]
        self.format = data[RESULT].get('format', '')
        self.width = data[RESULT].get('width' , '')
        self.height = data[RESULT].get('height', '')

        image_info = Kladun.get_image_info_from_response(data[RESULT])
        self.original_width = image_info['original_width']
        self.original_height = image_info['original_height']
        self.rotate_angle = image_info['rotate_angle']

    def details(self):
        return {
            'format': self.format,
            'width': self.width,
            'height': self.height,
            'original_width': self.original_width,
            'original_height': self.original_height,
            'rotate_angle': self.rotate_angle
        }


class SinglePreviewMID(Stage):

    def __init__(self, *args, **kwargs):
        super(SinglePreviewMID, self).__init__(*args, **kwargs)
        data = args[0]
        self.mulca_id = data[RESULT].get('mulca-id', '')

    def details(self):
        return {
                'mid' : self.mulca_id,
                }


class SinglePreviewImageMID(SinglePreviewMID):

    name = 'single_preview_mid'


class SinglePreviewVideoMID(SinglePreviewMID):

    name = 'single_preview_video_mid'


class ExifInfo(Stage):

    name = 'exif_info'

    def __init__(self, *args, **kwargs):
        super(ExifInfo, self).__init__(*args, **kwargs)
        data = args[0]
        etime = data[RESULT].get('creation-date')
        try:
            self.etime = calendar.timegm(datetime.strptime(etime, '%Y-%m-%dT%H:%M:%SZ').timetuple())
        except (ValueError, TypeError):
            self.etime = 0
        if int(self.etime) < 0:
            self.etime = 0


class MediaInfo(ExifInfo):

    name = 'media_info'


class VideoInfo(Stage):

    name = 'video_info'

    def __init__(self, *args, **kwargs):
        super(VideoInfo, self).__init__(*args, **kwargs)
        result = args[0][RESULT].text
        if result:
            self.content = from_json(result)
        else:
            self.content = {}

    def details(self):
        return {
                'content' : self.content,
                }


class CommitFileInfo(Stage):
    name = 'commit_file_info'
    main = True


class CommitFileUpload(Stage):
    name = 'commit_file_upload'


class SearchExport(Stage):
    name = 'search-export'

