# -*- coding: utf-8 -*-
import mpfs.engine.process

from mpfs.core.operations.base import Operation
from mpfs.core.bus import Bus
from mpfs.core.services import kladun_service
from mpfs.core.filesystem.hardlinks.controllers import HardlinksController
from mpfs.core.filesystem.helpers.lock import LockHelper
from mpfs.common.util.logger import TSKVMessage


log = mpfs.engine.process.get_default_log()


class SystemOperation(Operation):
    type = 'system'


class UpdateFileHash(SystemOperation):
    subtype = 'update_file_hash'

    def _process(self):
        uid = self.uid
        oid = self.id
        path = self.data['path']
        raw_address = '%s:%s' % (uid, path)
        filesystem = Bus()
        base_message = TSKVMessage(uid=uid, path=path, oid=oid, op_type=self.type, op_subtype=self.subtype)
        try:
            file_data = filesystem.info(uid, raw_address)['this']
        except Exception as e:
            log.info(unicode(base_message.update(status='failed', msg=type(e))))
            raise
        if 'md5' in self.data and 'sha256' in self.data and 'size' in self.data:
            if file_data['meta']['md5'] != self.data['md5'] or \
                    file_data['meta']['sha256'] != self.data['sha256'] or \
                    file_data['size'] != self.data['size']:
                self.set_failed({'message': 'Sums do not match'})

        lock_helper = LockHelper()
        base_message = TSKVMessage(uid=uid, path=path, oid=oid, op_type=self.type, op_subtype=self.subtype)
        try:
            lock_helper.lock(raw_address)
        except Exception as e:
            log.info(unicode(base_message.update(status='failed', msg='resource is locked')))
            self.set_failed(e)
            return

        try:
            data = kladun_service.UpdateCheckSum().post_request(file_data['meta']['file_mid'])
            filesystem.patch_file(uid, raw_address, data, force=True)
        except Exception as e:
            log.info(unicode(base_message.update(status='failed', msg=type(e))))
            raise
        else:
            success_data = {'hashes_matched': True}
            for hash_name in ('md5', 'sha256', 'size'):
                old, new = 'old_%s' % hash_name, 'new_%s' % hash_name
                success_data[new] = str(data.get(hash_name))
                if hash_name == 'size':
                    success_data[old] = str(file_data.get(hash_name))
                else:
                    success_data[old] = str(file_data['meta'].get(hash_name))
                success_data['hashes_matched'] &= success_data[new] == success_data[old]
            log.info(unicode(base_message.update(status='success', **success_data)))
            self.set_completed()
        finally:
            lock_helper.unlock(raw_address)


class UpdateFilesMIDs(SystemOperation):
    """
    Обновляет mids по hid-у

    В odata ожидаются следующие параметры: hid, file_mid, digest_mid, pmid
    """
    subtype = 'update_files_mids'
    hardlinks_controller = HardlinksController()
    mid_fields = ('file_mid', 'digest_mid', 'pmid')
    empty_file_hid = 'cbe7b309b4bbe3a94682616cb7e88e01'

    def _process(self):
        uid = self.uid
        hid = self.data.get('hid')
        if hid:
            if hid == self.empty_file_hid:
                log.info('ignore update empty file mids')
            else:
                mids = {mid: self.data[mid] for mid in self.mid_fields if mid in self.data}
                self.hardlinks_controller.update_files_mids(hid, mids)
        self.set_completed()
