# -*- coding: utf-8 -*-
import os
import traceback

import mpfs.engine.process
from mpfs.common import errors
from mpfs.common.util.threaded_action_repeater import ThreadedActionRepeater
from mpfs.common.util.ycrid_parser import YcridParser, YcridPlatformPrefix
from mpfs.config import settings
from mpfs.core.metastorage.control import user_index
from mpfs.core.user import base as user
from thread import interrupt_main

error_log = mpfs.engine.process.get_error_log()
default_log = mpfs.engine.process.get_default_log()
OPERATION_PINGER_DELAY = 300
OPERATION_PINGER_DELAY_FIRST_ACTION = True


def check_memory_limit_exceeded(operation):
    try:
        pagesize = os.sysconf("SC_PAGE_SIZE")
        rss_limit = settings.operations['rss_limit']
        vms_limit = settings.operations['vms_limit']
        fail_on_memory_limit_exceeded = settings.operations['fail_on_memory_limit_exceeded']
        with open('/proc/%s/statm' % os.getpid()) as f:
            vms, rss, shared, text, lib, data, dirty = [int(x) * pagesize for x in f.readline().split()[:7]]
        if (rss_limit and rss > rss_limit) or (vms_limit and vms > vms_limit):
            default_log.info(
                'RAM limit is reached by uid: %(uid)s oid: %(oid)s vms: %(vms)s rss: %(rss)s shared: %(shared)s text: '
                '%(text)s lib: %(lib)s data: %(data)s dirty: %(dirty)s' % {
                    'uid': operation.uid, 'oid': operation.id, 'vms': vms, 'rss': rss, 'shared': shared, 'text': text,
                    'lib': lib, 'data': data, 'dirty': dirty}
            )
            if fail_on_memory_limit_exceeded:
                default_log.info(
                    'Operation will be failed because of OOM: %(uid)s oid: %(oid)s' % {
                        'uid': operation.uid, 'oid': operation.id}
                )
                operation.set_failed({'message': 'Operation is out of memory'})
                interrupt_main()
                return True
    except:
        error_log.error('Failed to check memory limit: %s' % traceback.format_exc())
    return False


class OperationPinger(ThreadedActionRepeater):
    def __init__(self, operation, delay=OPERATION_PINGER_DELAY, delay_first_action=OPERATION_PINGER_DELAY_FIRST_ACTION):
        super(OperationPinger, self).__init__(delay, delay_first_action=delay_first_action)
        self.operation = operation

    def action(self):
        user_obj = user.User(self.operation.uid)
        for retry_sec in (3, 15, 90, 0):
            try:
                user_index.reset()
                user_obj.check_blocked()
                self.operation.update_dtime()
                if check_memory_limit_exceeded(self.operation):
                    break
            except errors.UserBlocked, e:
                error_log.error('User is blocked by interrupt main thread')
                interrupt_main()
                break
            except Exception:
                error_log.error(traceback.format_exc())
                self._stop_flag.wait(retry_sec)
            else:
                break


def check_store_for_zero_file_size(operation_type, size_from_uploader, size_from_client):
    if not settings.feature_toggles['fail_store_operations_with_zero_file_size']['enabled']:
        return
    if operation_type not in ('store', 'overwrite'):
        return

    platform = YcridParser.get_platform(mpfs.engine.process.get_cloud_req_id())
    checked_platforms = [checked_platform.lower()
                         for checked_platform in settings.feature_toggles['fail_store_operations_with_zero_file_size']['platforms']]
    if (platform is not None and
            platform.lower() in checked_platforms and
            size_from_uploader is not None and long(size_from_uploader) == 0 and
            size_from_client is not None and long(size_from_client) != 0):
        error_log.error("Failed to store zero file size (client specified non empty file: %s)" % size_from_client)
        raise errors.EmptyFileUploadedForNonEmpyStoreError
