# -*- coding: utf-8 -*-

from mpfs.dao.base import BaseDAO, BaseDAOItem, MongoBaseDAOImplementation, PostgresBaseDAOImplementation
from mpfs.dao.fields import BoolField, IntegerField, StringField
from mpfs.dao.session import Session
from mpfs.metastorage.postgres.queries import (
    SQL_SEARCH_ORGANIZATION_BY_ID,
    SQL_UPSERT_ORGANIZATION_QUOTA_LIMITS_AND_USED,
    SQL_UPSERT_ORGANIZATION_QUOTA_LIMITS_AND_PAID,
)
from mpfs.metastorage.postgres.schema import organizations


class OrganizationDAOItem(BaseDAOItem):
    mongo_collection_name = 'organizations'
    postgres_table_obj = organizations
    is_sharded = False
    is_migrated_to_postgres = False

    @classmethod
    def get_postgres_primary_key(cls):
        return 'id'

    id = StringField(mongo_path='_id', pg_path=organizations.c.id)
    is_paid = BoolField(mongo_path='is_paid', pg_path=organizations.c.is_paid, default_value=False)
    quota_limit = IntegerField(mongo_path='quota_limit', pg_path=organizations.c.quota_limit, default_value=None)
    quota_free = IntegerField(mongo_path='quota_free', pg_path=organizations.c.quota_free, default_value=None)
    quota_used_by_disk = IntegerField(mongo_path='quota_used_by_disk', pg_path=organizations.c.quota_used_by_disk, default_value=None)


class OrganizationDAO(BaseDAO):
    dao_item_cls = OrganizationDAOItem

    def __init__(self):
        super(OrganizationDAO, self).__init__()
        self._mongo_impl = MongoOrganizationDAOImplementation(self.dao_item_cls)
        self._pg_impl = PostgresOrganizationDAOImplementation(self.dao_item_cls)

    def find_by_id(self, _id):
        return self._get_impl(None).find_by_id(_id)

    def set_quota_limits_and_used(self, _id, quota_limit, quota_free, quota_used_by_disk):
        self._get_impl(None).set_quota_limits_and_used(_id, quota_limit, quota_free, quota_used_by_disk)

    def set_quota_limits_and_paid(self, _id, quota_limit, quota_free, is_paid):
        self._get_impl(None).set_quota_limits_and_paid(_id, quota_limit, quota_free, is_paid)


class MongoOrganizationDAOImplementation(MongoBaseDAOImplementation):
    def find_by_id(self, _id):
        query = {'_id': _id}
        result = super(MongoOrganizationDAOImplementation, self).find_one(query)
        if result:
            return OrganizationDAOItem.create_from_mongo_dict(result)
        return None

    def set_quota_limits_and_used(self, _id, quota_limit, quota_free, quota_used_by_disk):
        query = {'_id': _id}
        update = {
            '$set': {'quota_limit': quota_limit, 'quota_free': quota_free, 'quota_used_by_disk': quota_used_by_disk},
            '$setOnInsert': {'is_paid': False},
        }
        super(MongoOrganizationDAOImplementation, self).update(query, update, upsert=True)

    def set_quota_limits_and_paid(self, _id, quota_limit, quota_free, is_paid):
        query = {'_id': _id}
        update = {'$set': {'quota_limit': quota_limit, 'quota_free': quota_free, 'is_paid': is_paid}}
        super(MongoOrganizationDAOImplementation, self).update(query, update, upsert=True)


class PostgresOrganizationDAOImplementation(PostgresBaseDAOImplementation):
    def find_by_id(self, _id):
        query_params = {'id': _id}
        session = Session.create_common_shard()
        result = session.execute(SQL_SEARCH_ORGANIZATION_BY_ID, query_params).fetchone()
        if result:
            return OrganizationDAOItem.create_from_pg_data(result)
        return None

    def set_quota_limits_and_used(self, _id, quota_limit, quota_free, quota_used_by_disk):
        query_params = {'id': _id, 'quota_limit': quota_limit, 'quota_free': quota_free, 'quota_used_by_disk': quota_used_by_disk}
        session = Session.create_common_shard()
        session.execute(SQL_UPSERT_ORGANIZATION_QUOTA_LIMITS_AND_USED, query_params)

    def set_quota_limits_and_paid(self, _id, quota_limit, quota_free, is_paid):
        query_params = {'id': _id, 'quota_limit': quota_limit, 'quota_free': quota_free, 'is_paid': is_paid}
        session = Session.create_common_shard()
        session.execute(SQL_UPSERT_ORGANIZATION_QUOTA_LIMITS_AND_PAID, query_params)
