# -*- coding: utf-8 -*-
import urllib2

import mpfs.engine.process

from mpfs.common.errors import APIError
from mpfs.common.static.tags.billing import PID, PRODUCT
from mpfs.config import settings
from mpfs.core.billing import Subscription, Service
from mpfs.core.billing.client import Client
from mpfs.core.billing.processing import billing
from mpfs.core.billing.service import ServiceList
from mpfs.core.filesystem.quota import Quota
from mpfs.core.organizations.dao.organizations import OrganizationDAO
from mpfs.core.queue import mpfs_queue
from mpfs.core.services.directory_service import DirectoryService, MigrationInProcessException
from mpfs.core.user.dao.user import UserDAO

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

B2B_SHARED_ORGANIZATION_SPACE_ENABLED = settings.b2b['shared_organization_space_enabled']
B2B_SHARED_ORGANIZATION_SPACE_RECALCULATION_DELAY = settings.b2b['shared_organiaztion_space_recalculation_delay']
B2B_IGNORED_SERVICES = {'initial_10gb', 'b2b_10gb', 'initial_3gb', 'promo_shared', 'file_uploaded', 'app_install'}
B2B_FAKE_ORGANIZATION_SPACE_SERVICE_NAME = 'b2b_fake_organization_free_space'


def update_organization(organization_id):
    if not B2B_SHARED_ORGANIZATION_SPACE_ENABLED:
        return

    service = DirectoryService()
    try:
        info = service.get_organization_info(organization_id)

        user_dao = UserDAO()
        organization_dao = OrganizationDAO()
        previous_organization = organization_dao.find_by_id(organization_id)
        OrganizationDAO().set_quota_limits_and_paid(organization_id, info.disk_limit,
                                                    info.disk_limit - info.disk_usage, info.is_paid)

        is_new_organization = not previous_organization
        organization_was_paid = previous_organization and not previous_organization.is_paid
        organization_is_paid = info.is_paid
        # организация стала платной
        if (is_new_organization or organization_was_paid) and organization_is_paid:
            calculate_organization_used_space_async(organization_id)
            users = user_dao.get_users_by_b2b_key(organization_id)
            for user in users:
                try:
                    _cancel_paid_subscriptions(user.uid)
                except Exception:
                    error_log.exception('_cancel_paid_subscriptions %s failed' % user.uid)
    except MigrationInProcessException:
        error_log.exception('update_organization %s failed: migration in process' % organization_id)
    except (urllib2.HTTPError, APIError):
        error_log.exception('update_organization %s failed' % organization_id)


def update_organization_async(organization_id):
    if not B2B_SHARED_ORGANIZATION_SPACE_ENABLED:
        return

    mpfs_queue.put({'organization_id': organization_id}, 'update_organization')


def calculate_organization_used_space(organization_id):
    if not B2B_SHARED_ORGANIZATION_SPACE_ENABLED:
        return

    user_dao = UserDAO()
    organization_dao = OrganizationDAO()

    organization = organization_dao.find_by_id(organization_id)
    if not organization or not organization.is_paid:
        return

    users = user_dao.get_users_by_b2b_key(organization_id)
    used = 0
    for user in users:
        user_used = Quota().used(uid=user.uid)
        user_bought = organization_user_bought_space(user.uid)
        used += max(user_used - user_bought, 0)

    try:
        limits = DirectoryService().set_organization_disk_usage(organization_id, used)
        organization_dao.set_quota_limits_and_used(organization_id, limits.limit, limits.free, used)
    except MigrationInProcessException:
        error_log.exception('update_organization %s failed: migration in process' % organization_id)


def organization_user_bought_space(uid):
    client = Client(uid)
    services = ServiceList(client=client)
    paid_services = [x for x in services if x[PID] not in B2B_IGNORED_SERVICES]
    return sum([x[PRODUCT].attributes.amount for x in paid_services])


def _cancel_paid_subscriptions(uid):
    client = Client(uid)
    services = ServiceList(client=client)
    for service in services:
        if not service['auto']:
            continue
        if service['parent_sid']:  # если это услуга не покупателя, то не должно быть возможности отменить подписку
            continue

        try:
            service = Service(service['sid'])
            subscription = Subscription(service=service.sid)
            billing.subscription_unsubscribe(client, service, subscription)
        except Exception:
            error_log.exception('_cancel_paid_subscriptions uid %s, session %s failed' % (uid, service['sid']))


def calculate_organization_used_space_async(organization_id):
    if not B2B_SHARED_ORGANIZATION_SPACE_ENABLED:
        return

    mpfs_queue.put({'organization_id': organization_id}, 'calculate_organization_used_space',
                   deduplication_id='calculate_organization_used_space__' + organization_id,
                   delay=B2B_SHARED_ORGANIZATION_SPACE_RECALCULATION_DELAY)


def resync_organizations():
    if not B2B_SHARED_ORGANIZATION_SPACE_ENABLED:
        return

    service = DirectoryService()
    for info in service.get_organization_infos():
        OrganizationDAO().set_quota_limits_and_paid(info.id, info.disk_limit,
                                                    info.disk_limit - info.disk_usage, info.is_paid)
