# -*- coding: utf-8 -*-
import mpfs.engine.process
from mpfs.common import errors
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.config import settings
from mpfs.core.address import Address
from mpfs.core.billing.dao.overdraft import OverdraftDAO
from mpfs.core.metastorage.control import groups
from mpfs.core.overdraft.notifications import send_email_on_block, send_push_regular, send_email_on_kick_shared, \
    send_email_for_light, send_email_for_hard
from mpfs.core.services.bazinga_service import OnetimeTask
from mpfs.core.services.passport_service import passport
from mpfs.core.social.share import ShareProcessor, Group
from mpfs.core.user.base import User


OVERDRAFT_EMAIL_LIGHT_DATES = settings.user['overdraft']['email_light_dates']
OVERDRAFT_EMAIL_HARD_DATES = settings.user['overdraft']['email_hard_dates']
OVERDRAFT_HARD_RESTRICTION_DAYS = settings.user['overdraft']['hard_restriction_days']
OVERDRAFT_BLOCK_RESTRICTION_DAYS = settings.user['overdraft']['block_restriction_days']

log = mpfs.engine.process.get_default_log()


class BazingaOverdraftSendMessageWorker(OnetimeTask):
    BAZINGA_TASK_NAME = 'OverdraftSendMessageWorker'

    overdraft_info = None

    def __init__(self, uid):
        super(BazingaOverdraftSendMessageWorker, self).__init__()
        if not isinstance(uid, basestring):
            raise TypeError('OverdraftSendMessageWorker parameter type is incorrect: %s' % type(uid))
        self.uid = uid

    def build_command_parameters(self):
        return [self.uid]

    def run(self):
        from datetime import timedelta, date

        if not self.uid:
            log.info('No user id')
            return

        log.info('Overdraft user %s is processed' % self.uid)
        try:
            user = User(self.uid)
        except errors.StorageInitUser:
            log.info('Overdraft user %s: not found' % self.uid)
            return

        if user.is_blocked():
            log.info('Overdraft user %s: already blocked' % self.uid)
            return

        self.overdraft_info = OverdraftDAO().get(self.uid)
        if not self.overdraft_info:
            log.info('Overdraft user %s: overdraft info not found' % self.uid)
            return

        passport_user_info = passport.userinfo(self.uid)
        if not passport_user_info.get('has_disk'):
            log.info('Overdraft user %s: no disk' % self.uid)
            return

        experiment_manager.update_context(uid=self.uid)
        if not experiment_manager.is_feature_active('new_overdraft_strategy'):
            log.info('Overdraft user %s: experiment disabled' % self.uid)
            return

        if not user.is_in_overdraft_for_restrictions():
            log.info('Overdraft user %s: has free space' % self.uid)
            return

        self._send_push_if_need()

        today = date.today()
        overdraft_date = self.overdraft_info.overdraft_date
        if user.is_blocked_for_overdraft():
            log.info('overdraft_send_message uid: %s, status: blocked' % self.uid)
            user.set_block(1, 'blocked for overdraft')
            send_email_on_block(self.uid)
            self._block_groups()

        elif overdraft_date < today - timedelta(days=OVERDRAFT_HARD_RESTRICTION_DAYS):
            if self._need_send_email(OVERDRAFT_EMAIL_HARD_DATES):
                days_left = OVERDRAFT_HARD_RESTRICTION_DAYS + OVERDRAFT_BLOCK_RESTRICTION_DAYS + 1 - (
                    today - overdraft_date).days
                send_email_for_hard(self.uid, days_left)

        else:
            if self._need_send_email(OVERDRAFT_EMAIL_LIGHT_DATES):
                user_info = user.info()
                user_space = user_info['space']['used']
                user_limit = user_info['space']['limit']
                send_email_for_light(self.uid, user_space, user_limit)

    def _send_push_if_need(self):
        from datetime import datetime, timedelta

        if not self.overdraft_info.last_push_date or \
           self.overdraft_info.last_push_date.replace(tzinfo=None) < datetime.now() - timedelta(hours=12):
            log.info('Sending push about overdraft to uid: %s' % self.uid)
            send_push_regular(self.uid)

    def _need_send_email(self, need_send_dates):
        from datetime import timedelta, date
        overdraft_date = self.overdraft_info.overdraft_date
        last_send_date = self.overdraft_info.last_email_date

        if last_send_date is None:
            return True

        today = date.today()
        for need_send_date in need_send_dates:
            if last_send_date.date() < overdraft_date + timedelta(days=need_send_date) < today:
                return True
        return False

    def _block_groups(self):
        for item in groups.get_all(owner=self.uid):
            group = Group(**item)
            log.info('Block %s group', group.gid)

            uids = group.all_uids()
            uids.remove(self.uid)  # убираем владельца из списка
            folder_path_wo_area = Address(group.path, uid=self.uid).get_path_without_area()
            for uid_ in uids:
                log.info(
                    'Don\'t have access to shared folder %s for overdraft to uid: %s' % (group.path, uid_))
                ShareProcessor().kick_user(
                    self.uid,
                    group.gid,
                    uid_,
                )
                send_email_on_kick_shared(uid_, folder_path_wo_area)
