import datetime
import socket

from mpfs.core.billing.dao.overdraft import OverdraftDAO
from mpfs.core.email.logic import send_email_async_by_uid
from mpfs.core.pushnotifier.queue import PushQueue
from mpfs.core.user.base import User
from mpfs.engine.process import get_default_log, get_error_log
from mpfs.core.services.passport_service import passport

default_log = get_default_log()
error_log = get_error_log()


DEFAULT_LOCALE = 'en'
ALLOWED_LOCALES = ['ru', 'en', 'tr', 'ua', 'uk']


def send_email_on_block(uid):
    send_email_async_by_uid(
        uid,
        'the_user_is_locked',
        default_locale=DEFAULT_LOCALE,
        allowed_locales=ALLOWED_LOCALES
    )


def send_email_on_kick_shared(uid, path):
    send_email_async_by_uid(
        uid,
        'dont_have_access_to_shared_folder',
        template_args={'folder_name': path},
        default_locale=DEFAULT_LOCALE,
        allowed_locales=ALLOWED_LOCALES
    )


def send_email_for_hard(uid, days_left):
    overdraft_info = _get_overdraft_info(uid)

    try:
        default_log.info('overdraft_send_message uid: %s, status: hard' % uid)
        if _need_skip_email(uid):
            default_log.info('Overdraft user %s: has deleted passport account, skip sending email' % uid)
            return
        send_email_async_by_uid(
            uid,
            'the_blocking_is_going_in_days',
            template_args={
                'days': days_left,
            },
            default_locale=DEFAULT_LOCALE,
            allowed_locales=ALLOWED_LOCALES
        )
        OverdraftDAO().update_or_create(
            uid,
            overdraft_info.overdraft_date,
            last_push_date=overdraft_info.last_push_date,
            last_email_date=datetime.datetime.now()
        )
    except socket.error:
        error_log.exeception('Can\'t send the_blocking_is_going_in_days email for uid: %s' % uid)


def send_email_for_light(uid, used_space, limit_space):
    overdraft_info = _get_overdraft_info(uid)

    try:
        default_log.info('overdraft_send_message uid: %s, status: lite' % uid)
        if _need_skip_email(uid):
            default_log.info('Overdraft user %s: has deleted passport account, skip sending email' % uid)
            return
        send_email_async_by_uid(
            uid,
            'the_space_is_over',
            template_args={
                'space': used_space,
                'base_space': limit_space
            },
            default_locale=DEFAULT_LOCALE,
            allowed_locales=ALLOWED_LOCALES
        )
        OverdraftDAO().update_or_create(
            uid, overdraft_info.overdraft_date,
            last_push_date=overdraft_info.last_push_date,
            last_email_date=datetime.datetime.now()
        )
    except socket.error:
        error_log.exeception('Can\'t send the_space_is_over email for uid: %s' % uid)


def send_push_regular(uid):
    overdraft_info = _get_overdraft_info(uid)

    try:
        PushQueue().put(
            {
                'class': 'quota_overdraft_regular',
                'uid': uid,
                'xiva_data': {
                    'root': {
                        'tag': 'quota_overdraft_regular',
                        'parameters': User(uid).get_overdraft_info()
                    }
                }
            }
        )
        OverdraftDAO().update_or_create(
            uid,
            overdraft_info.overdraft_date,
            last_push_date=datetime.datetime.now(),
            last_email_date=overdraft_info.last_email_date
        )
    except socket.error:
        error_log.exeception('Can\'t send push for uid: %s' % uid)


def send_push(uid):
    PushQueue().put(
        {
            'class': 'quota_overdraft',
            'uid': uid,
            'xiva_data': {
                'root': {
                    'tag': 'quota_overdraft',
                    'parameters': User(uid).get_overdraft_info()
                }
            }
        }
    )


def _get_overdraft_info(uid):
    return OverdraftDAO().get(uid)


def _need_skip_email(uid):
    user_info = passport.userinfo(uid)
    if user_info.get('uid') is None:
        return True
    return False
